// 5.0.4
const proxmoxOnlineHelpInfo = {
   "chapter_mailfilter" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#chapter_mailfilter",
      "title" : "Rule-Based Mail Filter"
   },
   "chapter_pmgbackup" : {
      "link" : "/pmg-docs/chapter-pmgbackup.html#chapter_pmgbackup",
      "title" : "Backup and Restore"
   },
   "getting_help" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#getting_help",
      "title" : "Getting Help"
   },
   "pmg_documentation_index" : {
      "link" : "/pmg-docs/index.html",
      "title" : "Proxmox Mail Gateway Documentation Index"
   },
   "pmg_mailfilter_action" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#pmg_mailfilter_action",
      "subtitle" : "'Action' - objects",
      "title" : "Rule-Based Mail Filter"
   },
   "pmg_mailfilter_regex" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#pmg_mailfilter_regex",
      "subtitle" : "Using regular expressions",
      "title" : "Rule-Based Mail Filter"
   },
   "pmg_mailfilter_what" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#pmg_mailfilter_what",
      "subtitle" : "'What' objects",
      "title" : "Rule-Based Mail Filter"
   },
   "pmg_mailfilter_when" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#pmg_mailfilter_when",
      "subtitle" : "'When' objects",
      "title" : "Rule-Based Mail Filter"
   },
   "pmg_mailfilter_who" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#pmg_mailfilter_who",
      "subtitle" : "'Who' objects",
      "title" : "Rule-Based Mail Filter"
   },
   "pmg_package_repositories" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#pmg_package_repositories",
      "subtitle" : "Package Repositories",
      "title" : "Installing Proxmox Mail Gateway"
   },
   "pmg_userblockwelcomelist" : {
      "link" : "/pmg-docs/pmg-admin-guide.html#pmg_userblockwelcomelist",
      "title" : "Administration"
   },
   "pmgbackup_pbs_remotes" : {
      "link" : "/pmg-docs/chapter-pmgbackup.html#pmgbackup_pbs_remotes",
      "subtitle" : "Remotes",
      "title" : "Backup and Restore"
   },
   "pmgbackup_pbs_schedule" : {
      "link" : "/pmg-docs/chapter-pmgbackup.html#pmgbackup_pbs_schedule",
      "subtitle" : "Scheduled Backups",
      "title" : "Backup and Restore"
   },
   "pmgcm_join" : {
      "link" : "/pmg-docs/chapter-pmgcm.html#pmgcm_join",
      "subtitle" : "Adding Cluster Nodes",
      "title" : "Cluster Management"
   },
   "pmgconfig_clamav" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_clamav",
      "subtitle" : "Virus Detector Configuration",
      "title" : "Configuration Management"
   },
   "pmgconfig_clamav_options" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_clamav_options",
      "subtitle" : "Options",
      "title" : "Configuration Management"
   },
   "pmgconfig_clamav_quarantine" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_clamav_quarantine",
      "subtitle" : "Quarantine",
      "title" : "Configuration Management"
   },
   "pmgconfig_fetchmail" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_fetchmail",
      "subtitle" : "Fetchmail",
      "title" : "Configuration Management"
   },
   "pmgconfig_ldap" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_ldap",
      "subtitle" : "LDAP/Active Directory",
      "title" : "Configuration Management"
   },
   "pmgconfig_localuser" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_localuser",
      "subtitle" : "Local Users",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_before_after_queue" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_before_after_queue",
      "subtitle" : "Before and After Queue scanning",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_dkim" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_dkim",
      "subtitle" : "DKIM Signing",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_networks" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_networks",
      "subtitle" : "Networks",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_options" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_options",
      "subtitle" : "Options",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_ports" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_ports",
      "subtitle" : "Ports",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_relay_domains" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_relay_domains",
      "subtitle" : "Relay Domains",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_relaying" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_relaying",
      "subtitle" : "Relaying",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_tls" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_tls",
      "subtitle" : "TLS",
      "title" : "Configuration Management"
   },
   "pmgconfig_mailproxy_transports" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_mailproxy_transports",
      "subtitle" : "Transports",
      "title" : "Configuration Management"
   },
   "pmgconfig_spamdetector" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_spamdetector",
      "subtitle" : "Spam Detector Configuration",
      "title" : "Configuration Management"
   },
   "pmgconfig_spamdetector_customscores" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_spamdetector_customscores",
      "subtitle" : "Customization of Rulescores",
      "title" : "Configuration Management"
   },
   "pmgconfig_spamdetector_quarantine" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_spamdetector_quarantine",
      "subtitle" : "Quarantine",
      "title" : "Configuration Management"
   },
   "pmgconfig_systemconfig" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#pmgconfig_systemconfig",
      "subtitle" : "System Configuration",
      "title" : "Configuration Management"
   },
   "user_oidc" : {
      "link" : "/pmg-docs/chapter-pmgconfig.html#user_oidc",
      "subtitle" : "OpenID Connect",
      "title" : "Configuration Management"
   }
};
Ext.ns('PMG');

console.log('Starting PMG Manager');

Ext.define('PMG.Utils', {
    singleton: true,

    // this singleton contains miscellaneous utilities

    // use in panels with object spread (...) operator, for example:
    // ...PMG.Utils.onlineHelpTool('sysadmin_certificate_management'),
    onlineHelpTool: function (blockid) {
        let info = Proxmox.Utils.get_help_info(blockid);
        if (info === undefined) {
            info = Proxmox.Utils.get_help_info('pmg_documentation_index');
            if (info === undefined) {
                throw 'get_help_info failed'; // should not happen
            }
        }

        let docsURI = window.location.origin + info.link;
        let title = info.title || gettext('Help');
        if (info.subtitle) {
            title += ' - ' + info.subtitle;
        }
        return {
            tools: [
                {
                    type: 'help',
                    tooltip: title,
                    handler: () => window.open(docsURI),
                },
            ],
        };
    },

    senderText: gettext('Sender'),
    receiverText: gettext('Receiver'),
    scoreText: gettext('Score'),

    user_role_text: {
        root: gettext('Superuser'),
        admin: gettext('Administrator'),
        helpdesk: gettext('Help Desk'),
        qmanager: gettext('Quarantine Manager'),
        audit: gettext('Auditor'),
    },

    format_user_role: function (role) {
        return PMG.Utils.user_role_text[role] || role;
    },

    oclass_text: {
        who: gettext('Who Objects'),
        what: gettext('What Objects'),
        when: gettext('When Objects'),
        action: gettext('Action Objects'),
        from: gettext('From'),
        to: gettext('To'),
    },

    oclass_icon: {
        who: '<span class="fa fa-fw fa-user-circle"></span> ',
        what: '<span class="fa fa-fw fa-cube"></span> ',
        when: '<span class="fa fa-fw fa-clock-o"></span> ',
        action: '<span class="fa fa-fw fa-flag"></span> ',
        from: '<span class="fa fa-fw fa-user-circle"></span> ',
        to: '<span class="fa fa-fw fa-user-circle"></span> ',
    },

    mail_status_map: {
        2: 'delivered',
        4: 'deferred',
        5: 'bounced',
        N: 'rejected',
        G: 'greylisted',
        A: 'accepted',
        B: 'blocked',
        Q: 'quarantine',
    },

    icon_status_map_class: {
        2: 'check-circle',
        4: 'clock-o',
        5: 'mail-reply',
        N: 'times-circle',
        G: 'list',
        A: 'check',
        B: 'ban',
        Q: 'cube',
    },

    icon_status_map_color: {
        2: 'green',
        5: 'gray',
        A: 'green',
        B: 'red',
    },

    format_status_icon: function (status) {
        var icon = PMG.Utils.icon_status_map_class[status] || 'question-circle';
        var color = PMG.Utils.icon_status_map_color[status] || '';
        return '<i class="fa fa-' + icon + ' ' + color + '"></i> ';
    },

    format_oclass: function (oclass) {
        var icon = PMG.Utils.oclass_icon[oclass] || '';
        var text = PMG.Utils.oclass_text[oclass] || oclass;
        return icon + text;
    },

    rule_direction_text: {
        0: gettext('In'),
        1: gettext('Out'),
        2: gettext('In & Out'),
    },

    rule_direction_icon: {
        0: '<span class="fa fa-fw fa-long-arrow-left"></span> ',
        1: '<span class="fa fa-fw fa-long-arrow-right"></span> ',
        2: '<span class="fa fa-fw fa-exchange"></span> ',
    },

    format_rule_direction: function (dir) {
        var icon = PMG.Utils.rule_direction_icon[dir] || '';
        var text = PMG.Utils.rule_direction_text[dir] || dir;
        return icon + text;
    },

    format_otype_subject: function (otype) {
        let editor = PMG.Utils.object_editors[otype];
        return editor.subject ?? 'unknown';
    },

    format_otype: function (otype) {
        let editor = PMG.Utils.object_editors[otype];
        let iconCls = 'fa fa-question-circle';
        if (editor) {
            return `<span class="fa-fw ${editor.iconCls || iconCls}"></span> ${editor.subject}`;
        }
        return `<span class="fa-fw ${iconCls}"></span> unknown`;
    },

    format_ldap_protocol: function (p) {
        if (p === undefined) {
            return 'LDAP';
        }
        if (p === 'ldap') {
            return 'LDAP';
        }
        if (p === 'ldaps') {
            return 'LDAPS';
        }
        if (p === 'ldap+starttls') {
            return 'LDAP+STARTTLS';
        }
        return 'unknown';
    },

    convert_field_to_per_min: function (value, record) {
        return value / (record.data.timespan / 60);
    },

    object_editors: {
        1000: {
            onlineHelp: 'pmg_mailfilter_regex',
            iconCls: 'fa fa-filter',
            xtype: 'proxmoxWindowEdit',
            subdir: 'regex',
            subject: gettext('Regular Expression'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'regex',
                    reference: 'regex',
                    fieldLabel: gettext('Regex'),
                },
                {
                    xtype: 'pmgRegexTester',
                    fieldLabel: gettext('Test String'),
                    wholeMatch: true,
                    regexFieldReference: 'regex',
                },
            ],
        },
        1005: {
            onlineHelp: 'pmgconfig_ldap',
            iconCls: 'fa fa-users',
            xtype: 'pmgLDAPGroupEditor',
            subdir: 'ldap',
            subject: gettext('LDAP Group'),
        },
        1006: {
            onlineHelp: 'pmgconfig_ldap',
            iconCls: 'fa fa-user',
            xtype: 'pmgLDAPUserEditor',
            subdir: 'ldapuser',
            subject: gettext('LDAP User'),
        },
        1009: {
            onlineHelp: 'pmg_mailfilter_regex',
            iconCls: 'fa fa-filter',
            xtype: 'proxmoxWindowEdit',
            subdir: 'receiver_regex',
            subject: gettext('Regular Expression'),
            receivertest: true,
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'regex',
                    fieldLabel: gettext('Regex'),
                },
            ],
        },
        1001: {
            onlineHelp: 'pmg_mailfilter_who',
            iconCls: 'fa fa-envelope-o',
            xtype: 'proxmoxWindowEdit',
            subdir: 'email',
            subject: gettext('E-Mail'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'email',
                    fieldLabel: gettext('E-Mail'),
                },
            ],
        },
        1007: {
            onlineHelp: 'pmg_mailfilter_who',
            iconCls: 'fa fa-envelope-o',
            xtype: 'proxmoxWindowEdit',
            subdir: 'receiver',
            subject: gettext('E-Mail'),
            receivertest: true,
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'email',
                    fieldLabel: gettext('E-Mail'),
                },
            ],
        },
        1002: {
            onlineHelp: 'pmg_mailfilter_who',
            iconCls: 'fa fa-globe',
            xtype: 'proxmoxWindowEdit',
            subdir: 'domain',
            subject: gettext('Domain'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'domain',
                    fieldLabel: gettext('Domain'),
                },
            ],
        },
        1008: {
            onlineHelp: 'pmg_mailfilter_who',
            iconCls: 'fa fa-globe',
            xtype: 'proxmoxWindowEdit',
            subdir: 'receiver_domain',
            subject: gettext('Domain'),
            receivertest: true,
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'domain',
                    fieldLabel: gettext('Domain'),
                },
            ],
        },
        1003: {
            onlineHelp: 'pmg_mailfilter_who',
            iconCls: 'fa fa-globe',
            xtype: 'proxmoxWindowEdit',
            subdir: 'ip',
            subject: gettext('IP Address'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'ip',
                    fieldLabel: gettext('IP Address'),
                },
            ],
        },
        1004: {
            onlineHelp: 'pmg_mailfilter_who',
            iconCls: 'fa fa-globe',
            xtype: 'proxmoxWindowEdit',
            subdir: 'network',
            subject: gettext('IP Network'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'cidr',
                    fieldLabel: gettext('IP Network'),
                },
            ],
        },
        2000: {
            onlineHelp: 'pmg_mailfilter_when',
            iconCls: 'fa fa-clock-o',
            xtype: 'proxmoxWindowEdit',
            subdir: 'timeframe',
            subject: gettext('TimeFrame'),
            items: [
                {
                    xtype: 'timefield',
                    name: 'start',
                    format: 'H:i',
                    fieldLabel: gettext('Start Time'),
                },
                {
                    xtype: 'timefield',
                    name: 'end',
                    format: 'H:i',
                    fieldLabel: gettext('End Time'),
                },
            ],
        },
        3000: {
            onlineHelp: 'pmg_mailfilter_what',
            iconCls: 'fa fa-bullhorn',
            xtype: 'proxmoxWindowEdit',
            subdir: 'spamfilter',
            subject: gettext('Spam Filter'),
            items: [
                {
                    xtype: 'proxmoxintegerfield',
                    name: 'spamlevel',
                    allowBlank: false,
                    minValue: 0,
                    fieldLabel: gettext('Level'),
                },
            ],
        },
        3001: {
            onlineHelp: 'pmg_mailfilter_what',
            iconCls: 'fa fa-bug',
            xtype: 'proxmoxWindowEdit',
            subdir: 'virusfilter',
            subject: gettext('Virus Filter'),
            uneditable: true,
            // there are no parameters to give, so we simply submit it
            listeners: {
                show: function (win) {
                    win.submit();
                },
            },
        },
        3002: {
            onlineHelp: 'pmg_mailfilter_regex',
            iconCls: 'fa fa-code',
            xtype: 'proxmoxWindowEdit',
            subdir: 'matchfield',
            subject: gettext('Match Field'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'field',
                    allowBlank: false,
                    fieldLabel: gettext('Field'),
                },
                {
                    xtype: 'textfield',
                    name: 'value',
                    reference: 'value',
                    allowBlank: false,
                    fieldLabel: gettext('Value'),
                },
                {
                    xtype: 'pmgRegexTester',
                    fieldLabel: gettext('Test String'),
                    regexFieldReference: 'value',
                },
                {
                    xtype: 'proxmoxcheckbox',
                    name: 'top-part-only',
                    fieldLabel: gettext('Only top level headers'),
                    uncheckedValue: '0',
                },
            ],
        },
        3003: {
            onlineHelp: 'pmg_mailfilter_what',
            iconCls: 'fa fa-file-image-o',
            xtype: 'proxmoxWindowEdit',
            subdir: 'contenttype',
            width: 400,
            subject: gettext('Content Type Filter'),
            items: [
                {
                    xtype: 'combobox',
                    displayField: 'text',
                    valueField: 'mimetype',
                    name: 'contenttype',
                    editable: true,
                    queryMode: 'local',
                    store: {
                        autoLoad: true,
                        proxy: {
                            type: 'proxmox',
                            url: '/api2/json/config/mimetypes',
                        },
                    },
                    fieldLabel: gettext('Content Type'),
                    anyMatch: true,
                    matchFieldWidth: false,
                    listeners: {
                        change: function (cb, value) {
                            var me = this;
                            me.up().down('displayfield').setValue(value);
                        },
                    },
                },
                {
                    xtype: 'displayfield',
                    fieldLabel: gettext('Value'),
                    allowBlank: false,
                    reset: Ext.emptyFn,
                },
                {
                    xtype: 'proxmoxcheckbox',
                    name: 'only-content',
                    fieldLabel: gettext('Ignore header information'),
                    uncheckedValue: '0',
                },
            ],
        },
        3004: {
            onlineHelp: 'pmg_mailfilter_regex',
            iconCls: 'fa fa-file-o',
            xtype: 'proxmoxWindowEdit',
            subdir: 'filenamefilter',
            width: 400,
            subject: gettext('Match Filename'),
            items: [
                {
                    xtype: 'textfield',
                    name: 'filename',
                    reference: 'filename',
                    fieldLabel: gettext('Filename'),
                    allowBlank: false,
                },
                {
                    xtype: 'pmgRegexTester',
                    fieldLabel: gettext('Test String'),
                    wholeMatch: true,
                    regexFieldReference: 'filename',
                },
            ],
        },
        3005: {
            onlineHelp: 'pmg_mailfilter_what',
            iconCls: 'fa fa-file-archive-o',
            xtype: 'proxmoxWindowEdit',
            subdir: 'archivefilter',
            width: 400,
            subject: gettext('Archive Filter'),
            items: [
                {
                    xtype: 'combobox',
                    displayField: 'text',
                    valueField: 'mimetype',
                    name: 'contenttype',
                    editable: true,
                    queryMode: 'local',
                    store: {
                        autoLoad: true,
                        proxy: {
                            type: 'proxmox',
                            url: '/api2/json/config/mimetypes',
                        },
                    },
                    fieldLabel: gettext('Content Type'),
                    anyMatch: true,
                    matchFieldWidth: false,
                    listeners: {
                        change: function (cb, value) {
                            var me = this;
                            me.up().down('displayfield').setValue(value);
                        },
                    },
                },
                {
                    xtype: 'displayfield',
                    fieldLabel: gettext('Value'),
                    allowBlank: false,
                    reset: Ext.emptyFn,
                },
            ],
        },
        3006: {
            onlineHelp: 'pmg_mailfilter_regex',
            iconCls: 'fa fa-file-archive-o',
            xtype: 'proxmoxWindowEdit',
            subdir: 'archivefilenamefilter',
            width: 400,
            subject: gettext('Match Archive Filename'),
            items: [
                {
                    xtype: 'textfield',
                    name: 'filename',
                    reference: 'filename',
                    fieldLabel: gettext('Filename'),
                    allowBlank: false,
                },
                {
                    xtype: 'pmgRegexTester',
                    fieldLabel: gettext('Test String'),
                    wholeMatch: true,
                    regexFieldReference: 'filename',
                },
            ],
        },
        4002: {
            onlineHelp: 'pmg_mailfilter_action',
            xtype: 'proxmoxWindowEdit',
            subdir: 'notification',
            subject: gettext('Notification'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'name',
                    allowBlank: false,
                    fieldLabel: gettext('Name'),
                },
                {
                    xtype: 'textareafield',
                    name: 'info',
                    fieldLabel: gettext('Comment'),
                },
                {
                    xtype: 'textfield',
                    name: 'to',
                    allowBlank: false,
                    value: '__ADMIN__',
                    fieldLabel: gettext('Receiver'),
                },
                {
                    xtype: 'textfield',
                    name: 'subject',
                    allowBlank: false,
                    value: 'Notification: __SUBJECT__',
                    fieldLabel: gettext('Subject'),
                },
                {
                    xtype: 'textarea',
                    name: 'body',
                    allowBlank: false,
                    grow: true,
                    growMax: 250,
                    value:
                        'Proxmox Notification:\n\n' +
                        'Sender:   __SENDER__\n' +
                        'Receiver: __RECEIVERS__\n' +
                        'Targets:  __TARGETS__\n\n' +
                        'Subject:  __SUBJECT__\n\n' +
                        'Matching Rule: __RULE__\n\n' +
                        '__RULE_INFO__\n\n' +
                        '__VIRUS_INFO__\n' +
                        '__SPAM_INFO__\n',
                    fieldLabel: gettext('Body'),
                },
                {
                    xtype: 'proxmoxcheckbox',
                    name: 'attach',
                    fieldLabel: gettext('Attach orig. Mail'),
                },
            ],
        },
        4003: {
            onlineHelp: 'pmg_mailfilter_action',
            xtype: 'proxmoxWindowEdit',
            subdir: 'field',
            subject: gettext('Header Attribute'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'name',
                    allowBlank: false,
                    fieldLabel: gettext('Name'),
                },
                {
                    xtype: 'textareafield',
                    name: 'info',
                    fieldLabel: gettext('Comment'),
                },
                {
                    xtype: 'textfield',
                    name: 'field',
                    allowBlank: false,
                    fieldLabel: gettext('Field'),
                },
                {
                    xtype: 'textfield',
                    reference: 'value',
                    name: 'value',
                    allowBlank: false,
                    fieldLabel: gettext('Value'),
                },
            ],
        },
        4005: {
            onlineHelp: 'pmg_mailfilter_action',
            xtype: 'proxmoxWindowEdit',
            subdir: 'bcc',
            subject: gettext('BCC'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'name',
                    allowBlank: false,
                    fieldLabel: gettext('Name'),
                },
                {
                    xtype: 'textareafield',
                    name: 'info',
                    fieldLabel: gettext('Comment'),
                },
                {
                    xtype: 'textfield',
                    name: 'target',
                    allowBlank: false,
                    fieldLabel: gettext('Target'),
                },
                {
                    xtype: 'proxmoxcheckbox',
                    checked: true,
                    name: 'original',
                    boxLabel: gettext('Send Original Mail'),
                },
            ],
        },
        4007: {
            onlineHelp: 'pmg_mailfilter_action',
            xtype: 'proxmoxWindowEdit',
            subdir: 'removeattachments',
            subject: gettext('Remove Attachments'),
            width: 500,
            items: [
                {
                    xtype: 'textfield',
                    name: 'name',
                    allowBlank: false,
                    fieldLabel: gettext('Name'),
                },
                {
                    xtype: 'textareafield',
                    name: 'info',
                    fieldLabel: gettext('Comment'),
                },
                {
                    xtype: 'textareafield',
                    name: 'text',
                    grow: true,
                    growMax: 250,
                    fieldLabel: gettext('Text Replacement'),
                },
                {
                    xtype: 'proxmoxcheckbox',
                    checked: true,
                    name: 'all',
                    boxLabel: gettext('Remove all Attachments'),
                },
                {
                    xtype: 'proxmoxcheckbox',
                    checked: false,
                    name: 'quarantine',
                    boxLabel: gettext('Copy original mail to Attachment Quarantine'),
                },
            ],
        },
        4009: {
            onlineHelp: 'pmg_mailfilter_action',
            xtype: 'proxmoxWindowEdit',
            subdir: 'disclaimer',
            subject: gettext('Disclaimer'),
            width: 400,
            items: [
                {
                    xtype: 'textfield',
                    name: 'name',
                    allowBlank: false,
                    fieldLabel: gettext('Name'),
                },
                {
                    xtype: 'textareafield',
                    name: 'info',
                    fieldLabel: gettext('Comment'),
                },
                {
                    xtype: 'textareafield',
                    name: 'disclaimer',
                    grow: true,
                    growMax: 250,
                    fieldLabel: gettext('Disclaimer'),
                },
                {
                    xtype: 'proxmoxKVComboBox',
                    name: 'position',
                    fieldLabel: gettext('Position'),
                    deleteEmpty: false,
                    value: 'end',
                    comboItems: [
                        ['end', gettext('End')],
                        ['start', gettext('Start')],
                    ],
                },
                {
                    xtype: 'proxmoxcheckbox',
                    name: 'add-separator',
                    fieldLabel: gettext('Add Separator'),
                    uncheckedValue: '0',
                    value: true,
                },
            ],
        },
    },

    updateLoginData: function (data) {
        Proxmox.CSRFPreventionToken = data.CSRFPreventionToken;
        Proxmox.UserName = data.username;
        Ext.util.Cookies.set('PMGAuthCookie', data.ticket, null, '/', null, true);
    },

    quarantineActionExtracted: false,

    extractQuarantineAction: function () {
        if (PMG.Utils.quarantineActionExtracted) {
            return null;
        }

        PMG.Utils.quarantineActionExtracted = true;

        let qs = Ext.Object.fromQueryString(location.search);

        let cselect = qs.cselect;
        let action = qs.action;
        let dateString = qs.date;

        if (dateString) {
            let date = new Date(dateString).getTime() / 1000;

            // set from date for QuarantineList
            PMG.QuarantineList.from = date;
        }

        delete qs.cselect;
        delete qs.action;
        delete qs.ticket;
        delete qs.date;

        var newsearch = Ext.Object.toQueryString(qs);

        var newurl = location.protocol + '//' + location.host + location.pathname;
        if (newsearch) {
            newurl += '?' + newsearch;
        }
        newurl += location.hash;

        if (window.history) {
            window.history.pushState({ path: newurl }, '', newurl);
        }

        if (action || cselect) {
            return {
                action: action,
                cselect: cselect,
            };
        }
        return null;
    },

    doQuarantineAction: function (action, id, callback) {
        Proxmox.Utils.API2Request({
            url: '/quarantine/content/',
            params: {
                action: action,
                id: id,
            },
            method: 'POST',
            failure: function (response, opts) {
                Ext.Msg.alert(gettext('Error'), response.htmlStatus);
            },
            success: function (response, opts) {
                let count = id.split(';').length;
                let fmt =
                    count > 1
                        ? gettext("Action '{0}' for '{1}' items successful")
                        : gettext("Action '{0}' successful");
                let message = Ext.String.format(fmt, action, count);
                let title = Ext.String.format('{0} successful', Ext.String.capitalize(action));

                Ext.toast({
                    html: message,
                    title: title,
                    minWidth: 200,
                    hideDuration: 250,
                    slideBackDuration: 250,
                    slideBackAnimation: 'easeOut',
                    iconCls: 'fa fa-check',
                    shadow: true,
                    align: 'br',
                });

                if (Ext.isFunction(callback)) {
                    callback();
                }
            },
        });
    },

    render_filetype: function (value) {
        let iconCls = 'fa-file-o';
        let text = Proxmox.Utils.unknownText;

        if (!value) {
            return `<i class='fa ${iconCls}'></i> ${text}`;
        }

        text = value.toString().toLowerCase();
        const type = text.split('/')[0];

        switch (type) {
            case 'audio':
            case 'image':
            case 'video':
            case 'text':
                iconCls = `fa-file-${type}-o`;
                break;
            case 'application':
                {
                    const subtypes = ['excel', 'pdf', 'word', 'powerpoint'];
                    let found = subtypes.find((st) => text.includes(st));
                    if (found !== undefined) {
                        iconCls = `fa-file-${found}-o`;
                    }
                }
                break;
            default:
                break;
        }

        return `<i class='fa ${iconCls}'></i> ${text}`;
    },

    render_envelope: function (value, { data }, render_receiver) {
        let subject = Ext.htmlEncode(value);
        let from = Ext.htmlEncode(data.from);
        if (data.sender) {
            let sender = Ext.htmlEncode(data.sender);
            from = Ext.String.format(gettext('{0} on behalf of {1}'), sender, from);
        }
        if (render_receiver) {
            let receiver = Ext.htmlEncode(data.receiver);
            return `<small>${from}<br>To: ${receiver}</small><br>${subject}`;
        }
        return `<small>${from}</small><br>${subject}`;
    },

    render_sender: (value, _meta, rec) => PMG.Utils.render_envelope(value, rec, false),
    render_sender_receiver: (value, _meta, rec) => PMG.Utils.render_envelope(value, rec, true),

    constructor: function () {
        var _me = this;

        // use oidc instead of openid
        Proxmox.Schema.authDomains.oidc = Proxmox.Schema.authDomains.openid;
        Proxmox.Schema.authDomains.oidc.useTypeInUrl = false;
        Proxmox.Schema.authDomains.oidc.ipanel = 'pmgAuthOIDCPanel';
        delete Proxmox.Schema.authDomains.openid;

        // Disable LDAP/AD as a realm until LDAP/AD login is implemented
        Proxmox.Schema.authDomains.ldap.add = false;
        Proxmox.Schema.authDomains.ad.add = false;

        Proxmox.Schema.overrideAuthDomains({
            pmg: {
                name: 'Proxmox Mail Gateway authentication server',
                ipanel: 'pmxAuthSimplePanel',
                add: false,
                edit: true,
                pwchange: true,
                sync: false,
            },
        });

        // do whatever you want here
        Proxmox.Utils.override_task_descriptions({
            applycustomscores: ['', gettext('Apply custom SpamAssassin scores')],
            avupdate: ['', gettext('ClamAV update')],
            backup: ['', gettext('Backup')],
            clustercreate: ['', gettext('Create Cluster')],
            clusterjoin: ['', gettext('Join Cluster')],
            restore: ['', gettext('Restore')],
            saupdate: ['', gettext('SpamAssassin update')],
        });
    },
});

Ext.define('PMG.Async', {
    singleton: true,

    // Returns a Promise which executes a quarantine action when awaited.
    // Shows a Toast message box once completed, if batchNumber and batchTotal
    // are set, they will be included into the title of that toast.
    doQAction: function (action, ids, batchNumber, batchTotal) {
        if (!Ext.isArray(ids)) {
            ids = [ids];
        }
        return Proxmox.Async.api2({
            url: '/quarantine/content/',
            params: {
                action: action,
                id: ids.join(';'),
            },
            method: 'POST',
        }).then(
            (response) => {
                let count = ids.length;
                let fmt =
                    count > 1
                        ? gettext("Action '{0}' for '{1}' items successful")
                        : gettext("Action '{0}' successful");
                let message = Ext.String.format(fmt, action, count);
                let titleFmt =
                    batchNumber !== undefined && batchTotal > 1
                        ? gettext('{0} ({1}/{2}) successful')
                        : gettext('{0} successful');
                let title = Ext.String.format(
                    titleFmt,
                    Ext.String.capitalize(action),
                    batchNumber,
                    batchTotal,
                );

                Ext.toast({
                    html: message,
                    title: title,
                    minWidth: 200,
                    hideDuration: 250,
                    slideBackDuration: 250,
                    slideBackAnimation: 'easeOut',
                    iconCls: 'fa fa-check',
                    shadow: true,
                    align: 'br',
                });
            },
            (response) => Proxmox.Utils.alertResponseFailure(response),
        );
    },
});

// custom Vtypes
Ext.apply(Ext.form.field.VTypes, {
    // matches the pmg-email-address in pmg-api
    PMGMail: function (v) {
        return /^[^\s\\@]+@[^\s/\\@]+$/.test(v);
    },
    PMGMailText: gettext('Example') + ': user@example.com',
});
Ext.define('PMG.form.FilterField', {
    extend: 'Ext.form.field.Text',
    alias: 'widget.pmgFilterField',

    // the store to filter
    // optional, if not given the first parent grids store will be used
    store: undefined,

    // a list of fields of the records that will be searched
    // a field can be a string (dataIndex) or an object with 'name' and 'renderer'
    // the renderer will be used before testing the field
    filteredFields: [],

    fieldLabel: gettext('Filter'),
    labelAlign: 'right',

    triggers: {
        clear: {
            cls: 'pmx-clear-trigger',
            hidden: true,
            handler: function () {
                let me = this;
                me.setValue('');
                me.triggers.clear.setVisible(false);
            },
        },
    },

    listeners: {
        change: function (field, value) {
            let me = this;
            let grid = me.up('grid');
            if (!me.store) {
                me.store = grid.getStore();
            }

            me.store.clearFilter();
            field.triggers.clear.setVisible(value.length > 0);

            if (value) {
                me.store.filterBy((rec) =>
                    me.filteredFields.some((fieldDef) => {
                        let fieldname = fieldDef,
                            renderer = Ext.identityFn;
                        if (Ext.isObject(fieldDef)) {
                            fieldname = fieldDef.name;
                            renderer = fieldDef.renderer;
                        }
                        let testedValue = renderer(rec.data[fieldname]);
                        return (
                            testedValue.toString().toLowerCase().indexOf(value.toLowerCase()) !== -1
                        );
                    }),
                );
            }
        },
    },
});
Ext.define('PMG.MatchModeSelector', {
    extend: 'Proxmox.form.KVComboBox',
    alias: 'widget.pmgMatchModeSelector',

    comboItems: [
        ['all', gettext('All match')],
        ['any', gettext('Any matches')],
        ['notall', gettext('At least one does not match')],
        ['notany', gettext('None matches')],
    ],
});
Ext.define('PMG.FilterProxy', {
    extend: 'Proxmox.RestProxy',
    alias: 'proxy.pmgfilterproxy',

    filterId: undefined, // 'x-gridfilter-XXXXX'

    getParams: function (operation) {
        var me = this,
            i;
        if (!operation.isReadOperation) {
            return {};
        }
        var params = me.callParent(arguments);

        var filters = operation.getFilters() || [];
        for (i = 0; i < filters.length; i++) {
            let filter = filters[i];
            if (filter.config.id === me.filterId) {
                let v = filter.getValue();
                if (v !== undefined && v !== '') {
                    params.filter = v;
                }
            }
        }
        return params;
    },
});
Ext.define('PMG.LoginView', {
    extend: 'Ext.container.Container',
    xtype: 'loginview',

    viewModel: {
        data: {
            oidc: false,
        },
        formulas: {
            buttonText: function (get) {
                if (get('oidc') === true) {
                    return gettext('Login (OpenID Connect redirect)');
                } else {
                    return gettext('Login');
                }
            },
            showSaveUser: function (get) {
                return this.getView().targetview !== 'quarantineview' && !get('oidc');
            },
        },
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        init: function (view) {
            let me = this;

            let realmfield = me.lookup('realmfield');

            me.lookup('quarantineButton').setVisible(!!Proxmox.QuarantineLink);

            if (view.targetview !== 'quarantineview') {
                return;
            }

            // hide save username field for quarantine view
            me.lookup('saveunField').setVisible(false);

            // disable/hide realm field for quarantine view
            realmfield.setDisabled(true);
            realmfield.setHidden(true);

            realmfield.setValue('quarantine');

            // try autologin with quarantine ticket from URL

            let qs = Ext.Object.fromQueryString(location.search);
            if (qs.ticket === undefined) {
                return;
            }
            let ticket = decodeURIComponent(qs.ticket);
            let match = ticket.match(/^PMGQUAR:([^\s:]+):/);
            if (!match) {
                return;
            }
            let username = match[1];
            let loginwin = me.lookup('loginwindow');
            loginwin.autoShow = false;
            loginwin.setVisible(false);

            me.lookup('usernameField').setValue(username);
            me.lookup('passwordField').setValue(ticket);

            me.submitForm();
        },

        submitForm: async function () {
            let me = this;

            let loginForm = this.lookupReference('loginForm');
            let unField = this.lookupReference('usernameField');
            let saveunField = this.lookupReference('saveunField');
            let view = this.getView();

            if (!loginForm.isValid()) {
                return;
            }

            if (loginForm.isVisible()) {
                loginForm.mask(gettext('Please wait...'), 'x-mask-loading');
            }

            // set or clear username for admin view
            if (view.targetview !== 'quarantineview') {
                let sp = Ext.state.Manager.getProvider();
                if (saveunField.getValue() === true) {
                    sp.set(unField.getStateId(), unField.getValue());
                } else {
                    sp.clear(unField.getStateId());
                }
                sp.set(saveunField.getStateId(), saveunField.getValue());
            }

            let creds = loginForm.getValues();

            if (this.getViewModel().data.oidc === true) {
                const redirectURL = location.origin;
                Proxmox.Utils.API2Request({
                    url: '/api2/extjs/access/oidc/auth-url',
                    params: {
                        realm: creds.realm,
                        'redirect-url': redirectURL,
                    },
                    method: 'POST',
                    success: function (resp, opts) {
                        window.location = resp.result.data;
                    },
                    failure: function (resp, opts) {
                        Proxmox.Utils.authClear();
                        loginForm.unmask();
                        Ext.MessageBox.alert(
                            gettext('Error'),
                            gettext('OpenID Connect redirect failed.') + `<br>${resp.htmlStatus}`,
                        );
                    },
                });
                return;
            }

            try {
                let resp = await Proxmox.Async.api2({
                    url: '/api2/extjs/access/ticket',
                    params: creds,
                    method: 'POST',
                });

                let data = resp.result.data;
                if (data.ticket.startsWith('PMG:!tfa!')) {
                    data = await me.performTFAChallenge(data);
                }
                PMG.Utils.updateLoginData(data);
                PMG.app.changeView(view.targetview);
            } catch (_error) {
                Proxmox.Utils.authClear();
                loginForm.unmask();
                Ext.MessageBox.alert(gettext('Error'), gettext('Login failed. Please try again'));
            }
        },

        performTFAChallenge: async function (data) {
            let _me = this;

            let userid = data.username;
            let ticket = data.ticket;
            let challenge = JSON.parse(
                decodeURIComponent(ticket.split(':')[1].slice('!tfa!'.length)),
            );

            let resp = await new Promise((resolve, reject) => {
                Ext.create('Proxmox.window.TfaLoginWindow', {
                    userid,
                    ticket,
                    challenge,
                    onResolve: (value) => resolve(value),
                    onReject: reject,
                }).show();
            });

            return resp.result.data;
        },

        success: function (data) {
            let me = this;
            let view = me.getView();
            let handler = view.handler || Ext.emptyFn;
            handler.call(me, data);
            PMG.Utils.updateLoginData(data);
            PMG.app.changeView(view.targetview);
        },

        openQuarantineLinkWindow: function () {
            let me = this;
            me.lookup('loginwindow').setVisible(false);
            Ext.create('Proxmox.window.Edit', {
                title: gettext('Request Quarantine Link'),
                url: '/quarantine/sendlink',
                isCreate: true,
                submitText: gettext('OK'),
                method: 'POST',
                items: [
                    {
                        xtype: 'proxmoxtextfield',
                        name: 'mail',
                        fieldLabel: gettext('Your E-Mail'),
                    },
                ],
                listeners: {
                    destroy: function () {
                        me.lookup('loginwindow').show(true);
                    },
                },
            }).show();
        },

        control: {
            'field[name=lang]': {
                change: function (f, value) {
                    let dt = Ext.Date.add(new Date(), Ext.Date.YEAR, 10);
                    Ext.util.Cookies.set('PMGLangCookie', value, dt);

                    let loginwin = this.lookupReference('loginwindow');
                    loginwin.mask(gettext('Please wait...'), 'x-mask-loading');
                    window.location.reload();
                },
            },
            'field[name=realm]': {
                change: function (f, value) {
                    let record = f.store.getById(value);
                    if (!record) {
                        return;
                    }
                    let data = record.data;
                    this.getViewModel().set('oidc', data.type === 'oidc');
                },
            },
            'button[reference=quarantineButton]': {
                click: 'openQuarantineLinkWindow',
            },
            'button[reference=loginButton]': {
                click: 'submitForm',
            },
            'window[reference=loginwindow]': {
                show: function () {
                    let me = this;
                    let view = me.getView();
                    if (view.targetview !== 'quarantineview') {
                        let sp = Ext.state.Manager.getProvider();
                        let checkboxField = this.lookupReference('saveunField');
                        let unField = this.lookupReference('usernameField');

                        let checked = sp.get(checkboxField.getStateId());
                        checkboxField.setValue(checked);

                        if (checked === true) {
                            let username = sp.get(unField.getStateId());
                            unField.setValue(username);
                            let pwField = this.lookupReference('passwordField');
                            pwField.focus();
                        }

                        let auth = Proxmox.Utils.getOpenIDRedirectionAuthorization();
                        if (auth !== undefined) {
                            Proxmox.Utils.authClear();

                            let loginForm = this.lookupReference('loginForm');
                            loginForm.mask(
                                gettext('OpenID Connect login - please wait...'),
                                'x-mask-loading',
                            );

                            const redirectURL = location.origin;

                            Proxmox.Utils.API2Request({
                                url: '/api2/extjs/access/oidc/login',
                                params: {
                                    state: auth.state,
                                    code: auth.code,
                                    'redirect-url': redirectURL,
                                },
                                method: 'POST',
                                failure: function (response) {
                                    loginForm.unmask();
                                    let error = response.htmlStatus;
                                    Ext.MessageBox.alert(
                                        gettext('Error'),
                                        gettext('OpenID Connect login failed, please try again') +
                                            `<br>${error}`,
                                        () => {
                                            window.location = redirectURL;
                                        },
                                    );
                                },
                                success: function (response, options) {
                                    loginForm.unmask();
                                    let data = response.result.data;
                                    history.replaceState(null, '', redirectURL);
                                    me.success(data);
                                },
                            });
                        }
                    }
                },
            },
        },
    },

    plugins: 'viewport',

    layout: {
        type: 'border',
    },

    items: [
        {
            region: 'north',
            xtype: 'container',
            layout: {
                type: 'hbox',
                align: 'middle',
            },
            margin: '2 5 2 5',
            height: 38,
            items: [
                {
                    xtype: 'proxmoxlogo',
                },
                {
                    xtype: 'versioninfo',
                    makeApiCall: false,
                },
            ],
        },
        {
            region: 'center',
        },
        {
            xtype: 'window',
            closable: false,
            resizable: false,
            reference: 'loginwindow',
            autoShow: true,
            modal: true,
            width: 450,

            defaultFocus: 'usernameField',

            layout: {
                type: 'auto',
            },

            title: gettext('Proxmox Mail Gateway Login'),

            items: [
                {
                    xtype: 'form',
                    layout: {
                        type: 'form',
                    },
                    defaultButton: 'loginButton',
                    url: '/api2/extjs/access/ticket',
                    reference: 'loginForm',

                    fieldDefaults: {
                        labelAlign: 'right',
                        allowBlank: false,
                    },

                    items: [
                        {
                            xtype: 'textfield',
                            fieldLabel: gettext('User name'),
                            name: 'username',
                            itemId: 'usernameField',
                            reference: 'usernameField',
                            stateId: 'login-username',
                            inputAttrTpl: 'autocomplete=username',
                            bind: {
                                visible: '{!oidc}',
                                disabled: '{oidc}',
                            },
                        },
                        {
                            xtype: 'textfield',
                            inputType: 'password',
                            fieldLabel: gettext('Password'),
                            name: 'password',
                            reference: 'passwordField',
                            inputAttrTpl: 'autocomplete=current-password',
                            bind: {
                                visible: '{!oidc}',
                                disabled: '{oidc}',
                            },
                        },
                        {
                            xtype: 'pmxRealmComboBox',
                            reference: 'realmfield',
                            name: 'realm',
                            baseUrl: '/access/auth-realm',
                        },
                        {
                            xtype: 'proxmoxLanguageSelector',
                            fieldLabel: gettext('Language'),
                            value: Ext.util.Cookies.get('PMGLangCookie') || 'en',
                            name: 'lang',
                            submitValue: false,
                        },
                    ],
                    buttons: [
                        {
                            xtype: 'checkbox',
                            fieldLabel: gettext('Save User name'),
                            name: 'saveusername',
                            reference: 'saveunField',
                            stateId: 'login-saveusername',
                            labelAlign: 'right',
                            labelWidth: 150,
                            submitValue: false,
                            bind: {
                                visible: '{showSaveUser}',
                            },
                        },
                        {
                            text: gettext('Request Quarantine Link'),
                            reference: 'quarantineButton',
                        },
                        {
                            bind: {
                                text: '{buttonText}',
                            },
                            reference: 'loginButton',
                        },
                    ],
                },
            ],
        },
    ],
});
Ext.define('PMG.RoleSelector', {
    extend: 'Proxmox.form.KVComboBox',
    alias: 'widget.pmgRoleSelector',

    comboItems: [
        ['admin', PMG.Utils.format_user_role('admin')],
        ['helpdesk', PMG.Utils.format_user_role('helpdesk')],
        ['qmanager', PMG.Utils.format_user_role('qmanager')],
        ['audit', PMG.Utils.format_user_role('audit')],
    ],
});
Ext.define('PMG.ServerStatus', {
    extend: 'Ext.panel.Panel',
    alias: 'widget.pmgServerStatus',

    title: gettext('Status'),

    border: false,

    scrollable: true,

    bodyPadding: 5,
    defaults: {
        width: 700,
        padding: 5,
        columnWidth: 1,
    },

    layout: 'column',

    controller: {
        xclass: 'Ext.app.ViewController',

        openConsole: function () {
            Proxmox.Utils.openXtermJsViewer('shell', 0, Proxmox.NodeName);
        },

        nodeCommand: function (cmd) {
            Proxmox.Utils.API2Request({
                params: {
                    command: cmd,
                },
                url: `/nodes/${Proxmox.NodeName}/status`,
                method: 'POST',
                waitMsgTarget: this.getView(),
                failure: (response) => Ext.Msg.alert(gettext('Error'), response.htmlStatus),
            });
        },

        nodeShutdown: function () {
            this.nodeCommand('shutdown');
        },

        nodeReboot: function () {
            this.nodeCommand('reboot');
        },
    },

    tbar: [
        {
            text: gettext('Package versions'),
            iconCls: 'fa fa-gift',
            handler: () =>
                Proxmox.Utils.checked_command(() => {
                    Ext.create('Proxmox.window.PackageVersions', {
                        autoShow: true,
                    });
                }),
        },
        {
            text: gettext('Console'),
            iconCls: 'fa fa-terminal',
            handler: 'openConsole',
        },
        {
            xtype: 'proxmoxButton',
            text: gettext('Restart'),
            dangerous: true,
            confirmMsg: `${gettext('Node')} '${Proxmox.NodeName}' - ${gettext('Restart')}`,
            handler: 'nodeReboot',
            iconCls: 'fa fa-undo',
        },
        {
            xtype: 'proxmoxButton',
            text: gettext('Shutdown'),
            dangerous: true,
            confirmMsg: `${gettext('Node')} '${Proxmox.NodeName}' - ${gettext('Shutdown')}`,
            handler: 'nodeShutdown',
            iconCls: 'fa fa-power-off',
        },
        '->',
        {
            xtype: 'proxmoxRRDTypeSelector',
        },
    ],

    initComponent: function () {
        let me = this;

        let nodename = Proxmox.NodeName;
        let rrdstore = Ext.create('Proxmox.data.RRDStore', {
            rrdurl: `/api2/json/nodes/${nodename}/rrddata`,
            fields: [
                { type: 'number', name: 'loadavg' },
                { type: 'number', name: 'maxcpu' },
                {
                    type: 'number',
                    name: 'cpu',
                    convert: (val) => val * 100,
                },
                {
                    type: 'number',
                    name: 'iowait',
                    convert: (val) => val * 100,
                },
                { type: 'number', name: 'memtotal' },
                { type: 'number', name: 'memused' },
                { type: 'number', name: 'swaptotal' },
                { type: 'number', name: 'swapused' },
                { type: 'number', name: 'roottotal' },
                { type: 'number', name: 'rootused' },
                { type: 'number', name: 'netin' },
                { type: 'number', name: 'netout' },
                { type: 'date', dateFormat: 'timestamp', name: 'time' },
            ],
        });

        Ext.apply(me, {
            items: [
                {
                    xtype: 'proxmoxRRDChart',
                    title: gettext('CPU usage'),
                    unit: 'percent',
                    fields: ['cpu', 'iowait'],
                    fieldTitles: [gettext('CPU usage'), gettext('IO delay')],
                    store: rrdstore,
                },
                {
                    xtype: 'proxmoxRRDChart',
                    title: gettext('Server load'),
                    fields: ['loadavg'],
                    fieldTitles: [gettext('Load average')],
                    store: rrdstore,
                },
                {
                    xtype: 'proxmoxRRDChart',
                    title: gettext('Memory usage'),
                    unit: 'bytes',
                    fields: ['memtotal', 'memused'],
                    fieldTitles: [gettext('Total'), gettext('Used')],
                    store: rrdstore,
                },
                {
                    xtype: 'proxmoxRRDChart',
                    title: gettext('Swap usage'),
                    unit: 'bytes',
                    fields: ['swaptotal', 'swapused'],
                    fieldTitles: [gettext('Total'), gettext('Used')],
                    store: rrdstore,
                },
                {
                    xtype: 'proxmoxRRDChart',
                    title: gettext('Network traffic'),
                    unit: 'bytespersecond',
                    fields: ['netin', 'netout'],
                    fieldTitles: [gettext('Ingress'), gettext('Egress')],
                    store: rrdstore,
                },
                {
                    xtype: 'proxmoxRRDChart',
                    title: gettext('Disk usage'),
                    unit: 'bytes',
                    fields: ['roottotal', 'rootused'],
                    fieldTitles: [gettext('Total'), gettext('Used')],
                    store: rrdstore,
                },
            ],
            listeners: {
                resize: (panel) => Proxmox.Utils.updateColumnWidth(panel),
                activate: () => rrdstore.startUpdate(),
                destroy: () => rrdstore.stopUpdate(),
            },
        });

        me.callParent();

        let sp = Ext.state.Manager.getProvider();
        me.mon(sp, 'statechange', function (provider, key, value) {
            if (key !== 'summarycolumns') {
                return;
            }
            Proxmox.Utils.updateColumnWidth(me);
        });
    },
});
Ext.define('PMG.ServerAdministration', {
    extend: 'Ext.tab.Panel',
    alias: 'widget.pmgServerAdministration',

    title: gettext('Server Administration'),

    border: false,
    defaults: { border: false },

    controller: {
        xclass: 'Ext.app.ViewController',

        init: function (view) {
            var upgradeBtn = view.lookupReference('upgradeBtn');
            upgradeBtn.setDisabled(!(Proxmox.UserName && Proxmox.UserName === 'root@pam'));
        },
    },

    items: [
        {
            xtype: 'pmgServerStatus',
            itemId: 'status',
            iconCls: 'fa fa-area-chart',
        },
        {
            xtype: 'proxmoxNodeServiceView',
            title: gettext('Services'),
            itemId: 'services',
            iconCls: 'fa fa-cogs',
            startOnlyServices: {
                syslog: true,
                pmgproxy: true,
                pmgdaemon: true,
            },
            nodename: Proxmox.NodeName,
        },
        {
            xtype: 'proxmoxNodeAPT',
            title: gettext('Updates'),
            iconCls: 'fa fa-refresh',
            upgradeBtn: {
                xtype: 'button',
                reference: 'upgradeBtn',
                disabled: true,
                text: gettext('Upgrade'),
                handler: function () {
                    Proxmox.Utils.openXtermJsViewer('upgrade', 0, Proxmox.NodeName);
                },
            },
            itemId: 'updates',
            nodename: Proxmox.NodeName,
        },
        {
            xtype: 'proxmoxNodeAPTRepositories',
            title: gettext('Repositories'),
            iconCls: 'fa fa-files-o',
            itemId: 'aptrepositories',
            nodename: 'localhost',
            product: 'Proxmox Mail Gateway',
            onlineHelp: 'pmg_package_repositories',
        },
        {
            xtype: 'proxmoxJournalView',
            itemId: 'logs',
            iconCls: 'fa fa-list',
            title: gettext('Syslog'),
            url: '/api2/extjs/nodes/' + Proxmox.NodeName + '/journal',
        },
        {
            xtype: 'proxmoxNodeTasks',
            itemId: 'tasks',
            iconCls: 'fa fa-list-alt',
            title: gettext('Tasks'),
            height: 'auto',
            nodename: Proxmox.NodeName,
        },
    ],
});
Ext.define('PMG.LDAPProfileSelector', {
    extend: 'Proxmox.form.ComboGrid',
    alias: 'widget.pmgLDAPProfileSelector',

    store: {
        fields: ['profile', 'disable', 'comment'],
        proxy: {
            type: 'proxmox',
            url: '/api2/json/config/ldap',
        },
        filterOnLoad: true,
        sorters: [
            {
                property: 'profile',
                direction: 'ASC',
            },
        ],
    },

    valueField: 'profile',
    displayField: 'profile',

    allowBlank: false,

    listConfig: {
        columns: [
            {
                header: gettext('Profile'),
                dataIndex: 'profile',
                hideable: false,
                width: 100,
            },
            {
                header: gettext('Enabled'),
                width: 80,
                sortable: true,
                dataIndex: 'disable',
                renderer: Proxmox.Utils.format_neg_boolean,
            },
            {
                header: gettext('Comment'),
                sortable: false,
                renderer: Ext.String.htmlEncode,
                dataIndex: 'comment',
                flex: 1,
            },
        ],
    },

    initComponent: function () {
        var me = this;

        me.callParent();

        me.store.load();
    },
});
Ext.define('PMG.LDAPGroupSelector', {
    extend: 'Ext.form.ComboBox',
    alias: 'widget.pmgLDAPGroupSelector',

    profile: undefined,

    queryMode: 'local',

    store: {
        fields: ['dn'],
        filterOnLoad: true,
        sorters: [
            {
                property: 'dn',
                direction: 'ASC',
            },
        ],
    },

    valueField: 'dn',
    displayField: 'dn',

    allowBlank: false,

    setProfile: function (profile, force) {
        var me = this;

        if (!force && (profile === undefined || profile === null || me.profile === profile)) {
            return;
        }

        me.profile = profile;

        me.setValue('');

        me.store.setProxy({
            type: 'proxmox',
            url: '/api2/json/config/ldap/' + me.profile + '/groups',
        });

        me.store.load();
    },

    initComponent: function () {
        var me = this;

        me.callParent();

        if (me.profile !== undefined) {
            me.setProfile(me.profile, true);
        }
    },
});
Ext.define('PMG.LDAPGroupInputPanel', {
    extend: 'Proxmox.panel.InputPanel',
    alias: 'widget.pmgLDAPGroupInputPanel',

    onGetValues: function (values) {
        if (values.mode === 'profile-any') {
            values.mode = 'any';
        } else if (values.mode === 'profile-none') {
            values.mode = 'none';
        }

        return values;
    },

    setValues: function (values) {
        let me = this;

        if (values.profile !== undefined) {
            if (values.mode === 'any') {
                values.mode = 'profile-any';
            } else if (values.mode === 'none') {
                values.mode = 'profile-none';
            }
        }

        if (values.profile !== undefined) {
            let groupField = this.lookupReference('groupField');
            groupField.setProfile(values.profile);
        }

        me.callParent([values]);
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        changeMode: function (f, value) {
            let groupField = this.lookupReference('groupField');
            groupField.setDisabled(value !== 'group');
            groupField.setVisible(value === 'group');
            let profileField = this.lookupReference('profileField');
            let enabled = value !== 'any' && value !== 'none';
            profileField.setDisabled(!enabled);
            profileField.setVisible(enabled);
        },

        changeProfile: function (f, value) {
            let groupField = this.lookupReference('groupField');
            groupField.setProfile(value);
        },

        control: {
            'field[name=mode]': {
                change: 'changeMode',
            },
            'field[name=profile]': {
                change: 'changeProfile',
            },
        },
    },

    items: [
        {
            xtype: 'proxmoxKVComboBox',
            name: 'mode',
            value: 'group',
            comboItems: [
                ['group', gettext('Group member')],
                ['profile-any', gettext('Existing LDAP address')],
                ['any', gettext('Existing LDAP address') + ', any profile'],
                ['profile-none', gettext('Unknown LDAP address')],
                ['none', gettext('Unknown LDAP address') + ', any profile'],
            ],
            fieldLabel: gettext('Match'),
        },
        {
            xtype: 'pmgLDAPProfileSelector',
            name: 'profile',
            reference: 'profileField',
            fieldLabel: gettext('Profile'),
        },
        {
            xtype: 'pmgLDAPGroupSelector',
            name: 'group',
            reference: 'groupField',
            fieldLabel: gettext('Group'),
        },
    ],
});

Ext.define('PMG.LDAPGroupEditor', {
    extend: 'Proxmox.window.Edit',
    alias: 'widget.pmgLDAPGroupEditor',
    onlineHelp: 'pmgconfig_ldap',

    width: 500,

    items: [{ xtype: 'pmgLDAPGroupInputPanel' }],
});
Ext.define('PMG.LDAPUserSelector', {
    extend: 'Proxmox.form.ComboGrid',
    alias: 'widget.pmgLDAPUserSelector',

    profile: undefined,

    store: {
        fields: ['account', 'pmail', 'dn'],
        filterOnLoad: true,
        sorters: [
            {
                property: 'account',
                direction: 'ASC',
            },
        ],
    },

    valueField: 'account',
    displayField: 'account',

    allowBlank: false,

    listConfig: {
        columns: [
            {
                header: gettext('Account'),
                dataIndex: 'account',
                hideable: false,
                width: 100,
            },
            {
                header: gettext('E-Mail'),
                dataIndex: 'pmail',
                width: 150,
            },
            {
                header: 'DN',
                dataIndex: 'dn',
                width: 200,
            },
        ],
    },

    setProfile: function (profile, force) {
        var me = this;

        if (!force && (profile === undefined || profile === null || me.profile === profile)) {
            return;
        }

        me.profile = profile;

        me.setValue('');

        me.store.setProxy({
            type: 'proxmox',
            url: '/api2/json/config/ldap/' + me.profile + '/users',
        });

        me.store.load();
    },

    initComponent: function () {
        var me = this;

        me.callParent();

        if (me.profile !== undefined) {
            me.setProfile(me.profile, true);
        }
    },
});
Ext.define('PMG.LDAPUserInputPanel', {
    extend: 'Proxmox.panel.InputPanel',
    alias: 'widget.pmgLDAPUserInputPanel',

    setValues: function (values) {
        var me = this;

        if (values.profile !== undefined) {
            let accountField = this.lookupReference('accountField');
            accountField.setProfile(values.profile);
        }

        me.callParent([values]);
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        changeProfile: function (f, value) {
            var accountField = this.lookupReference('accountField');
            accountField.setProfile(value);
        },

        control: {
            'field[name=profile]': {
                change: 'changeProfile',
            },
        },
    },

    items: [
        {
            xtype: 'pmgLDAPProfileSelector',
            name: 'profile',
            reference: 'profileField',
            fieldLabel: gettext('Profile'),
        },
        {
            xtype: 'pmgLDAPUserSelector',
            name: 'account',
            reference: 'accountField',
            fieldLabel: gettext('Account'),
        },
    ],
});

Ext.define('PMG.LDAPUserEditor', {
    extend: 'Proxmox.window.Edit',
    alias: 'widget.pmgLDAPUserEditor',
    onlineHelp: 'pmgconfig_ldap',

    width: 500,

    items: [{ xtype: 'pmgLDAPUserInputPanel' }],
});
Ext.define('PMG.RegexTester', {
    extend: 'Ext.form.FieldContainer',
    alias: 'widget.pmgRegexTester',

    // the field reference which holds the regex value
    // has to be a sibling of the RegexTester component
    regexFieldReference: undefined,

    // if true, wraps the regex with ^ and $
    wholeMatch: false,

    layout: 'hbox',
    submitValue: false,

    items: [
        {
            xtype: 'textfield',
            submitValue: false,
            name: 'teststring',
            isDirty: () => false,
            reset: Ext.emptyFn,
            flex: 1,
        },
        {
            margin: '0 0 0 5',
            xtype: 'button',
            text: 'Test',
            handler: function (btn) {
                let view = this.up();
                let regexField = view.up().child(`field[reference=${view.regexFieldReference}]`);

                let regex = regexField.getValue();
                if (view.wholeMatch) {
                    regex = `^${regex}$`;
                }

                Proxmox.Utils.API2Request({
                    url: '/api2/extjs/config/regextest',
                    waitMsgTarget: view.up('window'),
                    params: {
                        regex: regex,
                        text: view.down('textfield[name=teststring]').getValue(),
                    },
                    method: 'POST',
                    success: function (response) {
                        let elapsed = response.result.data;
                        Ext.Msg.show({
                            title: gettext('Success'),
                            message: gettext('OK') + ` (elapsed time: ${elapsed}ms)`,
                            buttons: Ext.Msg.OK,
                            icon: Ext.MessageBox.INFO,
                        });
                    },
                    failure: function (response, opts) {
                        Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                    },
                });
            },
        },
    ],

    initComponent: function () {
        let me = this;

        if (!me.regexFieldReference) {
            throw 'No regex field reference given';
        }

        me.callParent();
    },
});
Ext.define('pmg-object-group', {
    extend: 'Ext.data.Model',
    fields: ['id', 'name', 'info'],
    idProperty: 'id',
});

Ext.define('pmg-object-list', {
    extend: 'Ext.data.Model',
    fields: [
        'id',
        'descr',
        { name: 'otype', type: 'integer' },
        { name: 'receivertest', type: 'boolean' },
    ],
    idProperty: 'id',
});

Ext.define('PMG.ObjectGroupList', {
    extend: 'Ext.grid.GridPanel',
    alias: ['widget.pmgObjectGroupList'],

    ogclass: undefined, //  'who', 'when', 'what'

    subject: 'Object Group List', // please overwrite

    baseurl: undefined,

    enableButtons: true,

    inputItems: [
        {
            xtype: 'textfield',
            name: 'name',
            allowBlank: false,
            fieldLabel: gettext('Name'),
        },
        {
            xtype: 'textareafield',
            name: 'info',
            fieldLabel: gettext('Description'),
        },
    ],

    reload: function () {
        var me = this;

        me.store.load();
    },

    run_editor: function () {
        var me = this;

        var rec = me.selModel.getSelection()[0];
        if (!rec) {
            return;
        }

        var config = {
            url: '/api2/extjs' + me.baseurl + '/' + rec.data.id + '/config',
            onlineHelp: 'chapter_mailfilter',
            method: 'PUT',
            subject: me.subject,
            width: 400,
            items: me.inputItems,
        };

        var win = Ext.createWidget('proxmoxWindowEdit', config);

        win.load();
        win.on('destroy', me.reload, me);
        win.show();
    },

    initComponent: function () {
        var me = this;

        if (!me.ogclass) {
            throw 'ogclass not initialized';
        }

        me.baseurl = '/config/ruledb/' + me.ogclass;

        me.store = new Ext.data.Store({
            model: 'pmg-object-group',
            proxy: {
                type: 'proxmox',
                url: '/api2/json' + me.baseurl,
            },
            sorters: {
                property: 'name',
                direction: 'ASC',
            },
        });

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        var tbar = [
            {
                text: gettext('Create'),
                handler: function () {
                    Ext.createWidget('proxmoxWindowEdit', {
                        method: 'POST',
                        url: `/api2/extjs${me.baseurl}`,
                        onlineHelp: 'chapter_mailfilter',
                        isCreate: true,
                        width: 400,
                        subject: me.subject,
                        items: me.inputItems,
                        autoShow: true,
                        listeners: {
                            destroy: () => me.reload(),
                        },
                    });
                },
            },
            '-',
            {
                xtype: 'proxmoxButton',
                text: gettext('Edit'),
                disabled: true,
                selModel: me.selModel,
                handler: () => me.run_editor(),
            },
            {
                xtype: 'proxmoxStdRemoveButton',
                selModel: me.selModel,
                baseurl: me.baseurl,
                callback: () => me.reload(),
                getRecordName: (rec) => rec.data.name,
                waitMsgTarget: me,
            },
        ];

        Proxmox.Utils.monStoreErrors(me, me.store, true);

        if (me.enableButtons) {
            me.tbar = tbar;
        }

        Ext.apply(me, {
            columns: [
                {
                    header: gettext('Name'),
                    sortable: true,
                    flex: 1,
                    dataIndex: 'name',
                    renderer: Ext.String.htmlEncode,
                },
                {
                    header: gettext('Description'),
                    sortable: true,
                    flex: 1,
                    dataIndex: 'info',
                    renderer: Ext.String.htmlEncode,
                    hidden: true,
                },
            ],
            listeners: {
                itemdblclick: function () {
                    if (me.enableButtons) {
                        me.run_editor();
                    }
                },
                activate: function () {
                    me.reload();
                },
            },
        });

        me.callParent();

        me.reload(); // initial load
    },
});
Ext.define('PMG.ObjectGroup', {
    extend: 'Ext.grid.GridPanel',
    alias: 'widget.pmgObjectGroup',

    baseurl: undefined,

    otype_list: [],

    hideGroupInfo: false,
    showDirection: false, // only important for SMTP Welcomelist

    ogdata: undefined,
    objectClass: undefined,

    emptyText: gettext('Please select an object.'),

    setBaseUrl: function (baseurl) {
        let me = this;

        me.baseurl = baseurl;

        if (me.baseurl === undefined) {
            me.store.proxy.setUrl(undefined);
            me.store.setData([]);
            me.setButtonState(me.store, [], false);
        } else {
            let url = '/api2/json' + me.baseurl + '/objects';
            me.store.proxy.setUrl(url);
            me.store.load();
        }
    },

    setObjectInfo: function (ogdata) {
        let me = this;

        let mode = ogdata?.invert ? 'not' : '';
        mode += ogdata?.and ? 'all' : 'any';

        me.ogdata = ogdata;

        if (me.ogdata === undefined) {
            me.down('#oginfo').update(me.emptyText);
            me.down('#separator').setHidden(true);
            me.down('#modeBox').setHidden(true);
            me.down('#whatWarning').setHidden(true);
        } else {
            let html =
                '<b style="font-weight: bold;">' + Ext.String.htmlEncode(me.ogdata.name) + '</b>';
            html += '<br><br>';
            html += Ext.String.htmlEncode(Ext.String.trim(me.ogdata.info));

            me.down('#oginfo').update(html);
            me.down('#ogdata').setHidden(false);
            me.down('#separator').setHidden(false);
            let modeSelector = me.down('#modeSelector');
            modeSelector.suspendEvents();
            me.down('#modeSelector').setValue(mode);
            modeSelector.resumeEvents();
            me.down('#modeBox').setHidden(false);
            me.down('#whatWarning').setHidden(me.objectClass !== 'what' || mode === 'any');
        }
    },

    setButtonState: function (store, records, success) {
        let me = this;
        if (!success || !me.baseurl) {
            me.down('#addMenuButton').setDisabled(true);
            return;
        }
        me.down('#addMenuButton').setDisabled(false);
    },

    initComponent: function () {
        let me = this;

        me.store = new Ext.data.Store({
            model: 'pmg-object-list',
            proxy: {
                type: 'proxmox',
            },
            sorters: [
                {
                    property: 'receivertest',
                },
                {
                    property: 'otype',
                    direction: 'ASC',
                },
            ],
        });

        me.columns = [
            {
                header: gettext('Type'),
                dataIndex: 'otype',
                renderer: PMG.Utils.format_otype,
                width: 200,
            },
        ];

        if (me.showDirection) {
            me.columns.push({
                header: gettext('Direction'),
                dataIndex: 'receivertest',
                renderer: function (value) {
                    return value ? PMG.Utils.receiverText : PMG.Utils.senderText;
                },
            });
        }

        me.columns.push({
            header: gettext('Value'),
            dataIndex: 'descr',
            renderer: Ext.String.htmlEncode,
            flex: 1,
        });

        let reload = function () {
            me.store.load();
        };

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        let remove_btn = Ext.createWidget('proxmoxStdRemoveButton', {
            selModel: me.selModel,
            getUrl: function (rec) {
                return me.baseurl + '/objects/' + rec.data.id;
            },
            callback: reload,
            getRecordName: function (rec) {
                return PMG.Utils.format_otype_subject(rec.data.otype) + ': ' + rec.data.descr;
            },
            waitMsgTarget: me,
        });

        let full_subject = function (subject, receivertest) {
            if (me.showDirection) {
                let direction = receivertest ? PMG.Utils.receiverText : PMG.Utils.senderText;

                return subject + ' (' + direction + ')';
            } else {
                return subject;
            }
        };

        let run_editor = function () {
            let rec = me.selModel.getSelection()[0];
            if (!rec) {
                return;
            }

            let editor = PMG.Utils.object_editors[rec.data.otype];
            if (!editor || editor.uneditable) {
                return;
            }

            let config = Ext.apply({ method: 'PUT' }, editor);
            config.subject = full_subject(editor.subject, rec.data.receivertest);
            config.url = me.baseurl + '/' + editor.subdir + '/' + rec.data.id;

            let win = Ext.createWidget(config);

            win.load();
            win.on('destroy', reload);
            win.show();
        };

        let menu_items = [];

        Ext.Array.each(me.otype_list, function (otype) {
            let editor = PMG.Utils.object_editors[otype];

            let config = Ext.apply({ method: 'POST' }, editor);
            config.subject = full_subject(editor.subject, editor.receivertest);

            menu_items.push({
                text: config.subject,
                iconCls: config.iconCls || 'fa fa-question-circle',
                handler: function () {
                    if (me.baseurl === undefined) {
                        return;
                    }
                    config.url = me.baseurl + '/' + editor.subdir;
                    let win = Ext.create(config);
                    win.on('destroy', reload);
                    win.show();
                },
            });
        });

        me.dockedItems = [];

        me.dockedItems.push({
            xtype: 'toolbar',
            dock: 'top',
            items: [
                {
                    text: gettext('Add'),
                    disabled: true,
                    itemId: 'addMenuButton',
                    menu: {
                        items: menu_items,
                    },
                },
                '-',
                {
                    xtype: 'proxmoxButton',
                    text: gettext('Edit'),
                    disabled: true,
                    selModel: me.selModel,
                    enableFn: function (rec) {
                        let editor = PMG.Utils.object_editors[rec.data.otype];
                        return editor && !editor.uneditable;
                    },
                    handler: run_editor,
                },
                remove_btn,
                '->',
                {
                    xtype: 'pmgFilterField',
                    filteredFields: [
                        {
                            name: 'otype',
                            renderer: (otype) => PMG.Utils.object_editors[otype].subject,
                        },
                        'descr',
                    ],
                },
            ],
        });

        me.dockedItems.push({
            dock: 'top',
            border: 1,
            layout: {
                type: 'hbox',
                align: 'stretch',
            },
            hidden: !!me.hideGroupInfo,
            itemId: 'ogdata',
            xtype: 'toolbar',
            items: [
                {
                    xtype: 'container',
                    itemId: 'modeBox',
                    hidden: true,
                    width: 220,
                    padding: 10,
                    layout: {
                        type: 'vbox',
                        align: 'stretch',
                    },
                    items: [
                        {
                            xtype: 'box',
                            html: `<b style="font-weight: bold;">${gettext('Match if')}</b>`,
                        },
                        {
                            xtype: 'pmgMatchModeSelector',
                            itemId: 'modeSelector',
                            padding: '10 0 0 0',
                            listeners: {
                                change: function (_field, value) {
                                    let invert = value.startsWith('not') ? 1 : 0;
                                    let and = value.endsWith('all') ? 1 : 0;

                                    Proxmox.Utils.API2Request({
                                        url: `${me.baseurl}/config`,
                                        method: 'PUT',
                                        params: {
                                            and,
                                            invert,
                                        },
                                        success: () => {
                                            me.fireEvent('modeUpdate', me, !!and, !!invert);
                                            me.down('#whatWarning').setHidden(
                                                me.objectClass !== 'what' || value === 'any',
                                            );
                                        },
                                    });
                                },
                            },
                        },
                    ],
                },
                {
                    xtype: 'tbseparator',
                    itemId: 'separator',
                    hidden: true,
                },
                {
                    xtype: 'component',
                    flex: 1,
                    itemId: 'oginfo',
                    style: { 'white-space': 'pre' },
                    padding: 10,
                    html: me.emptyText,
                    listeners: {
                        dblclick: {
                            fn: function (e, t) {
                                if (me.ogdata === undefined) {
                                    return;
                                }
                                me.fireEvent('dblclickOGInfo', me, e, t, me.ogdata);
                            },
                            element: 'el',
                            scope: this,
                        },
                    },
                },
            ],
        });

        me.dockedItems.push({
            dock: 'top',
            border: 1,
            hidden: true,
            itemId: 'whatWarning',
            bodyPadding: 5,
            items: {
                xtype: 'displayfield',
                margin: 0,
                value: gettext(
                    "Caution: 'What Objects' match each mail part separately, so be careful with any option besides 'Any matches'.",
                ),
                userCls: 'pmx-hint',
            },
        });

        Proxmox.Utils.monStoreErrors(me, me.store, true);

        Ext.apply(me, {
            run_editor: run_editor,
            listeners: {
                itemdblclick: run_editor,
                activate: reload,
            },
        });

        me.callParent();

        me.mon(me.store, 'load', me.setButtonState, me);

        if (me.baseurl) {
            me.setBaseUrl(me.baseurl); // configure store, load()
        }
    },
});
Ext.define('PMG.ObjectGroupSelector', {
    extend: 'Ext.window.Window',
    alias: 'widget.pmgObjectGroupSelector',

    width: 600,
    layout: 'auto',
    modal: true,
    bodyPadding: 5,

    rulegroup: undefined,

    initComponent: function () {
        var me = this;

        if (!me.rulegroup) {
            throw 'undefined rulegroup';
        }

        var ogclass;

        if (me.rulegroup === 'from') {
            ogclass = 'who';
            me.title = gettext('From');
        } else if (me.rulegroup === 'to') {
            ogclass = 'who';
            me.title = gettext('To');
        } else if (me.rulegroup === 'when') {
            ogclass = 'when';
            me.title = gettext('When');
        } else if (me.rulegroup === 'what') {
            ogclass = 'what';
            me.title = gettext('What');
        } else if (me.rulegroup === 'action') {
            ogclass = 'action';
            me.title = gettext('Action');
        } else {
            throw 'unknown rulegroup';
        }

        if (me.rulegroup === 'action') {
            me.items = {
                xtype: 'pmgActionList',
                title: undefined,
                enableButtons: false,
                border: true,
                listeners: {
                    itemdblclick: function (view, rec) {
                        me.fireEvent('selectObjectGroup', me, rec);
                    },
                },
            };
        } else {
            me.items = {
                xtype: 'pmgObjectGroupList',
                enableButtons: false,
                ogclass: ogclass,
                listeners: {
                    itemdblclick: function (view, rec) {
                        me.fireEvent('selectObjectGroup', me, rec);
                    },
                },
            };
        }

        me.callParent();
    },
});
Ext.define('PMG.ObjectGroupConfiguration', {
    extend: 'Ext.panel.Panel',

    ogclass: undefined, // who, when, what
    otype_list: [],

    layout: 'border',
    border: false,

    initComponent: function () {
        var me = this;

        if (me.ogclass === undefined) {
            throw 'undefined object group class';
        }

        if (!PMG.Utils.oclass_text[me.ogclass]) {
            throw 'unknown object group class';
        }

        var left = Ext.create('PMG.ObjectGroupList', {
            width: 250,
            ogclass: me.ogclass,
            subject: PMG.Utils.oclass_text[me.ogclass],
            title: PMG.Utils.oclass_text[me.ogclass],
            border: false,
            split: true,
            region: 'west',
        });

        var right = Ext.create('PMG.ObjectGroup', {
            otype_list: me.otype_list,
            objectClass: me.ogclass,
            border: false,
            region: 'center',
            listeners: {
                dblclickOGInfo: function (w, e, t, ogdata) {
                    // test if the correct groups is selected (just to be sure)
                    var rec = left.selModel.getSelection()[0];
                    if (rec && rec.data && rec.data.id === ogdata.id) {
                        left.run_editor();
                    }
                },
                modeUpdate: (_cmp, and, invert) => {
                    let rec = left.selModel.getSelection()[0];
                    rec.set('and', and);
                    rec.set('invert', invert);
                    rec.commit();
                },
            },
        });

        me.mon(left.store, 'refresh', function () {
            var rec = left.selModel.getSelection()[0];
            if (!(rec && rec.data && rec.data.id)) {
                return;
            }
            right.setObjectInfo(rec.data);
        });

        me.mon(left.selModel, 'selectionchange', function () {
            var rec = left.selModel.getSelection()[0];
            if (!(rec && rec.data && rec.data.id)) {
                right.setObjectInfo(undefined);
                right.setBaseUrl(undefined);
                return;
            }
            right.setObjectInfo(rec.data);
            var baseurl = '/config/ruledb/' + me.ogclass + '/' + rec.data.id;
            right.setBaseUrl(baseurl);
        });

        me.items = [left, right];

        me.callParent();
    },
});

Ext.define('PMG.WhoConfiguration', {
    extend: 'PMG.ObjectGroupConfiguration',
    xtype: 'pmgWhoConfiguration',

    ogclass: 'who',
    otype_list: [1000, 1001, 1002, 1003, 1004, 1005, 1006],
});

Ext.define('PMG.WhenConfiguration', {
    extend: 'PMG.ObjectGroupConfiguration',
    xtype: 'pmgWhenConfiguration',

    ogclass: 'when',
    otype_list: [2000],
});

Ext.define('PMG.WhatConfiguration', {
    extend: 'PMG.ObjectGroupConfiguration',
    xtype: 'pmgWhatConfiguration',

    ogclass: 'what',
    otype_list: [3000, 3001, 3002, 3003, 3004, 3005, 3006],
});
Ext.define('pmg-action-list', {
    extend: 'Ext.data.Model',
    fields: ['id', 'name', 'info', 'descr', 'editable', { name: 'otype', type: 'integer' }],
    idProperty: 'id',
});

Ext.define('PMG.ActionList', {
    extend: 'Ext.grid.GridPanel',
    alias: ['widget.pmgActionList'],

    title: PMG.Utils.oclass_text.action,
    border: false,

    baseurl: '/config/ruledb/action',

    otype_list: [4002, 4003, 4005, 4007, 4009],

    enableButtons: true,

    initComponent: function () {
        var me = this;

        me.store = new Ext.data.Store({
            model: 'pmg-action-list',
            proxy: {
                type: 'proxmox',
                url: '/api2/json' + me.baseurl + '/objects',
            },
            sorters: {
                property: 'name',
                direction: 'ASC',
            },
        });

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        var reload = function () {
            me.store.load();
        };

        var run_editor = function () {
            var rec = me.selModel.getSelection()[0];
            if (!rec) {
                return;
            }

            var editor = PMG.Utils.object_editors[rec.data.otype];
            if (!editor) {
                return;
            }

            var config = Ext.apply({ method: 'PUT' }, editor);

            config.url = me.baseurl + '/' + editor.subdir + '/' + rec.data.id;

            var win = Ext.createWidget('proxmoxWindowEdit', config);

            win.load();
            win.on('destroy', reload);
            win.show();
        };

        var remove_btn = Ext.createWidget('proxmoxStdRemoveButton', {
            selModel: me.selModel,
            getUrl: function (rec) {
                return me.baseurl + '/objects/' + rec.data.id;
            },
            enableFn: (rec) => !!rec.data.editable,
            callback: reload,
            getRecordName: function (rec) {
                return rec.data.descr;
            },
            waitMsgTarget: me,
        });

        var menu_items = [];

        Ext.Array.each(me.otype_list, function (otype) {
            var editor = PMG.Utils.object_editors[otype];

            var config = Ext.apply({ method: 'POST' }, editor);

            config.isCreate = true;
            menu_items.push({
                text: config.subject,
                handler: function () {
                    if (me.baseurl === undefined) {
                        return;
                    }
                    config.url = me.baseurl + '/' + editor.subdir;
                    var win = Ext.createWidget('proxmoxWindowEdit', config);
                    win.on('destroy', reload);
                    win.show();
                },
            });
        });

        var tbar = [
            {
                text: gettext('Add'),
                menu: new Ext.menu.Menu({
                    items: menu_items,
                }),
            },
            {
                xtype: 'proxmoxButton',
                text: gettext('Edit'),
                disabled: true,
                selModel: me.selModel,
                enableFn: (rec) => !!rec.data.editable,
                handler: run_editor,
            },
            remove_btn,
        ];

        Proxmox.Utils.monStoreErrors(me, me.store, true);

        if (me.enableButtons) {
            me.tbar = tbar;
        }

        Ext.apply(me, {
            run_editor: run_editor,
            columns: [
                {
                    header: gettext('Name'),
                    sortable: true,
                    width: 200,
                    dataIndex: 'name',
                    renderer: Ext.String.htmlEncode,
                },
                {
                    header: gettext('Description'),
                    sortable: true,
                    width: 300,
                    dataIndex: 'descr',
                    renderer: Ext.String.htmlEncode,
                },
                {
                    header: gettext('Comment'),
                    sortable: false,
                    flex: 1,
                    dataIndex: 'info',
                    renderer: Ext.String.htmlEncode,
                },
                {
                    header: gettext('Editable'),
                    dataIndex: 'editable',
                    renderer: Proxmox.Utils.format_boolean,
                },
            ],
            listeners: {
                itemdblclick: function () {
                    if (me.enableButtons) {
                        run_editor();
                    }
                },
                activate: reload,
            },
        });

        me.callParent();

        reload(); // initial load
    },
});
Ext.define('PMG.RuleInfo', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgRuleInfo',

    controller: {
        xclass: 'Ext.app.ViewController',

        setBaseUrl: function (baseurl) {
            var me = this;
            me.getViewModel().set('baseurl', baseurl);
            me.reload();
        },

        reload: function () {
            var me = this;
            var viewmodel = me.getViewModel();
            var baseurl = viewmodel.get('baseurl');

            if (!baseurl) {
                me.setRuleInfo(undefined);
                return;
            }

            Proxmox.Utils.API2Request({
                url: baseurl + '/config',
                method: 'GET',
                success: function (response, opts) {
                    me.setRuleInfo(response.result.data);
                },
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        removeObjectGroup: function (rec) {
            var me = this;
            Ext.Msg.confirm(
                gettext('Confirm'),
                Ext.String.format(
                    gettext('Are you sure you want to remove entry {0}'),
                    "'" + rec.data.name + "'",
                ),
                function (button) {
                    if (button === 'yes') {
                        Proxmox.Utils.API2Request({
                            url:
                                me.getViewModel().get('baseurl') +
                                '/' +
                                rec.data.oclass +
                                '/' +
                                rec.data.typeid,
                            method: 'DELETE',
                            waitMsgTarget: me.getView(),
                            callback: function () {
                                me.reload();
                            },
                            failure: function (response, opts) {
                                Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                            },
                        });
                    }
                },
            );
        },

        addObjectGroup: function (type, record) {
            var me = this;
            var baseurl = me.getViewModel().get('baseurl');
            var url = baseurl + '/' + type;
            var id = type === 'action' ? record.data.ogroup : record.data.id;
            Proxmox.Utils.API2Request({
                url: url,
                params: { ogroup: id },
                method: 'POST',
                waitMsgTarget: me.getView(),
                callback: function () {
                    me.reload();
                },
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        setRuleInfo: function (ruledata) {
            var me = this;

            var viewmodel = me.getViewModel();

            if (ruledata === undefined) {
                viewmodel.set('selectedRule', null);
                viewmodel.get('objects').setData([]);
            } else {
                viewmodel.set('selectedRule', ruledata);

                let data = {
                    leaf: false,
                    expanded: true,
                    children: [],
                };
                Ext.Array.each(['from', 'to', 'when', 'what', 'action'], function (oc) {
                    var store = viewmodel.get(oc + 'objects');
                    if (ruledata[oc] === undefined || store === undefined) {
                        return;
                    }

                    // we build a filter for the objects,
                    // which are already added to the rule,
                    // so what we only show the ones,
                    // which are still available

                    var ids = Ext.Array.pluck(ruledata[oc], 'id');
                    // for the actions, we have a different id field
                    var idField = oc === 'action' ? 'ogroup' : 'id';
                    store.clearFilter();
                    store.addFilter({
                        filterFn: function (record) {
                            // FIXME
                            // actions have the ogroup as a string
                            // -> parseInt
                            return ids.indexOf(parseInt(record.data[idField], 10)) === -1;
                        },
                    });
                    store.load();

                    let group = {
                        name: oc,
                        oclass: oc,
                        type: true,
                        invert: ruledata[`${oc}-invert`],
                        and: ruledata[`${oc}-and`],
                        leaf: false,
                        expanded: true,
                        expandable: false,
                        children: [],
                    };
                    Ext.Array.each(ruledata[oc], function (og) {
                        group.children.push({
                            oclass: oc,
                            name: og.name,
                            typeid: og.id,
                            leaf: true,
                        });
                    });

                    if (group.children.length) {
                        data.children.push(group);
                    }
                });
                viewmodel.get('objects').setRoot(data);
            }
        },

        removeIconClick: function (gridView, rowindex, colindex, button, event, record) {
            var me = this;
            me.removeObjectGroup(record);
        },

        removeDrop: function (gridView, data, overModel) {
            var me = this;
            var record = data.records[0]; // only one
            me.removeObjectGroup(record);
            return true;
        },

        addIconClick: function (gridView, rowindex, colindex, button, event, record) {
            var me = this;
            me.addObjectGroup(gridView.panel.type, record);
            return true;
        },

        addDrop: function (gridView, data, overModel) {
            var me = this;
            var record = data.records[0]; // only one
            me.addObjectGroup(data.view.panel.type, record);
            return true;
        },

        updateMode: function (field, value) {
            let me = this;
            let vm = me.getViewModel();
            let oclass = field.getWidgetRecord().data.oclass;

            let params = {};
            params[`${oclass}-invert`] = value.startsWith('not') ? 1 : 0;
            params[`${oclass}-and`] = value.endsWith('all') ? 1 : 0;

            Proxmox.Utils.API2Request({
                url: `${vm.get('baseurl')}/config`,
                method: 'PUT',
                params,
                success: () => me.reload(),
            });
        },

        control: {
            'treepanel[reference=usedobjects]': {
                drop: 'addDrop',
            },
            'tabpanel[reference=availobjects] > grid': {
                drop: 'removeDrop',
            },
            pmgMatchModeSelector: {
                change: 'updateMode',
            },
        },
    },

    viewModel: {
        data: {
            baseurl: '',
        },

        stores: {
            objects: {
                type: 'tree',
                fields: ['oclass', 'name', 'typeid'],
                groupField: 'oclass',
                sorters: 'name',
            },

            actionobjects: {
                model: 'pmg-action-list',
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/config/ruledb/action/objects',
                },
                sorters: 'name',
            },
            fromobjects: {
                model: 'pmg-object-group',
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/config/ruledb/who',
                },
                sorters: 'name',
            },
            toobjects: {
                model: 'pmg-object-group',
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/config/ruledb/who',
                },
                sorters: 'name',
            },
            whatobjects: {
                model: 'pmg-object-group',
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/config/ruledb/what',
                },
                sorters: 'name',
            },
            whenobjects: {
                model: 'pmg-object-group',
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/config/ruledb/when',
                },
                sorters: 'name',
            },
        },
    },

    defaults: {
        padding: '5 10 5 10',
    },

    bodyPadding: '5 0 5 0',

    layout: {
        type: 'vbox',
        align: 'stretch',
    },

    scrollable: true,

    items: [
        {
            xtype: 'panel',
            bodyPadding: '10 10 10 10',
            data: {
                name: '',
            },
            bind: {
                data: {
                    name: '{selectedRule.name:htmlEncode}',
                    priority: '{selectedRule.priority}',
                    active: '{selectedRule.active}',
                    direction: '{selectedRule.direction}',
                    selected: '{selectedRule}',
                },
            },
            tpl: [
                '<tpl if="selected">',
                '<b>{name}</b><br><br>',
                gettext('Priority') + ': {priority}<br>',
                gettext('Direction') +
                    ': {[PMG.Utils.format_rule_direction(values.direction)]}<br>',
                gettext('Active') + ': {[Proxmox.Utils.format_boolean(values.active)]}<br>',
                '<tpl else>',
                gettext('Please select a rule.'),
                '</tpl>',
            ],
        },
        {
            xtype: 'treepanel',
            reference: 'usedobjects',
            hidden: true,
            emptyText: gettext('No Objects'),

            title: gettext('Used Objects'),
            rootVisible: false,
            useArrows: true,
            rowLines: true,
            userCls: 'pmx-rule-tree',

            viewConfig: {
                getRowClass: (record) => (record.data.type ? 'pmx-type-row' : ''),
                plugins: {
                    ptype: 'gridviewdragdrop',
                    copy: true,
                    dragGroup: 'usedobjects',
                    dropGroup: 'unusedobjects',

                    // do not show default grid dragdrop behaviour
                    dropZone: {
                        indicatorHtml: '',
                        indicatorCls: '',
                        handleNodeDrop: Ext.emptyFn,
                    },
                },
            },

            columns: [
                {
                    header: gettext('Name'),
                    dataIndex: 'name',
                    xtype: 'treecolumn',
                    renderer: PMG.Utils.format_oclass,
                    sorter: function (a, b) {
                        if (a.data.type && b.data.type) {
                            return a.data.oclass.localeCompare(b.data.oclass);
                        }
                        return a.data.text.localeCompare(b.data.text);
                    },
                    flex: 1,
                },
                {
                    header: gettext('Match if'),
                    xtype: 'widgetcolumn',
                    width: 200,
                    widget: {
                        xtype: 'pmgMatchModeSelector',
                    },
                    onWidgetAttach: function (col, widget, rec) {
                        if (rec.data.type && rec.data.oclass !== 'action') {
                            let mode = rec.data.invert ? 'not' : '';
                            mode += rec.data.and ? 'all' : 'any';
                            widget.suspendEvents();
                            widget.setValue(mode);
                            widget.resumeEvents();
                            widget.setHidden(false);
                        } else {
                            widget.setHidden(true);
                        }
                    },
                },
                {
                    text: '',
                    xtype: 'actioncolumn',
                    align: 'center',
                    width: 40,
                    items: [
                        {
                            tooltip: gettext('Remove'),
                            isActionDisabled: (v, rI, cI, i, rec) => rec.data.type,
                            getClass: (v, mD, { data }) =>
                                data.type ? 'pmx-hidden' : 'fa fa-fw fa-minus-circle',
                            handler: 'removeIconClick',
                        },
                    ],
                },
            ],

            bind: {
                store: '{objects}',
                hidden: '{!selectedRule}',
            },
        },
        {
            xtype: 'tabpanel',
            title: gettext('Available Objects'),
            reference: 'availobjects',
            hidden: true,
            bind: {
                hidden: '{!selectedRule}',
            },
            defaults: {
                xtype: 'grid',
                emptyText: gettext('No Objects'),
                viewConfig: {
                    plugins: {
                        ptype: 'gridviewdragdrop',
                        dragGroup: 'unusedobjects',
                        dropGroup: 'usedobjects',

                        // do not show default grid dragdrop behaviour
                        dropZone: {
                            indicatorHtml: '',
                            indicatorCls: '',
                            handleNodeDrop: Ext.emptyFn,
                        },
                    },
                },
                columns: [
                    {
                        header: gettext('Name'),
                        dataIndex: 'name',
                        flex: 1,
                    },
                    {
                        text: '',
                        xtype: 'actioncolumn',
                        align: 'center',
                        width: 40,
                        items: [
                            {
                                iconCls: 'fa fa-fw fa-plus-circle',
                                tooltip: gettext('Add'),
                                handler: 'addIconClick',
                            },
                        ],
                    },
                ],
            },
            items: [
                {
                    title: gettext('Action'),
                    bind: {
                        store: '{actionobjects}',
                    },
                    type: 'action',
                    iconCls: 'fa fa-flag',
                },
                {
                    title: gettext('From'),
                    iconCls: 'fa fa-user-circle',
                    type: 'from',
                    bind: {
                        store: '{fromobjects}',
                    },
                },
                {
                    title: gettext('To'),
                    iconCls: 'fa fa-user-circle',
                    type: 'to',
                    bind: {
                        store: '{toobjects}',
                    },
                },
                {
                    title: gettext('What'),
                    iconCls: 'fa fa-cube',
                    type: 'what',
                    bind: {
                        store: '{whatobjects}',
                    },
                },
                {
                    title: gettext('When'),
                    iconCls: 'fa fa-clock-o',
                    type: 'when',
                    bind: {
                        store: '{whenobjects}',
                    },
                },
            ],
        },
    ],
});
Ext.define('PMG.RuleEditor', {
    extend: 'Proxmox.window.Edit',
    xtype: 'ruleeditwindow',
    onlineHelp: 'chapter_mailfilter',

    url: undefined,

    method: 'PUT',

    subject: gettext('Rules'),

    width: 400,

    items: [
        {
            xtype: 'textfield',
            name: 'name',
            allowBlank: false,
            fieldLabel: gettext('Name'),
        },
        {
            xtype: 'proxmoxintegerfield',
            name: 'priority',
            allowBlank: false,
            minValue: 0,
            maxValue: 100,
            fieldLabel: gettext('Priority'),
        },
        {
            xtype: 'proxmoxKVComboBox',
            name: 'direction',
            comboItems: [
                [0, PMG.Utils.rule_direction_text[0]],
                [1, PMG.Utils.rule_direction_text[1]],
                [2, PMG.Utils.rule_direction_text[2]],
            ],
            value: 2,
            fieldLabel: gettext('Direction'),
        },
        {
            xtype: 'proxmoxcheckbox',
            name: 'active',
            defaultValue: 0,
            uncheckedValue: 0,
            checked: false,
            fieldLabel: gettext('Active'),
        },
    ],
});
Ext.define('PMG.MainView', {
    extend: 'Ext.container.Container',
    xtype: 'mainview',

    title: 'Proxmox Mail Gateway',

    controller: {
        xclass: 'Ext.app.ViewController',
        routes: {
            ':path:subpath': {
                action: 'changePath',
                before: 'beforeChangePath',
                conditions: {
                    ':path': '(?:([%a-zA-Z0-9\\-\\_\\s,]+))',
                    ':subpath': '(?:(?::)([%a-zA-Z0-9\\-\\_\\s,]+))?',
                },
            },
        },

        beforeChangePath: function (path, subpathOrAction, action) {
            let me = this;

            let subpath = subpathOrAction;
            if (!action) {
                action = subpathOrAction;
                subpath = undefined;
            }

            if (!Ext.ClassManager.getByAlias('widget.' + path)) {
                console.warn('xtype "' + path + '" not found');
                action.stop();
                return;
            }

            let lastpanel = me.lookupReference('contentpanel').getLayout().getActiveItem();
            if (lastpanel && lastpanel.xtype === path) {
                // we have the right component already,
                // we just need to select the correct tab
                // default to the first
                subpath = subpath || 0;
                if (lastpanel.getActiveTab) {
                    // we assume lastpanel is a tabpanel
                    if (lastpanel.getActiveTab().getItemId() !== subpath) {
                        // set the active tab
                        lastpanel.setActiveTab(subpath);
                    }
                    // else we are already there
                }
                action.stop();
                return;
            }

            action.resume();
        },

        changePath: function (path, subpath) {
            let me = this;
            let contentpanel = me.lookupReference('contentpanel');
            let lastpanel = contentpanel.getLayout().getActiveItem();

            let obj = contentpanel.add({ xtype: path });
            let treelist = me.lookupReference('navtree');

            treelist.suspendEvents();
            treelist.select(path);
            treelist.resumeEvents();

            if (Ext.isFunction(obj.setActiveTab)) {
                obj.setActiveTab(subpath || 0);
                obj.addListener('tabchange', function (tabpanel, newc, oldc) {
                    let newpath = path;

                    // only add the subpath part for the
                    // non-default tabs
                    if (tabpanel.items.findIndex('id', newc.id) !== 0) {
                        newpath += ':' + newc.getItemId();
                    }

                    me.redirectTo(newpath);
                });
            }

            contentpanel.setActiveItem(obj);

            if (lastpanel) {
                contentpanel.remove(lastpanel, { destroy: true });
            }
        },

        logout: function () {
            PMG.app.logout();
        },

        navigate: function (treelist, item) {
            this.redirectTo(item.get('path'));
        },

        changeLanguage: function () {
            Ext.create('Proxmox.window.LanguageEditWindow', {
                cookieName: 'PMGLangCookie',
            }).show();
        },

        control: {
            '[reference=logoutButton]': {
                click: 'logout',
            },
            '[reference=languageButton]': {
                click: 'changeLanguage',
            },
        },

        init: function (view) {
            let me = this;

            // load username
            me.lookupReference('usernameinfo').setText(Proxmox.UserName);

            // show login on requestexception
            // fixme: what about other errors
            Ext.Ajax.on('requestexception', function (conn, response, options) {
                if (response.status === 401) {
                    // auth failure
                    me.logout();
                }
            });

            // get ticket periodically
            Ext.TaskManager.start({
                run: function () {
                    let ticket = Proxmox.Utils.authOK();
                    if (!ticket || !Proxmox.UserName) {
                        return;
                    }

                    Ext.Ajax.request({
                        params: {
                            username: Proxmox.UserName,
                            password: ticket,
                        },
                        url: '/api2/json/access/ticket',
                        method: 'POST',
                        failure: function () {
                            me.logout();
                        },
                        success: function (response, opts) {
                            let obj = Ext.decode(response.responseText);
                            PMG.Utils.updateLoginData(obj.data);
                        },
                    });
                },
                interval: 15 * 60 * 1000,
            });

            // select treeitem and load page from url fragment
            let token = Ext.util.History.getToken() || 'pmgDashboard';
            this.redirectTo(token, { force: true });
        },
    },

    plugins: 'viewport',

    layout: { type: 'border' },

    items: [
        {
            region: 'north',
            xtype: 'container',
            layout: {
                type: 'hbox',
                align: 'middle',
            },
            margin: '2 0 2 5',
            height: 38,
            items: [
                {
                    xtype: 'proxmoxlogo',
                },
                {
                    padding: '0 0 0 5',
                    xtype: 'versioninfo',
                },
                {
                    xtype: 'box', // keep markup simple, has no borders/background
                    flex: 1,
                },
                {
                    xtype: 'proxmoxHelpButton',
                    text: gettext('Documentation'),
                    hidden: false,
                    baseCls: 'x-btn',
                    iconCls: 'fa fa-info-circle x-btn-icon-el-default-toolbar-small ',
                    margin: '0 5 0 0',
                    listenToGlobalEvent: false,
                    onlineHelp: 'pmg_documentation_index',
                },
                {
                    xtype: 'button',
                    reference: 'usernameinfo',
                    style: {
                        // proxmox dark grey p light grey as border
                        backgroundColor: '#464d4d',
                        borderColor: '#ABBABA',
                    },
                    margin: '0 5 0 0',
                    iconCls: 'fa fa-user',
                    menu: [
                        {
                            iconCls: 'fa fa-gear',
                            text: gettext('My Settings'),
                            handler: () => Ext.create('PMG.window.Settings').show(),
                        },
                        {
                            iconCls: 'fa fa-paint-brush',
                            text: gettext('Color Theme'),
                            handler: () =>
                                Ext.create('Proxmox.window.ThemeEditWindow', {
                                    cookieName: 'PMGThemeCookie',
                                    autoShow: true,
                                }),
                        },
                        {
                            iconCls: 'fa fa-language',
                            text: gettext('Language'),
                            reference: 'languageButton',
                        },
                        '-',
                        {
                            reference: 'logoutButton',
                            iconCls: 'fa fa-sign-out',
                            text: gettext('Logout'),
                        },
                    ],
                },
            ],
        },
        {
            xtype: 'panel',
            scrollable: 'y',
            border: false,
            region: 'west',
            layout: {
                type: 'vbox',
                align: 'stretch',
            },
            items: [
                {
                    xtype: 'navigationtree',
                    minWidth: 180,
                    reference: 'navtree',
                    // we have to define it here until extjs 6.2 because of a bug where a
                    // viewcontroller does not detect the selectionchange event of a treelist
                    listeners: {
                        selectionchange: 'navigate',
                    },
                },
                {
                    xtype: 'box',
                    cls: 'x-treelist-pve-nav',
                    flex: 1,
                },
            ],
        },
        {
            xtype: 'panel',
            layout: { type: 'card' },
            region: 'center',
            border: false,
            reference: 'contentpanel',
        },
    ],
});
Ext.define('PMG.controller.QuarantineController', {
    extend: 'Ext.app.ViewController',
    alias: 'controller.quarantine',

    updatePreview: function (raw, rec) {
        let preview = this.lookupReference('preview');

        if (!rec || !rec.data || !rec.data.id) {
            preview.update('');
            preview.setDisabled(true);
            return;
        }

        let url = `/api2/htmlmail/quarantine/content?id=${rec.data.id}`;
        if (raw) {
            url += '&raw=1';
        }
        preview.setDisabled(false);
        this.lookupReference('raw').setDisabled(false);
        this.lookupReference('download').setDisabled(false);
        preview.update(
            "<iframe frameborder=0 width=100% height=100% sandbox='allow-same-origin' src='" +
                url +
                "'></iframe>",
        );
    },

    multiSelect: function (selection) {
        let me = this;
        me.lookupReference('raw').setDisabled(true);
        me.lookupReference('download').setDisabled(true);
        me.lookupReference('mailinfo').setVisible(false);
        me.lookup('attachmentlist')?.setVisible(false);

        let preview = me.lookupReference('preview');
        preview.setDisabled(false);
        preview.update(
            `<h3 style="padding-left:5px;">${gettext('Multiple E-Mails selected')} (${selection.length})</h3>`,
        );
    },

    toggleTheme: function (button) {
        let preview = this.lookup('preview');
        this.themed = !this.themed;

        if (this.themed) {
            preview.addCls('pmg-mail-preview-themed');
        } else {
            preview.removeCls('pmg-mail-preview-themed');
        }
    },

    hideThemeToggle: function (argument) {
        let me = this;
        let themeButton = me.lookup('themeCheck');
        themeButton.disable();
        themeButton.hide();
        me.lookup('themeCheckSep').hide();
        me.themed = true;
        me.toggleTheme();
    },

    showThemeToggle: function (argument) {
        let me = this;
        let themeButton = me.lookup('themeCheck');
        me.themed = false;
        me.toggleTheme();
        themeButton.setValue(true);
        themeButton.enable();
        themeButton.show();
        me.lookup('themeCheckSep').show();
    },

    toggleRaw: function (button) {
        let me = this;
        let list = me.lookupReference('list');
        let rec = list.selModel.getSelection()[0];
        me.lookupReference('mailinfo').setVisible(me.raw);
        me.raw = !me.raw;
        me.updatePreview(me.raw, rec);
    },

    btnHandler: function (button, e) {
        let me = this;
        let action = button.reference;
        let list = me.lookupReference('list');
        let selected = list.getSelection();
        me.doAction(action, selected);
    },

    doAction: function (action, selected) {
        if (!selected.length) {
            return;
        }

        let list = this.lookupReference('list');

        if (selected.length > 1) {
            let idlist = selected.map((item) => item.data.id);
            Ext.Msg.confirm(
                gettext('Confirm'),
                Ext.String.format(gettext("Action '{0}' for '{1}' items"), action, selected.length),
                async function (button) {
                    if (button !== 'yes') {
                        return;
                    }

                    list.mask(gettext('Processing...'), 'x-mask-loading');

                    const sliceSize = 2500,
                        maxInFlight = 2;
                    let batches = [],
                        batchCount = Math.ceil(selected.length / sliceSize);
                    for (let i = 0; i * sliceSize < selected.length; i++) {
                        let sliceStart = i * sliceSize;
                        let sliceEnd = Math.min(sliceStart + sliceSize, selected.length);
                        batches.push(
                            PMG.Async.doQAction(
                                action,
                                idlist.slice(sliceStart, sliceEnd),
                                i + 1,
                                batchCount,
                            ),
                        );
                        if (batches.length >= maxInFlight) {
                            await Promise.allSettled(batches); // eslint-disable-line no-await-in-loop
                            batches = [];
                        }
                    }
                    await Promise.allSettled(batches); // await possible remaining ones
                    list.unmask();
                    // below can be slow, we could remove directly from the in-memory store, but
                    // with lots of elements and some failures we could be quite out of sync?
                    list.getController().load();
                },
            );
            return;
        }

        PMG.Utils.doQuarantineAction(action, selected[0].data.id, function () {
            let listController = list.getController();
            listController.allowPositionSave = false;
            // success -> remove directly to avoid slow store reload for a single-element action
            list.getStore().remove(selected[0]);
            listController.restoreSavedSelection();
            listController.allowPositionSave = true;
        });
    },

    onSelectMail: function () {
        let me = this;
        let list = this.lookupReference('list');
        let selection = list.selModel.getSelection();
        if (selection.length > 1) {
            me.multiSelect(selection);
            return;
        }

        let rec = selection[0] || {};
        me.lookup('spaminfo')?.setID(rec);
        me.lookup('attachmentlist')?.setID(rec);
        me.lookup('attachmentlist')?.setVisible(!!rec.data);

        me.getViewModel().set('mailid', rec.data ? rec.data.id : '');
        me.updatePreview(me.raw || false, rec);
        me.lookupReference('mailinfo').setVisible(!!rec.data && !me.raw);
        me.lookupReference('mailinfo').update(rec.data);
    },

    openContextMenu: function (table, record, tr, index, event) {
        event.stopEvent();
        let me = this;
        let list = me.lookup('list');
        Ext.create('PMG.menu.QuarantineContextMenu', {
            callback: (action) => me.doAction(action, list.getSelection()),
        }).showAt(event.getXY());
    },

    keyPress: function (table, record, item, index, event) {
        let me = this;
        let list = me.lookup('list');
        let key = event.getKey();
        let action = '';
        switch (key) {
            case event.DELETE:
            case 127:
                action = 'delete';
                break;
            case Ext.event.Event.D:
            case Ext.event.Event.D + 32:
                action = 'deliver';
                break;
        }

        if (action !== '') {
            me.doAction(action, list.getSelection());
        }
    },

    beforeRender: function () {
        let me = this;
        let currentTheme = Ext.util.Cookies.get('PMGThemeCookie');

        if (currentTheme === '__default__' || currentTheme === null) {
            me.mediaQueryList = window.matchMedia('(prefers-color-scheme: dark)');

            me.themeListener = (e) => {
                if (e.matches) {
                    me.showThemeToggle();
                } else {
                    me.hideThemeToggle();
                }
            };

            me.themeListener(me.mediaQueryList);
            me.mediaQueryList.addEventListener('change', me.themeListener);
        } else if (currentTheme === 'crisp') {
            me.hideThemeToggle();
        } else {
            me.showThemeToggle();
        }
    },

    destroy: function () {
        let me = this;

        me.mediaQueryList?.removeEventListener('change', me.themeListener);

        me.callParent();
    },

    control: {
        'button[reference=raw]': {
            click: 'toggleRaw',
        },
        'proxmoxcheckbox[reference=themeCheck]': {
            change: 'toggleTheme',
        },
        pmgQuarantineList: {
            selectionChange: 'onSelectMail',
            itemkeypress: 'keyPress',
            rowcontextmenu: 'openContextMenu',
        },
    },
});
Ext.define('PMG.menu.QuarantineContextMenu', {
    extend: 'Ext.menu.Menu',
    xtype: 'pmgQuarantineMenu',

    showSeparator: false,

    controller: {
        xclass: 'Ext.app.ViewController',
        callCallback: function (btn) {
            let view = this.getView();
            if (Ext.isFunction(view.callback)) {
                view.callback(btn.action);
            }
        },
    },

    items: [
        {
            text: gettext('Deliver'),
            iconCls: 'fa fa-fw fa-paper-plane-o info-blue',
            action: 'deliver',
            handler: 'callCallback',
        },
        {
            text: gettext('Delete'),
            iconCls: 'fa fa-fw fa-trash-o critical',
            action: 'delete',
            handler: 'callCallback',
        },
    ],
});
Ext.define('PMG.QuarantineList', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgQuarantineList',

    emptyText: gettext('No E-Mail address selected'),
    viewConfig: {
        deferEmptyText: false,
    },

    config: {
        quarantineType: 'spam',
        notFoundText: gettext('No data in database'),
    },

    statics: {
        from: 0,
        to: 0,
    },

    allowPositionSave: false,

    controller: {
        xclass: 'Ext.app.ViewController',

        init: function (view) {
            let me = this;
            let emailCombobox = me.lookupReference('email');
            if (PMG.view === 'quarantineview') {
                view.autoLoadAll = false;
                me.setEmptyText();
            } else {
                emailCombobox.setVisible(true);
                emailCombobox.setDisabled(false);
                emailCombobox.getStore().on('load', me.injectAllOption, me);
            }

            if (view.quarantineType) {
                emailCombobox.getStore().getProxy().setExtraParams({
                    'quarantine-type': view.quarantineType,
                });
            }

            let from;
            if (PMG.QuarantineList.from !== 0) {
                from = new Date(PMG.QuarantineList.from * 1000);
            } else {
                from = new Date(Date.now() - 7 * 24 * 60 * 60 * 1000);
            }

            let to;
            if (PMG.QuarantineList.to !== 0) {
                to = new Date(PMG.QuarantineList.to * 1000);
            } else {
                to = new Date();
            }

            // we to this to trigger the change event of those fields
            me.lookupReference('from').setValue(from);
            me.lookupReference('to').setValue(to);

            Proxmox.Utils.monStoreErrors(view.getView(), view.getStore());
            me.load(function () {
                if (view.cselect) {
                    view.setSelection(view.getStore().getById(view.cselect));
                }
            });
        },
        // ExtJS cannot dynamically change the emptyText on grids, so implement ourself
        setEmptyText: function (emptyText) {
            let me = this;
            let view = me.getView();
            let tableview = view.getView();
            tableview.emptyText = `<div class="x-grid-empty">${emptyText || view.notFoundText}</div>`;
        },

        load: function (callback) {
            let me = this;
            me.allowPositionSave = false;
            let view = me.getView();
            let store = view.getStore();
            if (view.quarantineType === 'spam' && PMG.view !== 'quarantineview') {
                if (!me.lookupReference('email').getSelection()) {
                    return; // if the combobox has no selection we do not reload
                }
                me.setEmptyText();
            }
            // deselect all first, else ExtJS does some funky O(n^3) comparissions as it tries
            // to keep the selection, but we do not care for that on a new load anyway
            view.getSelectionModel().deselectAll();

            store.load(() => {
                me.restoreSavedSelection();
                if (Ext.isFunction(callback)) {
                    callback();
                }
                me.allowPositionSave = true;
            });
        },

        restoreSavedSelection: function () {
            let me = this;
            let view = me.getView();
            if (me.savedPosition !== undefined) {
                let store = view.getStore();
                if (store.getCount() - 1 < me.savedPosition) {
                    me.savedPosition = store.getCount() - 1;
                }
                view.setSelection(store.getAt(me.savedPosition));
            } else {
                view.setSelection();
            }
        },

        setFrom: function (from) {
            let view = this.getView();
            let params = view.getStore().getProxy().getExtraParams();
            params.starttime = from;
            PMG.QuarantineList.from = from;
            view.getStore().getProxy().setExtraParams(params);
        },

        setTo: function (to) {
            let end_of_to = to + 24 * 60 * 60; // we want the end of the day
            let view = this.getView();
            let params = view.getStore().getProxy().getExtraParams();
            params.endtime = end_of_to;
            PMG.QuarantineList.to = to; // we save the start of the day here
            view.getStore().getProxy().setExtraParams(params);
        },

        setUser: function (user) {
            let view = this.getView();
            let params = view.getStore().getProxy().getExtraParams();
            if (!user) {
                delete params.pmail;
            } else {
                params.pmail = user;
            }
            view.getStore().getProxy().setExtraParams(params);
            view.user = user;
        },

        changeTime: function (field, value) {
            let me = this;

            me.allowPositionSave = false;
            me.savedPosition = undefined;

            if (!value) {
                return;
            }

            let val = value.getTime() / 1000;
            let combobox = me.lookupReference('email');
            let params = combobox.getStore().getProxy().getExtraParams();

            let to = me.lookupReference('to');
            let from = me.lookupReference('from');

            if (field.name === 'from') {
                me.setFrom(val);
                params.starttime = val;
                to.setMinValue(value);
            } else if (field.name === 'to') {
                me.setTo(val);
                params.endtime = val + 24 * 60 * 60;
                from.setMaxValue(value);
            } else {
                return;
            }

            combobox.getStore().getProxy().setExtraParams(params);
            combobox.getStore().load();

            me.load();
        },

        resetEmail: function () {
            let me = this;
            me.setUser(undefined);
        },

        changeEmail: function (tb, value) {
            let me = this;
            me.savedPosition = undefined;
            me.allowPositionSave = false;
            if (value === 'all') {
                me.setUser(null);
            } else {
                me.setUser(value);
            }
            tb.triggers.clear.setVisible(value?.length > 0 && value !== 'all');
            me.load();
        },

        savePosition: function (grid, selected, eopts) {
            let me = this;
            if (!me.allowPositionSave) {
                return;
            }
            if (selected.length <= 0) {
                me.savedPosition = undefined;
                return;
            }

            let view = me.getView();
            let id = view.getStore().indexOf(selected[0]);

            me.savedPosition = id;
        },

        doFilter: function (searchValue, store, sm) {
            const selected = sm.getSelection();
            const selectedRecordId = selected.length === 1 ? selected[0].id : null;
            let clearSelectedMail = true;
            let toDeselect = [];
            store.filterBy(function (record) {
                let match = false;

                Ext.each(['subject', 'from'], (property) => {
                    if (record.data[property] === null) {
                        return;
                    }

                    let v = record.data[property].toString();
                    if (v !== undefined) {
                        v = v.toLowerCase();
                        if (v.includes(searchValue)) {
                            match = true;
                            if (record.id === selectedRecordId) {
                                clearSelectedMail = false;
                            }
                        }
                    }
                });
                if (!match && sm.isSelected(record)) {
                    toDeselect.push(record);
                }
                return match;
            });
            if (toDeselect.length > 0) {
                sm.deselect(toDeselect, true);
                sm.maybeFireSelectionChange(true);
            }
            return selectedRecordId !== null && clearSelectedMail;
        },

        updateFilter: async function (field) {
            let me = this;
            let view = me.getView();
            let store = view.getStore();
            let sm = view.getSelectionModel();

            let searchValue = field.getValue().toLowerCase();

            // supress store event if not empty, let filterBy below trigger it to avoid glitches
            store.clearFilter(searchValue.length > 0);
            field.triggers.clear.setVisible(searchValue.length > 0);

            if (searchValue.length === 0) {
                me.setEmptyText();
                return;
            }
            me.setEmptyText(gettext('No match found'));

            let clearSelection = me.doFilter(searchValue, store, sm);

            if (clearSelection) {
                view.setSelection();
            }
        },

        injectAllOption: function (store, records, successfull) {
            let me = this;
            let view = me.getView();
            if (successfull && records.length > 1) {
                store.insert(0, { mail: 'all' });
            }
            let emailCombobox = me.lookup('email');
            if (!emailCombobox.getSelection() && view.quarantineType !== 'spam') {
                emailCombobox.setSelection(store.getAt(0));
            }
        },

        control: {
            '#': {
                beforedestroy: 'resetEmail',
                selectionchange: 'savePosition',
            },
            'combobox[reference=email]': {
                change: 'changeEmail',
            },
            datefield: {
                change: {
                    fn: 'changeTime',
                },
            },
        },
    },

    features: [
        {
            ftype: 'grouping',
            groupHeaderTpl: '{columnName}: {name} ({children.length})',
        },
    ],

    tbar: {
        layout: {
            type: 'vbox',
            align: 'stretch',
        },
        defaults: {
            margin: 2,
        },
        items: [
            {
                xtype: 'datefield',
                name: 'from',
                fieldLabel: gettext('Since'),
                reference: 'from',
                format: 'Y-m-d',
            },
            {
                xtype: 'datefield',
                name: 'to',
                fieldLabel: gettext('Until'),
                reference: 'to',
                format: 'Y-m-d',
            },
            {
                xtype: 'combobox',
                hidden: true,
                disabled: true,
                displayField: 'mail',
                valueField: 'mail',
                listConfig: {
                    emptyText: `<div class="x-grid-empty">${gettext('No data in database')}</div>`,
                },
                store: {
                    proxy: {
                        type: 'proxmox',
                        url: '/api2/json/quarantine/spamusers',
                    },
                    fields: [
                        {
                            name: 'mail',
                            renderer: Ext.htmlEncode,
                        },
                    ],
                },
                queryMode: 'local',
                editable: true,
                typeAhead: true,
                forceSelection: true,
                autoSelect: true,
                anyMatch: true,
                selectOnFocus: true,
                reference: 'email',
                fieldLabel: gettext('E-Mail'),
                triggers: {
                    clear: {
                        cls: 'pmx-clear-trigger',
                        weight: -1,
                        hidden: true,
                        handler: function () {
                            this.triggers.clear.setVisible(false);
                            // 'all' is unfiltered here, so empty/originalValue makes no sense
                            this.setValue('all');
                        },
                    },
                },
            },
            {
                xtype: 'textfield',
                name: 'filter',
                fieldLabel: gettext('Search'),
                emptyText: gettext('Subject, Sender'),
                enableKeyEvents: true,
                triggers: {
                    clear: {
                        cls: 'pmx-clear-trigger',
                        weight: -1,
                        hidden: true,
                        handler: function () {
                            let me = this;
                            me.setValue('');
                            // setValue does not results in a keyup event, so trigger manually
                            me.up('grid').getController().updateFilter(me);
                        },
                    },
                },
                listeners: {
                    buffer: 500,
                    keyup: 'updateFilter',
                },
            },
        ],
    },
});
Ext.define('PMG.grid.SpamInfoGrid', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgSpamInfoGrid',

    store: {
        autoDestroy: true,
        fields: ['desc', 'name', { type: 'number', name: 'score' }],
        proxy: {
            type: 'proxmox',
            root: 'data.spaminfo',
        },
        sorters: 'score',
    },

    setID: function (rec) {
        let me = this;
        let id = rec?.data?.id;
        if (!id) {
            me.getStore().removeAll();
            return;
        }
        me.store.proxy.setUrl(`/api2/json/quarantine/content?id=${id}`);
        me.store.load();
    },

    emptyText: gettext('No Spam Info'),
    hidden: true,

    features: [
        {
            ftype: 'summary',
        },
    ],

    viewConfig: {
        enableTextSelection: true,
    },

    columns: [
        {
            text: gettext('Test Name'),
            dataIndex: 'name',
            flex: 1,
            summaryType: 'count',
            summaryRenderer: (_v) => gettext('Spamscore'),
            tdCls: 'txt-monospace',
        },
        {
            text: gettext('Score'),
            dataIndex: 'score',
            align: 'right',
            tdCls: 'txt-monospace',
            renderer: function (score, meta) {
                if (score === 0) {
                    return score;
                }

                let absScore = Math.abs(score),
                    fontWeight = '400';
                let background = score < 0 ? '--pmg-spam-mid-neg' : '--pmg-spam-mid-pos';

                if (absScore >= 3) {
                    fontWeight = '900';
                    background = score < 0 ? '--pmg-spam-high-neg' : '--pmg-spam-high-pos';
                } else if (absScore >= 1.5) {
                    fontWeight = '600';
                } else if (absScore <= 0.1) {
                    fontWeight = '200';
                    background = score < 0 ? '--pmg-spam-low-neg' : '--pmg-spam-low-pos';
                }

                meta.tdStyle = `font-weight: ${fontWeight};background-color: var(${background});`;
                return score;
            },
            summaryType: 'sum',
            summaryRenderer: (value) => Ext.util.Format.round(value, 5),
        },
        {
            text: gettext('Description'),
            dataIndex: 'desc',
            flex: 2,
        },
    ],
});
Ext.define('PMG.MailInfoBox', {
    extend: 'Ext.container.Container',
    xtype: 'pmgMailInfo',

    cls: 'x-toolbar-default',
    style: {
        'border-left': '0px',
        'border-right': '0px',
    },

    update: function (data) {
        let me = this;
        let escaped = {};
        for (const [key, value] of Object.entries(data || {})) {
            escaped[key] = Ext.util.Format.ellipsis(Ext.htmlEncode(value), 103);
        }
        me.items.each((item) => item.update(escaped));
    },

    layout: {
        type: 'vbox',
        align: 'stretch',
    },

    defaults: {
        xtype: 'tbtext',
        margin: '2 2 0 0 ',
    },

    items: [
        {
            tpl:
                `<b class="bold">${gettext('From')}:</b> {from}` +
                `<span style="float:right;white-space:normal;overflow-wrap:break-word;">` +
                `<b class="bold">${gettext('Receiver')}:</b> {receiver}</span>`,
        },
        { tpl: `<b class="bold">${gettext('Subject')}:</b> {subject}` },
    ],
});
Ext.define('pmg-spam-archive', {
    extend: 'Ext.data.Model',
    fields: [
        { type: 'number', name: 'spamavg' },
        { type: 'integer', name: 'count' },
        { type: 'date', dateFormat: 'timestamp', name: 'day' },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/quarantine/spam',
    },
    idProperty: 'day',
});

Ext.define('pmg-spam-list', {
    extend: 'Ext.data.Model',
    fields: [
        'id',
        'envelope_sender',
        'from',
        'sender',
        'receiver',
        'subject',
        { type: 'number', name: 'spamlevel' },
        { type: 'integer', name: 'bytes' },
        { type: 'date', dateFormat: 'timestamp', name: 'time' },
        {
            type: 'string',
            name: 'day',
            convert: function (v, rec) {
                return Ext.Date.format(rec.get('time'), 'Y-m-d');
            },
            depends: ['time'],
        },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/quarantine/spam',
    },
    idProperty: 'id',
});

Ext.define('PMG.SpamQuarantineController', {
    extend: 'PMG.controller.QuarantineController',
    xtype: 'pmgSpamQuarantineController',
    alias: 'controller.spamquarantine',

    updatePreview: function (raw, rec) {
        let me = this;
        me.lookupReference('spam').setDisabled(false);

        me.callParent(arguments);
    },

    multiSelect: function (selection) {
        let me = this;
        let spam = me.lookupReference('spam');
        spam.setDisabled(true);
        spam.setPressed(false);
        me.lookupReference('spaminfo').setVisible(false);
        me.callParent(arguments);
    },

    toggleSpamInfo: function (btn) {
        var grid = this.lookupReference('spaminfo');
        grid.setVisible(!grid.isVisible());
    },

    openContextMenu: function (table, record, tr, index, event) {
        event.stopEvent();
        let me = this;
        let list = me.lookup('list');
        Ext.create('PMG.menu.SpamContextMenu', {
            callback: (action) => me.doAction(action, list.getSelection()),
        }).showAt(event.getXY());
    },

    keyPress: function (table, record, item, index, event) {
        var me = this;
        var list = me.lookup('list');
        var key = event.getKey();
        var action = '';
        switch (key) {
            case event.DELETE:
            case 127:
                action = 'delete';
                break;
            case Ext.event.Event.D:
            case Ext.event.Event.D + 32:
                action = 'deliver';
                break;
            case Ext.event.Event.W:
            case Ext.event.Event.W + 32:
                action = 'welcomelist';
                break;
            case Ext.event.Event.B:
            case Ext.event.Event.B + 32:
                action = 'blocklist';
                break;
        }

        if (action !== '') {
            me.doAction(action, list.getSelection());
        }
    },

    init: function (view) {
        this.lookup('list').cselect = view.cselect;
    },

    control: {
        'button[reference=raw]': {
            click: 'toggleRaw',
        },
        'button[reference=spam]': {
            click: 'toggleSpamInfo',
        },
        pmgQuarantineList: {
            itemkeypress: 'keyPress',
            rowcontextmenu: 'openContextMenu',
        },
    },
});

Ext.define('PMG.SpamQuarantine', {
    extend: 'Ext.container.Container',
    xtype: 'pmgSpamQuarantine',

    border: false,
    layout: { type: 'border' },

    defaults: { border: false },

    // from mail link
    cselect: undefined,

    viewModel: {
        parent: null,
        data: {
            mailid: '',
        },
        formulas: {
            downloadMailURL: (get) =>
                '/api2/json/quarantine/download?mailid=' + encodeURIComponent(get('mailid')),
        },
    },
    controller: 'spamquarantine',

    items: [
        {
            title: gettext('Spam Quarantine'),
            xtype: 'pmgQuarantineList',
            selModel: 'checkboxmodel',
            reference: 'list',
            region: 'west',
            width: 500,
            split: true,
            collapsible: false,
            store: {
                model: 'pmg-spam-list',
                groupField: 'day',
                groupDir: 'DESC',
                sorters: [
                    {
                        property: 'time',
                        direction: 'DESC',
                    },
                ],
            },

            columns: [
                {
                    header: gettext('Sender/Subject'),
                    dataIndex: 'subject',
                    renderer: PMG.Utils.render_sender,
                    flex: 1,
                },
                {
                    header: gettext('Score'),
                    dataIndex: 'spamlevel',
                    align: 'right',
                    width: 70,
                },
                {
                    header: gettext('Size') + ' (KB)',
                    renderer: (v) => Ext.Number.toFixed(v / 1024, 0),
                    dataIndex: 'bytes',
                    align: 'right',
                    width: 90,
                },
                {
                    header: gettext('Date'),
                    dataIndex: 'day',
                    hidden: true,
                },
                {
                    xtype: 'datecolumn',
                    header: gettext('Time'),
                    dataIndex: 'time',
                    format: 'H:i:s',
                },
            ],
        },
        {
            title: gettext('Selected Mail'),
            border: false,
            region: 'center',
            layout: 'fit',
            split: true,
            reference: 'preview',
            disabled: true,
            dockedItems: [
                {
                    xtype: 'toolbar',
                    dock: 'top',
                    overflowHandler: 'scroller',
                    style: {
                        // docked items have set the bottom with to 0px with '! important'
                        // but we still want one here, so we can remove the borders of the grids
                        'border-bottom-width': '1px ! important',
                    },
                    items: [
                        {
                            xtype: 'button',
                            reference: 'raw',
                            text: gettext('Toggle Raw'),
                            enableToggle: true,
                            iconCls: 'fa fa-file-code-o',
                        },
                        {
                            xtype: 'button',
                            reference: 'spam',
                            text: gettext('Toggle Spam Info'),
                            enableToggle: true,
                            iconCls: 'fa fa-bullhorn',
                        },
                        {
                            xtype: 'tbseparator',
                            reference: 'themeCheckSep',
                        },
                        {
                            xtype: 'proxmoxcheckbox',
                            reference: 'themeCheck',
                            checked: true,
                            boxLabel: gettext('Dark-mode filter'),
                            iconCls: 'fa fa-paint-brush',
                        },
                        '->',
                        {
                            xtype: 'button',
                            reference: 'download',
                            text: gettext('Download'),
                            setDownload: function (id) {
                                this.el.dom.download = id + '.eml';
                            },
                            bind: {
                                href: '{downloadMailURL}',
                                download: '{mailid}',
                            },
                            iconCls: 'fa fa-download',
                        },
                        '-',
                        {
                            reference: 'welcomelist',
                            text: gettext('Welcomelist'),
                            iconCls: 'fa fa-check',
                            handler: 'btnHandler',
                        },
                        {
                            reference: 'blocklist',
                            text: gettext('Blocklist'),
                            iconCls: 'fa fa-times',
                            handler: 'btnHandler',
                        },
                        {
                            reference: 'deliver',
                            text: gettext('Deliver'),
                            iconCls: 'fa fa-paper-plane-o info-blue',
                            handler: 'btnHandler',
                        },
                        {
                            reference: 'delete',
                            text: gettext('Delete'),
                            iconCls: 'fa fa-trash-o critical',
                            handler: 'btnHandler',
                        },
                    ],
                },
                {
                    xtype: 'pmgSpamInfoGrid',
                    reference: 'spaminfo',
                    border: false,
                },
                {
                    xtype: 'pmgMailInfo',
                    hidden: true,
                    reference: 'mailinfo',
                    border: false,
                },
                {
                    xtype: 'pmgAttachmentGrid',
                    reference: 'attachmentlist',
                    showDownloads: false,
                    border: false,
                    dock: 'bottom',
                },
            ],
        },
    ],
});
Ext.define('pmg-address-list', {
    extend: 'Ext.data.Model',
    fields: ['address'],
    idProperty: 'address',
});

// base class - do not use directly
Ext.define('PMG.UserBlockWelcomeList', {
    extend: 'Ext.grid.GridPanel',

    border: false,
    listname: undefined, // 'blocklist' or 'welcomelist',

    selModel: 'checkboxmodel',

    emptyText: gettext('No data in database'),

    controller: {
        xclass: 'Ext.app.ViewController',

        onAddAddress: function () {
            let view = this.getView();
            let params = view.getStore().getProxy().getExtraParams() || {};

            let url = '/quarantine/' + view.listname;

            let items = [
                {
                    xtype: 'proxmoxtextfield',
                    name: 'address',
                    minLength: 3,
                    regex: /^[^,;\s]*$/, // no whitespace no , and no ;
                    fieldLabel: gettext('Address'),
                },
            ];

            Ext.Object.each(params, function (key, value) {
                items.push({
                    xtype: 'hidden',
                    name: key,
                    value: value,
                });
            });

            let config = {
                method: 'POST',
                url: url,
                onlineHelp: 'pmg_userblockwelcomelist',
                isCreate: true,
                isAdd: true,
                items: items,
            };

            if (view.listname === 'blocklist') {
                config.subject = gettext('Blocklist');
            } else if (view.listname === 'welcomelist') {
                config.subject = gettext('Welcomelist');
            } else {
                throw 'unknown list - internal error';
            }

            let win = Ext.createWidget('proxmoxWindowEdit', config);
            win.on('destroy', function () {
                view.store.load();
            });
            win.show();
        },

        onRemoveAddress: function () {
            let view = this.getView();
            let records = view.selModel.getSelection();
            if (records.length < 1) {
                return;
            }

            let url = '/quarantine/' + view.listname + '/';

            let params = {
                address: records.map((rec) => rec.getId()).join(','),
            };
            Ext.applyIf(params, view.getStore().getProxy().getExtraParams());

            Proxmox.Utils.API2Request({
                url: url + '?' + Ext.Object.toQueryString(params),
                method: 'DELETE',
                waitMsgTarget: view,
                callback: function (options, success, response) {
                    view.store.load();
                },
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        changeEmail: function (combobox, value) {
            let view = this.getView();
            if (value && combobox.isValid()) {
                view.getStore().getProxy().setExtraParams({
                    pmail: value,
                });
                view.getStore().load();
            }
        },

        init: function (view) {
            let emailcb = this.lookupReference('email');
            if (PMG.view === 'quarantineview') {
                emailcb.setVisible(false);
                view.getStore().load();
            } else {
                emailcb
                    .getStore()
                    .getProxy()
                    .setExtraParams({
                        list: view.listname === 'blocklist' ? 'BL' : 'WL',
                    });
            }
            Proxmox.Utils.monStoreErrors(view.getView(), view.getStore(), true);
        },

        control: {
            combobox: {
                change: {
                    fn: 'changeEmail',
                    buffer: 500,
                },
            },
        },
    },

    tbar: [
        {
            xtype: 'combobox',
            displayField: 'mail',
            vtype: 'PMGMail',
            allowBlank: false,
            valueField: 'mail',
            store: {
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/quarantine/quarusers',
                },
                fields: [
                    {
                        name: 'mail',
                        renderer: Ext.htmlEncode,
                    },
                ],
            },
            queryParam: false,
            queryCaching: false,
            editable: true,
            reference: 'email',
            name: 'email',
            listConfig: {
                emptyText: '<div class="x-grid-empty">' + gettext('No data in database') + '</div>',
            },
            fieldLabel: gettext('E-Mail'),
        },
        {
            text: gettext('Add'),
            handler: 'onAddAddress',
        },
        {
            xtype: 'proxmoxButton',
            text: gettext('Remove'),
            disabled: true,
            handler: 'onRemoveAddress',
            confirmMsg: function () {
                let view = this.up('gridpanel');

                let selection = view.selModel.getSelection();
                let text, param;
                if (selection.length > 1) {
                    text = gettext('Are you sure you want to remove {0} entries');
                    param = selection.length.toString();
                } else if (selection.length > 0) {
                    let rec = selection[0];
                    let name = rec.getId();
                    text = gettext('Are you sure you want to remove entry {0}');
                    param = "'" + Ext.String.htmlEncode(name) + "'";
                }

                if (text && param) {
                    return Ext.String.format(text, param);
                }
                return false;
            },
        },
    ],

    columns: [
        {
            header: gettext('Address'),
            dataIndex: 'address',
            renderer: Ext.String.htmlEncode,
            flex: 1,
        },
    ],
});

Ext.define('PMG.UserBlocklist', {
    extend: 'PMG.UserBlockWelcomeList',
    xtype: 'pmgUserBlocklist',

    title: gettext('Blocklist'),

    listname: 'blocklist',

    store: {
        model: 'pmg-address-list',
        autoDestroy: true,
        proxy: {
            type: 'proxmox',
            url: '/api2/json/quarantine/blocklist',
        },
        sorters: {
            property: 'address',
        },
    },

    dockedItems: [
        {
            dock: 'top',
            bodyStyle: {
                padding: '10px',
                'border-left': '0px',
                'border-right': '0px',
            },
            html:
                gettext(
                    'With this feature, you can manually mark E-mails from certain domains or addresses as spam.',
                ) +
                `<br><br>
		<b>*.com</b> (all mails from <b>.com</b> domains)<br>
		<b>*@example.com</b> (all mails from domain <b>example.com</b>)<br>
		<b>john@example.com</b> (all mails from <b>john@example.com</b>)`,
        },
    ],
});

Ext.define('PMG.UserWelcomelist', {
    extend: 'PMG.UserBlockWelcomeList',
    xtype: 'pmgUserWelcomelist',

    title: gettext('Welcomelist'),

    listname: 'welcomelist',

    store: {
        model: 'pmg-address-list',
        autoDestroy: true,
        proxy: {
            type: 'proxmox',
            url: '/api2/json/quarantine/welcomelist',
        },
        sorters: {
            property: 'address',
        },
    },

    dockedItems: [
        {
            dock: 'top',
            bodyStyle: {
                padding: '10px',
                'border-left': '0px',
                'border-right': '0px',
            },
            html:
                gettext(
                    'With this feature, you can manually bypass spam checking for certain domains or E-mail addresses.',
                ) +
                `<br><br>
		<b>*.com</b> (all mails from <b>.com</b> domains)<br>
		<b>*@example.com</b> (all mails from domain <b>example.com</b>)<br>
		<b>john@example.com</b> (all mails from <b>john@example.com</b>)`,
        },
    ],
});
Ext.define('PMG.QuarantineNavigationTree', {
    extend: 'Ext.list.Tree',
    xtype: 'quarantinenavigationtree',

    select: function (path) {
        let me = this;
        let item = me.getStore().findRecord('path', path, 0, false, true, true);
        me.setSelection(item);
    },

    store: {
        root: {
            expanded: true,
            children: [
                {
                    text: gettext('Spam Quarantine'),
                    iconCls: 'fa fa-cubes',
                    path: 'pmgSpamQuarantine',
                    expanded: true,
                    children: [
                        {
                            text: gettext('Welcomelist'),
                            iconCls: 'fa fa-file-o',
                            path: 'pmgUserWelcomelist',
                            leaf: true,
                        },
                        {
                            text: gettext('Blocklist'),
                            iconCls: 'fa fa-file',
                            path: 'pmgUserBlocklist',
                            leaf: true,
                        },
                    ],
                },
                {
                    text: gettext('Help'),
                    iconCls: 'fa fa-support',
                    path: 'pmgQuarantineAbout',
                    leaf: true,
                    expanded: true,
                },
            ],
        },
    },

    animation: false,
    expanderOnly: true,
    expanderFirst: false,
    ui: 'pve-nav',
});

Ext.define('PMG.QuarantineView', {
    extend: 'Ext.container.Container',
    xtype: 'quarantineview',

    title: 'Proxmox Mail Gateway Quarantine',

    controller: {
        xclass: 'Ext.app.ViewController',
        routes: {
            ':path:subpath': {
                action: 'changePath',
                before: 'beforeChangePath',
                conditions: {
                    ':path': '(?:([%a-zA-Z0-9\\-\\_\\s,]+))',
                    ':subpath': '(?:(?::)([%a-zA-Z0-9\\-\\_\\s,]+))?',
                },
            },
        },

        beforeChangePath: function (path, subpathOrAction, action) {
            let me = this;

            let subpath = subpathOrAction;
            if (!action) {
                action = subpathOrAction;
                subpath = undefined;
            }

            if (!Ext.ClassManager.getByAlias('widget.' + path)) {
                console.warn('xtype "' + path + '" not found');
                action.stop();
                return;
            }

            let lastpanel = me.lookupReference('contentpanel').getLayout().getActiveItem();
            if (lastpanel && lastpanel.xtype === path) {
                // we have the right component already,
                // we just need to select the correct tab
                // default to the first
                subpath = subpath || 0;
                if (lastpanel.getActiveTab) {
                    // we assume lastpanel is a tabpanel
                    if (lastpanel.getActiveTab().getItemId() !== subpath) {
                        // set the active tab
                        lastpanel.setActiveTab(subpath);
                    }
                    // else we are already there
                }
                action.stop();
                return;
            }

            action.resume();
        },

        changePath: function (path, subpath) {
            let me = this;
            let contentpanel = me.lookupReference('contentpanel');
            let lastpanel = contentpanel.getLayout().getActiveItem();

            let obj = contentpanel.add({ xtype: path, cselect: subpath });
            let treelist = me.lookupReference('navtree');

            treelist.suspendEvents();
            treelist.select(path);
            treelist.resumeEvents();

            if (Ext.isFunction(obj.setActiveTab)) {
                obj.setActiveTab(subpath || 0);
                obj.addListener('tabchange', function (tabpanel, newc, oldc) {
                    let newpath = path;

                    // only add the subpath part for the
                    // non-default tabs
                    if (tabpanel.items.findIndex('id', newc.id) !== 0) {
                        newpath += ':' + newc.getItemId();
                    }

                    me.redirectTo(newpath);
                });
            }

            contentpanel.setActiveItem(obj);

            if (lastpanel) {
                contentpanel.remove(lastpanel, { destroy: true });
            }
        },

        logout: function () {
            PMG.app.logout();
        },

        changeLanguage: function () {
            Ext.create('Proxmox.window.LanguageEditWindow', {
                cookieName: 'PMGLangCookie',
            }).show();
        },

        changeTheme: () =>
            Ext.create('Proxmox.window.ThemeEditWindow', {
                cookieName: 'PMGThemeCookie',
                autoShow: true,
            }),
        navigate: function (treelist, item) {
            this.redirectTo(item.get('path'));
        },

        execQuarantineAction: function (qa) {
            PMG.Utils.doQuarantineAction(qa.action, qa.cselect);
        },

        switchToMobile: function () {
            let params = new URLSearchParams(document.location.search);
            params.set('mobile', '1');
            document.location.search = params.toString();
        },

        control: {
            '[reference=logoutButton]': {
                click: 'logout',
            },
            '[reference=languageButton]': {
                click: 'changeLanguage',
            },
            '[reference=themeButton]': {
                click: 'changeTheme',
            },
            '[reference=mobileButton]': {
                click: 'switchToMobile',
            },
        },

        listen: {
            global: {
                unmatchedroute: 'onUnmatchedRoute',
            },
        },

        onUnmatchedRoute: function (token) {
            console.warn(
                `got unmatched app route token ${token}, redirecting to quarantine overview.`,
            );
            this.redirectTo('pmgSpamQuarantine');
        },

        init: function (view) {
            let me = this;

            // load username
            let username = Proxmox.UserName.replace(/@quarantine$/, '');
            me.lookupReference('usernameinfo').setText(username);

            // show login on requestexception
            // fixme: what about other errors
            Ext.Ajax.on('requestexception', function (conn, response, options) {
                if (response.status === 401) {
                    // auth failure
                    me.logout();
                }
            });

            let qa = PMG.Utils.extractQuarantineAction();
            let token;
            if (qa) {
                token = 'pmgSpamQuarantine';
                if (qa.action === 'blocklist') {
                    token = 'pmgUserBlocklist';
                }
                if (qa.action === 'welcomelist') {
                    token = 'pmgUserWelcomelist';
                }
                if (qa.cselect) {
                    token += ':' + qa.cselect;
                }
                this.redirectTo(token, true);
                if (qa.action) {
                    me.execQuarantineAction(qa);
                }
            } else {
                // select treeitem and load page from url fragment

                token = Ext.util.History.getToken() || 'pmgSpamQuarantine';
                this.redirectTo(token, true);
            }

            if (Ext.getBody().getViewSize().width <= 1100) {
                Ext.toast({
                    title: gettext('Narrow viewport detected'),
                    items: {
                        xtype: 'button',
                        iconCls: ' x-btn-icon-el-default-toolbar-small fa fa-mobile',
                        cls: 'x-btn-default-toolbar-small proxmox-inline-button',
                        text: gettext('Switch to mobile view'),
                        href: '?mobile=1',
                        hrefTarget: '_self',
                    },
                    minWidth: 250,
                    closable: true,
                    autoCloseDelay: 7000,
                    iconCls: 'fa fa-question-circle',
                    shadow: true,
                    align: 'b',
                });
            }
        },
    },

    plugins: 'viewport',

    layout: {
        type: 'border',
    },

    items: [
        {
            region: 'north',
            xtype: 'container',
            layout: {
                type: 'hbox',
                align: 'middle',
            },
            margin: '2 0 2 5',
            height: 38,
            items: [
                {
                    xtype: 'proxmoxlogo',
                },
                {
                    padding: '0 0 0 5',
                    xtype: 'versioninfo',
                },
                {
                    flex: 1,
                },
                {
                    xtype: 'button',
                    reference: 'usernameinfo',
                    style: {
                        // proxmox dark grey p light grey as border
                        backgroundColor: '#464d4d',
                        borderColor: '#ABBABA',
                    },
                    margin: '0 5 0 0',
                    iconCls: 'fa fa-user',
                    menu: [
                        {
                            reference: 'themeButton',
                            iconCls: 'fa fa-paint-brush',
                            text: gettext('Color Theme'),
                        },
                        {
                            iconCls: 'fa fa-language',
                            text: gettext('Language'),
                            reference: 'languageButton',
                        },
                        {
                            iconCls: 'fa fa-mobile',
                            text: gettext('Switch to Mobile View'),
                            reference: 'mobileButton',
                        },
                        '-',
                        {
                            reference: 'logoutButton',
                            iconCls: 'fa fa-sign-out',
                            text: gettext('Logout'),
                        },
                    ],
                },
            ],
        },
        {
            xtype: 'panel',
            scrollable: 'y',
            border: false,
            region: 'west',
            layout: {
                type: 'vbox',
                align: 'stretch',
            },
            items: [
                {
                    xtype: 'quarantinenavigationtree',
                    reference: 'navtree',
                    minWidth: 180,
                    // we have to define it here until extjs 6.2 because of a bug where a
                    // viewcontroller does not detect the selectionchange event of a treelist
                    listeners: {
                        selectionchange: 'navigate',
                    },
                },
                {
                    xtype: 'box',
                    cls: 'x-treelist-pve-nav',
                    flex: 1,
                },
            ],
        },
        {
            xtype: 'panel',
            layout: {
                type: 'card',
            },
            region: 'center',
            border: false,
            reference: 'contentpanel',
        },
    ],
});
// Needs to be its own xtype for `path` to work in `NavigationTree`
Ext.define('PMG.QuarantineAboutPage', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgQuarantineAbout',

    bodyPadding: 10,
    bodyStyle: {
        fontSize: '14px',
    },

    title: gettext('Proxmox Mail Gateway Quarantine Help'),

    html: Proxmox.Markdown.parse(
        `# About
This is the end-user email quarantine interface provided by your email provider.

Proxmox Mail Gateway is software that scans email for threats such as spam or viruses.

Typically, emails that contain viruses or are identified as specific spam are blocked by your
provider.
Emails that are not classified as specific spam can be quarantined for the recipient to decide
whether to receive or delete them. In most setups, you will receive a spam report email notifying
you when mail is quarantined for your address.

You also have the option to block or welcomelist certain addresses:

* Allow, in the Welcomelist menu, results in mails from these addresses to be delivered directly
  instead of being quarantined.
* Blocking, in the Blocklist menu, results in mails from these addresses to be deleted directly
  instead of being quarantined.

**Note:** The sending of *Spam Report* emails and this web application is controlled by your email
provider.

Proxmox Server Solutions GmbH develops the software and does not operate email services for users.

## Keyboard Shortcuts

Once you have selected an item in Quarantine, you can either use the dedicated buttons to deliver or
delete the selected email, or use the following keyboard shortcuts instead:

* <kbd>D</kbd>: Deliver the mail.
* <kbd>Delete</kbd>: Delete the mail.
* <kbd>B</kbd>: Add the sender to the Blocklist.
* <kbd>W</kbd>: Add the sender to the Welcomelist.
`,
    ),
});
Ext.define('PMG.Dashboard', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgDashboard',

    controller: {
        xclass: 'Ext.app.ViewController',

        openDashboardOptions: function () {
            var me = this;
            var viewModel = me.getViewModel();
            Ext.create('Ext.window.Window', {
                modal: true,
                width: 300,
                title: gettext('Dashboard Options'),
                layout: {
                    type: 'auto',
                },
                items: [
                    {
                        xtype: 'form',
                        bodyPadding: '10 10 10 10',
                        defaultButton: 'savebutton',
                        items: [
                            {
                                xtype: 'proxmoxintegerfield',
                                itemId: 'hours',
                                labelWidth: 100,
                                anchor: '100%',
                                allowBlank: false,
                                minValue: 1,
                                maxValue: 24,
                                value: viewModel.get('hours'),
                                fieldLabel: gettext('Hours to show'),
                            },
                        ],
                        buttons: [
                            {
                                text: gettext('Save'),
                                reference: 'loginButton',
                                formBind: true,
                                handler: function () {
                                    var win = this.up('window');
                                    var hours = win.down('#hours').getValue();
                                    me.setHours(hours, true);
                                    win.close();
                                },
                            },
                        ],
                    },
                ],
            }).show();
        },

        setHours: function (hours, setState) {
            var me = this;
            var viewModel = me.getViewModel();
            viewModel.set('hours', hours);
            viewModel.notify();

            Ext.Array.forEach(['recentmails', 'receivers'], function (item) {
                viewModel.getStore(item).reload();
            });

            if (setState) {
                let sp = Ext.state.Manager.getProvider();
                sp.set('dashboard-hours', hours);
            }
        },

        updateMailStats: function (store, records, success) {
            if (!success) {
                return;
            }
            var me = this;
            var viewModel = me.getViewModel();

            var count = 0;
            var bytes_in = 0;
            var bytes_out = 0;
            var ptime = 0;
            var avg_ptime = 'N/A';

            records.forEach(function (item) {
                bytes_in += item.data.bytes_in;
                bytes_out += item.data.bytes_out;
                // unnormalize
                count += (item.data.count * item.data.timespan) / 60;
                ptime += item.data.ptimesum;
            });

            if (count) {
                avg_ptime = (ptime / count).toFixed(2) + ' s';
            }

            viewModel.set('bytes_in', Proxmox.Utils.format_size(bytes_in));
            viewModel.set('bytes_out', Proxmox.Utils.format_size(bytes_out));
            viewModel.set('avg_ptime', avg_ptime);
        },

        updateClusterStats: function (store, records, success) {
            if (!success) {
                return;
            }
            var me = this;
            var viewmodel = me.getViewModel();

            var subStatus = 2; // 2 = all good, 1 = different leves, 0 = none
            var subLevel = '';

            var cpu = 0;
            var mem = 0;
            var hd = 0;
            var count = records.length;
            var errors = [];

            records.forEach(function (item) {
                // subscription level check
                if (subStatus && item.data.level) {
                    if (subLevel !== '' && subLevel !== item.data.level) {
                        subStatus = 1;
                    } else if (subLevel === '') {
                        subLevel = item.data.level;
                    }
                } else {
                    subStatus = 0;
                }

                if (item.data.name === Proxmox.NodeName) {
                    let repoStatus = me.lookup('nodeInfo').down('#repositoryStatus');
                    repoStatus.setSubscriptionStatus(!!item.data.level);
                }

                // resources count
                cpu += item.data.cpu || 0;

                var memory = item.data.memory || { used: 0, total: 1 };
                mem += memory.used / memory.total;

                var rootfs = item.data.rootfs || { used: 0, total: 1 };
                hd += rootfs.used / rootfs.total;

                if (item.data.conn_error && count > 1) {
                    count--;
                    errors.push({
                        name: item.data.name,
                        msg: item.data.conn_error,
                    });
                }
            });

            var subscriptionPanel = me.lookup('subscription');
            subscriptionPanel.setSubStatus(subStatus);

            // the node info already displays this information in case there is no cluster
            me.lookup('clusterResources').setHidden(records.length === 1);

            cpu = cpu / count;
            mem = mem / count;
            hd = hd / count;

            var cpuPanel = me.lookup('cpu');
            cpuPanel.updateValue(cpu);

            var memPanel = me.lookup('mem');
            memPanel.updateValue(mem);

            var hdPanel = me.lookup('hd');
            hdPanel.updateValue(hd);

            if (errors.length && !viewmodel.get('error_shown')) {
                let text = '';
                errors.forEach(function (error) {
                    text += error.name + ':<br>' + error.msg + '<br>';
                });
                Ext.Msg.alert(gettext('Error'), text);
                viewmodel.set('error_shown', true);
            }
        },

        updateRepositoryStatus: function (store, records, success) {
            if (!success) {
                return;
            }

            let me = this;
            let repoStatus = me.lookup('nodeInfo').down('#repositoryStatus');
            repoStatus.setRepositoryInfo(records[0].data['standard-repos']);
        },

        init: function (view) {
            var me = this;
            var sp = Ext.state.Manager.getProvider();
            var hours = sp.get('dashboard-hours') || 12;
            me.setHours(hours, false);

            view.mon(sp, 'statechange', function (provider, key, value) {
                if (key !== 'summarycolumns') {
                    return;
                }
                Proxmox.Utils.updateColumnWidth(view);
            });
        },
    },

    listeners: {
        resize: (panel) => Proxmox.Utils.updateColumnWidth(panel),
    },

    viewModel: {
        data: {
            timespan: 300, // in seconds
            hours: 12, // in hours
            error_shown: false,
            bytes_in: 0,
            bytes_out: 0,
            avg_ptime: 0.0,
        },

        stores: {
            cluster: {
                storeid: 'dash-cluster',
                type: 'update',
                interval: 5000,
                autoStart: true,
                autoDestroy: true,
                proxy: {
                    extraParams: { list_single_node: 1 },
                    type: 'proxmox',
                    url: '/api2/json/config/cluster/status',
                },
                listeners: {
                    load: 'updateClusterStats',
                },
            },
            recentmails: {
                storeid: 'dash-recent',
                interval: 5000,
                type: 'update',
                autoStart: true,
                autoDestroy: true,
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/statistics/recent',
                    extraParams: {
                        hours: '{hours}',
                        timespan: '{timespan}',
                    },
                },
                fields: [
                    {
                        type: 'number',
                        name: 'count',
                        convert: PMG.Utils.convert_field_to_per_min,
                    },
                    {
                        type: 'number',
                        name: 'count_in',
                        convert: PMG.Utils.convert_field_to_per_min,
                    },
                    {
                        type: 'number',
                        name: 'count_out',
                        convert: PMG.Utils.convert_field_to_per_min,
                    },
                    {
                        type: 'number',
                        name: 'spam',
                        convert: PMG.Utils.convert_field_to_per_min,
                    },
                    {
                        type: 'number',
                        name: 'spam_in',
                        convert: PMG.Utils.convert_field_to_per_min,
                    },
                    {
                        type: 'number',
                        name: 'spam_out',
                        convert: PMG.Utils.convert_field_to_per_min,
                    },
                    {
                        type: 'number',
                        name: 'virus',
                        convert: PMG.Utils.convert_field_to_per_min,
                    },
                    {
                        type: 'number',
                        name: 'virus_in',
                        convert: PMG.Utils.convert_field_to_per_min,
                    },
                    { type: 'integer', name: 'virus_out' },
                    { type: 'integer', name: 'bytes_in' },
                    { type: 'integer', name: 'bytes_out' },
                    { type: 'number', name: 'ptimesum' },
                    { type: 'date', dateFormat: 'timestamp', name: 'time' },
                ],
                listeners: {
                    load: 'updateMailStats',
                },
            },
            receivers: {
                storeid: 'dash-receivers',
                interval: 10000,
                type: 'update',
                autoStart: true,
                autoDestroy: true,
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/statistics/recentreceivers',
                    extraParams: {
                        hours: '{hours}',
                        limit: 10, // make this also configurable?
                    },
                },
                fields: [
                    { type: 'integer', name: 'count' },
                    { type: 'string', name: 'receiver' },
                ],
            },
            repositories: {
                storeid: 'dash-repositories',
                type: 'update',
                interval: 15000,
                autoStart: true,
                autoLoad: true,
                autoDestroy: true,
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/nodes/localhost/apt/repositories',
                },
                listeners: {
                    load: 'updateRepositoryStatus',
                },
            },
        },
    },

    bind: {
        title:
            gettext('Dashboard') + ' (' + Ext.String.format(gettext('{0} hours'), '{hours}') + ')',
    },

    layout: {
        type: 'column',
    },
    border: false,

    bodyPadding: '20 0 0 20',

    defaults: {
        columnWidth: 1,
        xtype: 'panel',
        margin: '0 20 20 0',
    },

    tools: [
        {
            type: 'gear',
            handler: 'openDashboardOptions',
        },
    ],

    scrollable: true,

    items: [
        {
            height: 300,
            flex: 1,
            iconCls: 'fa fa-tachometer',
            title: gettext('E-Mail Volume'),
            layout: {
                type: 'vbox',
                align: 'stretch',
            },
            defaults: {
                xtype: 'pmgMiniGraph',
                bind: {
                    store: '{recentmails}',
                },
            },
            items: [
                {
                    fields: ['count'],
                    fieldTitles: [gettext('Mails / min')],
                    seriesConfig: {
                        colors: ['#00617F'],
                        style: {
                            opacity: 0.6,
                            lineWidth: 1,
                        },
                        highlightCfg: {
                            opacity: 1,
                            scaling: 1,
                        },
                    },
                },
                {
                    fields: ['spam'],
                    fieldTitles: [gettext('Spam / min')],
                    seriesConfig: {
                        colors: ['#E67300'],
                        style: {
                            opacity: 0.6,
                            lineWidth: 1,
                        },
                        highlightCfg: {
                            opacity: 1,
                            scaling: 1,
                        },
                    },
                },
            ],
        },
        {
            xtype: 'container',
            height: 300,
            layout: {
                type: 'vbox',
                align: 'stretch',
            },
            items: [
                {
                    xtype: 'pmgMailProcessing',
                    title: gettext('E-Mail Processing'),
                    iconCls: 'fa fa-hourglass-half',
                    height: 180,
                    bind: {
                        data: {
                            bytes_in: '{bytes_in}',
                            bytes_out: '{bytes_out}',
                            avg_ptime: '{avg_ptime}',
                        },
                    },
                },
                {
                    iconCls: 'fa fa-ticket',
                    title: gettext('Subscription'),
                    reference: 'subscription',
                    xtype: 'pmgSubscriptionInfo',
                    margin: '10 0 0 0',
                    height: 110,
                },
            ],
        },
        {
            xtype: 'pmgNodeInfoPanel',
            reference: 'nodeInfo',
            height: 300,
            bodyPadding: '15 5 15 5',
            iconCls: 'fa fa-tasks',
        },
        {
            height: 300,
            iconCls: 'fa fa-list',
            title: gettext('Top Receivers'),

            bodyPadding: '10 10 10 10',
            layout: {
                type: 'vbox',
                pack: 'center',
                align: 'stretch',
            },
            items: [
                {
                    xtype: 'grid',
                    bind: {
                        store: '{receivers}',
                    },
                    emptyText: gettext('No data in database'),
                    // remove all borders/lines/headers
                    border: false,
                    bodyBorder: false,
                    hideHeaders: true,
                    header: false,
                    columnLines: false,
                    rowLines: false,
                    viewConfig: {
                        stripeRows: false,
                    },
                    columns: [
                        {
                            dataIndex: 'receiver',
                            flex: 1,
                            text: gettext('Receiver'),
                        },
                        {
                            dataIndex: 'count',
                            align: 'right',
                            text: gettext('Count'),
                        },
                    ],
                },
            ],
        },
        {
            height: 250,
            iconCls: 'fa fa-tasks',
            title: gettext('Cluster Resources (average)'),
            reference: 'clusterResources',
            hidden: true,
            bodyPadding: '0 20 0 20',
            layout: {
                type: 'hbox',
                align: 'center',
            },
            defaults: {
                xtype: 'proxmoxGauge',
                spriteFontSize: '20px',
                flex: 1,
            },
            items: [
                {
                    title: gettext('CPU'),
                    reference: 'cpu',
                },
                {
                    title: gettext('Memory'),
                    reference: 'mem',
                },
                {
                    title: gettext('Storage'),
                    reference: 'hd',
                },
            ],
        },
    ],
});
Ext.define('PMG.dashboard.MailProcessing', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgMailProcessing',

    setData: function (data) {
        var me = this;
        me.down('#ptime').update(data);
        me.down('#traffic').update(data);
    },

    layout: {
        type: 'hbox',
        align: 'center',
        pack: 'center',
    },

    defaults: {
        xtype: 'box',
        flex: 1,
        style: {
            'text-align': 'center',
        },
    },

    items: [
        {
            itemId: 'traffic',
            data: {
                bytes_in: 0,
                bytes_out: 0,
            },
            tpl: [
                '<h3><i class="fa fa-exchange green"></i> ' + gettext('Traffic') + '</h3>',
                '<table class="dash"><tr>',
                '<td class="right half"><h2>{bytes_in}</h2></td>',
                '<td class="left">' + PMG.Utils.format_rule_direction(0) + '</td>',
                '</tr><tr>',
                '<td class="right half"><h2>{bytes_out}</h2></td>',
                '<td class="left">' + PMG.Utils.format_rule_direction(1) + '</td>',
                '</tr></table>',
            ],
        },
        {
            itemId: 'ptime',
            data: {
                avg_ptime: 0,
            },
            tpl: [
                '<h3><i class="fa fa-clock-o"></i> ' +
                    gettext('Avg. Mail Processing Time') +
                    '</h3>',
                '<p><h2>{avg_ptime}</h2></p>',
            ],
        },
    ],
});
Ext.define('PMG.NodeInfoPanel', {
    extend: 'Proxmox.panel.StatusView',
    alias: 'widget.pmgNodeInfoPanel',

    layout: {
        type: 'table',
        columns: 2,
        tableAttrs: {
            style: {
                width: '100%',
            },
        },
    },

    defaults: {
        xtype: 'pmxInfoWidget',
        padding: '0 10 5 10',
    },

    items: [
        {
            itemId: 'nodecpu',
            iconCls: 'fa fa-fw pmx-itype-icon-processor pmx-icon',
            title: gettext('CPU usage'),
            valueField: 'cpu',
            maxField: 'cpuinfo',
            renderer: Proxmox.Utils.render_node_cpu_usage,
        },
        {
            itemId: 'wait',
            iconCls: 'pmx-icon-size fa fa-fw fa-clock-o',
            title: gettext('IO delay'),
            valueField: 'wait',
        },
        {
            xtype: 'box',
            colspan: 2,
            padding: '0 0 20 0',
        },
        {
            iconCls: 'fa fa-fw pmx-itype-icon-memory pmx-icon',
            itemId: 'memory',
            title: gettext('RAM usage'),
            valueField: 'memory',
            maxField: 'memory',
            renderer: Proxmox.Utils.render_node_size_usage,
        },
        {
            itemId: 'load',
            iconCls: 'pmx-icon-size fa fa-fw fa-tasks',
            title: gettext('Load average'),
            printBar: false,
            textField: 'loadavg',
        },
        {
            iconCls: 'pmx-icon-size fa fa-fw fa-hdd-o',
            itemId: 'rootfs',
            title: gettext('HD space') + ' (root)',
            valueField: 'rootfs',
            maxField: 'rootfs',
            renderer: ({ used, total }) => Proxmox.Utils.render_size_usage(used, total, true),
        },
        {
            iconCls: 'pmx-icon-size fa fa-fw fa-refresh',
            itemId: 'swap',
            printSize: true,
            title: gettext('SWAP usage'),
            valueField: 'swap',
            maxField: 'swap',
            renderer: Proxmox.Utils.render_node_size_usage,
        },
        {
            xtype: 'box',
            colspan: 2,
            padding: '0 0 20 0',
        },
        {
            itemId: 'cpus',
            colspan: 2,
            printBar: false,
            title: gettext('CPU(s)'),
            textField: 'cpuinfo',
            renderer: Proxmox.Utils.render_cpu_model,
            value: '',
        },
        {
            colspan: 2,
            title: gettext('Kernel Version'),
            printBar: false,
            // TODO: remove with next major and only use newish current-kernel textfield
            multiField: true,
            //textField: 'current-kernel',
            renderer: ({ data }) => {
                if (!data['current-kernel']) {
                    return data.kversion;
                }
                let kernel = data['current-kernel'];
                let buildDate = kernel.version.match(/\((.+)\)\s*$/)?.[1] ?? 'unknown';
                return `${kernel.sysname} ${kernel.release} (${buildDate})`;
            },
            value: '',
        },
        {
            colspan: 2,
            title: gettext('Boot Mode'),
            printBar: false,
            textField: 'boot-info',
            renderer: (boot) => {
                if (boot.mode === 'legacy-bios') {
                    return 'Legacy BIOS';
                } else if (boot.mode === 'efi') {
                    return `EFI${boot.secureboot ? ' (Secure Boot)' : ''}`;
                }
                return Proxmox.Utils.unknownText;
            },
            value: '',
        },
        {
            xtype: 'pmxNodeInfoRepoStatus',
            itemId: 'repositoryStatus',
            product: 'Proxmox Mail Gateway',
            repoLink: '#pmgServerAdministration:aptrepositories',
        },
    ],

    updateTitle: function () {
        var me = this;
        var uptime = Proxmox.Utils.render_uptime(me.getRecordValue('uptime'));
        me.setTitle(Proxmox.NodeName + ' (' + gettext('Uptime') + ': ' + uptime + ')');
    },

    initComponent: function () {
        let me = this;

        me.rstore = Ext.create('Proxmox.data.ObjectStore', {
            interval: 3000,
            url: '/api2/json/nodes/localhost/status',
            autoStart: true,
        });

        me.callParent();

        me.on('destroy', function () {
            me.rstore.stopUpdate();
        });
    },
});
Ext.define('PMG.dashboard.SubscriptionInfo', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgSubscriptionInfo',

    data: {
        icon: 'question-circle',
        message: gettext('Unknown'),
    },

    style: {
        cursor: 'pointer',
    },

    setSubStatus: function (status) {
        var me = this;
        var data = {};

        switch (status) {
            case 2:
                data.icon = 'check green';
                data.message = gettext('Your subscription status is valid.');
                break;
            case 1:
                data.icon = 'exclamation-triangle yellow';
                data.message = gettext('Warning: Your subscription levels are not the same.');
                break;
            case 0:
                data.icon = 'times-circle red';
                data.message = gettext('You have at least one node without subscription.');
                break;
            default:
                throw 'invalid subscription status';
        }
        me.update(data);
    },
    tpl: [
        '<table style="height: 100%;" class="dash">',
        '<tr><td class="center">',
        '<i class="fa fa-3x fa-{icon}"></i>',
        '</td><td class="center">{message}</td></tr>',
        '</table>',
    ],

    listeners: {
        click: {
            element: 'body',
            fn: function () {
                var mainview = this.component.up('mainview');
                mainview.getController().redirectTo('pmgSubscription');
            },
        },
    },
});
Ext.define('PMG.dashboard.MiniGraph', {
    extend: 'Proxmox.widget.RRDChart',
    xtype: 'pmgMiniGraph',

    legend: undefined,
    interactions: undefined,
    insetPadding: 20,
    padding: '5 5 0 0',
    axes: [
        {
            type: 'numeric',
            position: 'left',
            minimum: 0,
            grid: true,
            majorTickSteps: 2,
            label: {
                fillStyle: '#5f5f5f',
            },
            style: {
                axisLine: false,
                majorTickSize: 0,
            },
        },
        {
            type: 'time',
            position: 'bottom',
            dateFormat: 'H:i',
            fields: ['time'],
            label: {
                fillStyle: '#5f5f5f',
            },
            style: {
                axisLine: false,
                majorTickSize: 0,
            },
        },
    ],
    border: false,
    flex: 1,
    noTool: true,
});
Ext.define('PMG.view.main.VersionInfo', {
    extend: 'Ext.Component',
    xtype: 'versioninfo',

    makeApiCall: true,

    data: {
        version: false,
    },

    style: {
        'font-size': '14px',
        'line-height': '18px',
    },

    tpl: ['Mail Gateway', '<tpl if="version">', ' {version}', '</tpl>'],

    initComponent: function () {
        var me = this;
        me.callParent();

        if (me.makeApiCall) {
            Proxmox.Utils.API2Request({
                url: '/version',
                method: 'GET',
                success: function (response) {
                    me.update(response.result.data);
                },
            });
        }
    },
});
Ext.define('pmg-mail-tracker', {
    extend: 'Ext.data.Model',
    fields: [
        'id',
        'from',
        'to',
        'dstatus',
        'rstatus',
        'qid',
        'msgid',
        'client',
        { type: 'number', name: 'size' },
        { type: 'date', dateFormat: 'timestamp', name: 'time' },
    ],
    proxy: {
        type: 'proxmox',
    },
    // do not use field 'id', because "id/to" is the unique Id
    // this way we display an entry for each receiver
    idProperty: 'none',
});

Ext.define('PMG.MailTrackerFilter', {
    extend: 'Ext.container.Container',
    xtype: 'pmgMailTrackerFilter',

    layout: {
        type: 'hbox',
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        onFilterChange: function () {
            let view = this.getView();
            view.fireEvent('filterChanged');
        },

        onSpecialKey: function (field, e) {
            if (e.getKey() === e.ENTER) {
                this.onFilterChange();
            }
        },
    },

    getFilterParams: function () {
        let me = this;
        let param = {};

        let names = ['from', 'target', 'xfilter', 'starttime', 'endtime', 'ndr', 'greylist'];
        Ext.Array.each(names, function (name) {
            let value = me.lookupReference(name).getSubmitValue();
            if (value) {
                param[name] = value;
            }
        });

        // there must always be a start and endtime, otherwise the field was invalid
        if (!param.starttime || !param.endtime) {
            return undefined;
        }
        return param;
    },

    items: [
        {
            width: 400,
            border: false,
            padding: 10,
            layout: {
                type: 'vbox',
                align: 'stretch',
            },
            items: [
                {
                    fieldLabel: gettext('Sender'),
                    xtype: 'textfield',
                    listeners: { specialkey: 'onSpecialKey' },
                    reference: 'from',
                },
                {
                    fieldLabel: gettext('Receiver'),
                    xtype: 'textfield',
                    listeners: { specialkey: 'onSpecialKey' },
                    reference: 'target',
                },
                {
                    fieldLabel: gettext('Filter'),
                    xtype: 'textfield',
                    listeners: { specialkey: 'onSpecialKey' },
                    reference: 'xfilter',
                },
            ],
        },
        {
            border: false,
            padding: 10,
            layout: {
                type: 'vbox',
                align: 'stretch',
            },
            items: [
                {
                    fieldLabel: gettext('Start'),
                    reference: 'starttime',
                    listeners: {
                        change: {
                            fn: 'onFilterChange',
                            buffer: 500,
                        },
                    },
                    value: (function () {
                        let now = new Date();
                        return new Date(now.getTime() - 3600000);
                    })(),
                    xtype: 'promxoxDateTimeField',
                },
                {
                    fieldLabel: gettext('End'),
                    reference: 'endtime',
                    listeners: {
                        change: {
                            fn: 'onFilterChange',
                            buffer: 500,
                        },
                    },
                    value: (function () {
                        let now = new Date();
                        let tomorrow = new Date();
                        tomorrow.setDate(now.getDate() + 1);
                        tomorrow.setHours(0);
                        tomorrow.setMinutes(0);
                        tomorrow.setSeconds(0);
                        return tomorrow;
                    })(),
                    xtype: 'promxoxDateTimeField',
                },
                {
                    layout: {
                        type: 'hbox',
                    },
                    border: false,
                    items: [
                        {
                            boxLabel: gettext('Include Empty Senders'),
                            xtype: 'proxmoxcheckbox',
                            listeners: { change: 'onFilterChange' },
                            reference: 'ndr',
                            name: 'ndrs',
                        },
                        {
                            boxLabel: gettext('Include Greylist'),
                            xtype: 'proxmoxcheckbox',
                            listeners: { change: 'onFilterChange' },
                            margin: { left: 20 },
                            reference: 'greylist',
                            name: 'greylist',
                        },
                    ],
                },
            ],
        },
    ],
});

Ext.define('PMG.MaiLogWindow', {
    extend: 'Ext.window.Window',
    xtype: 'pmgMaiLogWindow',

    title: gettext('Syslog'),

    logid: undefined,
    starttime: undefined,
    endtime: undefined,

    width: 1024,
    height: 400,
    scrollable: true,

    layout: {
        type: 'auto',
    },
    modal: true,
    bodyPadding: 5,

    load: function () {
        let me = this;

        Proxmox.Utils.API2Request({
            method: 'GET',
            params: { starttime: me.starttime, endtime: me.endtime },
            url: '/nodes/' + Proxmox.NodeName + '/tracker/' + me.logid,
            waitMsgTarget: me,
            failure: function (response, opts) {
                me.update(gettext('Error') + ' ' + response.htmlStatus);
            },
            success: function (response, opts) {
                let data = response.result.data;

                let logs = "<pre style='margin: 0;'>";
                Ext.Array.each(data.logs, function (line) {
                    logs += Ext.htmlEncode(line) + '\n';
                });
                logs += '</pre>';
                me.update(logs);
            },
        });
    },

    initComponent: function () {
        let me = this;

        if (!me.logid) {
            throw 'no logid specified';
        }

        if (!me.starttime) {
            throw 'no starttime specified';
        }
        if (!me.endtime) {
            throw 'no endtime specified';
        }

        me.callParent();

        me.setHtml('Loading...');
        me.load();
    },
});

Ext.define('PMG.MailTracker', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgMailTracker',

    title: gettext('Tracking Center'),

    border: false,

    emptyText: gettext("Please enter your search parameters and press 'Search'."),
    disableSelection: true,

    viewConfig: {
        deferEmptyText: false,
        enableTextSelection: true,
        getRowClass: function (record, index) {
            let status = record.data.rstatus || record.data.dstatus;
            return PMG.Utils.mail_status_map[status];
        },
    },

    plugins: [
        {
            ptype: 'rowexpander',
            expandOnDblClick: false,
            rowBodyTpl: '<p class="logs">{logs}</p>',
        },
    ],

    store: {
        autoDestroy: true,
        model: 'pmg-mail-tracker',
        sorters: 'time',
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        onSearch: function () {
            let view = this.getView();
            view.setEmptyText(gettext('No data in database'));
            let filter = this.lookupReference('filter');
            let status = this.lookupReference('status');
            let params = filter.getFilterParams();
            if (params === undefined) {
                return; // something went wrong with the filters bail out
            }
            status.update(''); // clear status before load
            view.store.proxy.setExtraParams(params);
            view.store.proxy.setUrl('/api2/json/nodes/' + Proxmox.NodeName + '/tracker');
            view.store.load(function (records, operation, success) {
                let response = operation.getResponse();
                if (success) {
                    // fixme: howto avoid duplicate Ext.decode ?
                    let result = Ext.decode(response.responseText);
                    if (result.changes) {
                        status.update(result.changes);
                    }
                }
            });
        },

        showDetails: function (rowNode, record) {
            let view = this.getView();

            let params = view.store.proxy.getExtraParams();

            Proxmox.Utils.API2Request({
                method: 'GET',
                params: { starttime: params.starttime, endtime: params.endtime },
                url: '/nodes/' + Proxmox.NodeName + '/tracker/' + record.data.id,
                waitMsgTarget: view,
                failure: function (response, opts) {
                    record.set('logs', gettext('Error') + ' ' + response.htmlStatus);
                },
                success: function (response, opts) {
                    let data = response.result.data;
                    let logs = '';

                    Ext.Array.each(data.logs, function (line) {
                        logs += Ext.htmlEncode(line) + '<br>';
                    });

                    record.set('logs', logs);
                },
            });
        },

        // only expand row on dblclick, but do not collapse
        expand: function (view, record, row, rowIdx, e) {
            // inspired by RowExpander.js
            let rowNode = view.getNode(rowIdx);
            let normalRow = Ext.fly(rowNode);

            let collapsedCls = view.rowBodyFeature.rowCollapsedCls;

            if (normalRow.hasCls(collapsedCls)) {
                view.rowBodyFeature.rowExpander.toggleRow(rowIdx, record);
            }
        },

        control: {
            gridview: {
                expandbody: 'showDetails',
                itemdblclick: 'expand',
            },
        },
    },

    // extjs has no method to dynamically change the emptytext on
    // grids, so we have to do it this way
    setEmptyText: function (emptyText) {
        let me = this;
        let tableview = me.getView();
        tableview.emptyText = `<div class="x-grid-empty">${emptyText || ''}</div>`;
    },

    dockedItems: [
        {
            xtype: 'pmgMailTrackerFilter',
            reference: 'filter',
            listeners: { filterChanged: 'onSearch' },
            border: false,
            dock: 'top',
        },
        {
            xtype: 'toolbar',
            items: [
                { text: 'Search', handler: 'onSearch' },
                { xtype: 'component', html: '', reference: 'status' },
            ],
        },
    ],

    columns: [
        {
            xtype: 'datecolumn',
            header: gettext('Time'),
            width: 120,
            dataIndex: 'time',
            format: 'M d H:i:s',
        },
        {
            header: gettext('From'),
            flex: 1,
            dataIndex: 'from',
            renderer: Ext.htmlEncode,
        },
        {
            header: gettext('To'),
            flex: 1,
            dataIndex: 'to',
            renderer: Ext.htmlEncode,
        },
        {
            header: gettext('Status'),
            width: 150,
            renderer: function (v, metaData, rec) {
                let returntext = 'unknown';
                let icon = 'question-circle';
                let rstatus = rec.data.rstatus;
                if (v !== undefined && v !== '') {
                    let vtext = PMG.Utils.mail_status_map[v] || v;
                    icon = v;
                    if (v === 'Q' || v === 'B') {
                        returntext = vtext;
                    } else if (rstatus !== undefined && rstatus !== '') {
                        let rtext = PMG.Utils.mail_status_map[rstatus] || rstatus;
                        returntext = vtext + '/' + rtext;
                        icon = rstatus;
                    } else if (rec.data.qid !== undefined) {
                        returntext = 'queued/' + vtext;
                    } else {
                        returntext = vtext;
                    }
                }

                return PMG.Utils.format_status_icon(icon) + returntext;
            },
            dataIndex: 'dstatus',
        },
        {
            header: gettext('Size'),
            hidden: true,
            dataIndex: 'size',
        },
        {
            header: 'MSGID',
            width: 300,
            hidden: true,
            dataIndex: 'msgid',
            renderer: Ext.htmlEncode,
        },
        {
            header: gettext('Client'),
            width: 200,
            hidden: true,
            dataIndex: 'client',
            renderer: Ext.htmlEncode,
        },
    ],

    initComponent: function () {
        let me = this;

        me.callParent();

        Proxmox.Utils.monStoreErrors(me.getView(), me.store);
    },
});
Ext.define('PMG.store.NavigationStore', {
    extend: 'Ext.data.TreeStore',

    storeId: 'NavigationStore',

    root: {
        expanded: true,
        children: [
            {
                text: gettext('Dashboard'),
                iconCls: 'fa fa-tachometer',
                path: 'pmgDashboard',
                leaf: true,
            },
            {
                text: gettext('Mail Filter'),
                iconCls: 'fa fa-envelope',
                path: 'pmgRuleConfiguration',
                expanded: true,
                children: [
                    {
                        text: gettext('Action Objects'),
                        iconCls: 'fa fa-flag',
                        path: 'pmgActionList',
                        leaf: true,
                    },
                    {
                        text: gettext('Who Objects'),
                        iconCls: 'fa fa-user-circle',
                        path: 'pmgWhoConfiguration',
                        leaf: true,
                    },
                    {
                        text: gettext('What Objects'),
                        iconCls: 'fa fa-cube',
                        path: 'pmgWhatConfiguration',
                        leaf: true,
                    },
                    {
                        text: gettext('When Objects'),
                        iconCls: 'fa fa-clock-o',
                        path: 'pmgWhenConfiguration',
                        leaf: true,
                    },
                ],
            },
            {
                text: gettext('Configuration'),
                iconCls: 'fa fa-gears',
                path: 'pmgSystemConfiguration',
                expanded: true,
                children: [
                    {
                        text: gettext('Mail Proxy'),
                        iconCls: 'fa fa-envelope-o',
                        path: 'pmgMailProxyConfiguration',
                        leaf: true,
                    },
                    {
                        text: gettext('Spam Detector'),
                        iconCls: 'fa fa-bullhorn',
                        path: 'pmgSpamDetectorConfiguration',
                        leaf: true,
                    },
                    {
                        text: gettext('Virus Detector'),
                        iconCls: 'fa fa-bug',
                        path: 'pmgVirusDetectorConfiguration',
                        leaf: true,
                    },
                    {
                        text: gettext('User Management'),
                        iconCls: 'fa fa-users',
                        path: 'pmgUserManagement',
                        leaf: true,
                    },
                    {
                        text: gettext('Cluster'),
                        iconCls: 'fa fa-server',
                        path: 'pmgClusterAdministration',
                        leaf: true,
                    },
                    {
                        text: gettext('Subscription'),
                        iconCls: 'fa fa-support',
                        path: 'pmgSubscription',
                        leaf: true,
                    },
                    {
                        text: gettext('Backup/Restore'),
                        iconCls: 'fa fa-floppy-o',
                        path: 'pmgBackupConfiguration',
                        leaf: true,
                    },
                    {
                        text: gettext('Certificates'),
                        iconCls: 'fa fa-certificate',
                        path: 'pmgCertificateConfiguration',
                        leaf: true,
                    },
                ],
            },
            {
                text: gettext('Administration'),
                iconCls: 'fa fa-wrench',
                path: 'pmgServerAdministration',
                expanded: true,
                children: [
                    {
                        text: gettext('Spam Quarantine'),
                        iconCls: 'fa fa-bullhorn',
                        path: 'pmgSpamQuarantine',
                        leaf: true,
                    },
                    {
                        text: gettext('Virus Quarantine'),
                        iconCls: 'fa fa-bug',
                        path: 'pmgVirusQuarantine',
                        leaf: true,
                    },
                    {
                        text: gettext('Attachment Quarantine'),
                        iconCls: 'fa fa-paperclip',
                        path: 'pmgAttachmentQuarantine',
                        leaf: true,
                    },
                    {
                        text: gettext('User Welcomelist'),
                        iconCls: 'fa fa-file-o',
                        path: 'pmgUserWelcomelist',
                        leaf: true,
                    },
                    {
                        text: gettext('User Blocklist'),
                        iconCls: 'fa fa-file',
                        path: 'pmgUserBlocklist',
                        leaf: true,
                    },
                    {
                        text: gettext('Tracking Center'),
                        iconCls: 'fa fa-map-o',
                        path: 'pmgMailTracker',
                        leaf: true,
                    },
                    {
                        text: gettext('Queues'),
                        iconCls: 'fa fa-bars',
                        path: 'pmgQueueAdministration',
                        leaf: true,
                    },
                ],
            },
            {
                text: gettext('Statistics'),
                iconCls: 'fa fa-bar-chart',
                path: 'pmgGeneralMailStatistics',
                expanded: true,
                children: [
                    {
                        text: gettext('Spam Scores'),
                        iconCls: 'fa fa-table',
                        path: 'pmgSpamScoreDistribution',
                        leaf: true,
                    },
                    {
                        text: gettext('Virus Charts'),
                        iconCls: 'fa fa-bug',
                        path: 'pmgVirusCharts',
                        leaf: true,
                    },
                    {
                        text: gettext('Hourly Distribution'),
                        iconCls: 'fa fa-area-chart',
                        path: 'pmgHourlyMailDistribution',
                        leaf: true,
                    },
                    {
                        text: gettext('Postscreen'),
                        iconCls: 'fa fa-line-chart',
                        path: 'pmgRBLStatistics',
                        leaf: true,
                    },
                    {
                        text: gettext('Domain'),
                        iconCls: 'fa fa-table',
                        path: 'pmgDomainStatistics',
                        leaf: true,
                    },
                    {
                        text: gettext('Sender'),
                        iconCls: 'fa fa-table',
                        path: 'pmgSenderStatistics',
                        leaf: true,
                    },
                    {
                        text: gettext('Receiver'),
                        iconCls: 'fa fa-table',
                        path: 'pmgReceiverStatistics',
                        leaf: true,
                    },
                    {
                        text: gettext('Contact'),
                        iconCls: 'fa fa-table',
                        path: 'pmgContactStatistics',
                        leaf: true,
                    },
                ],
            },
        ],
    },
});

Ext.define('PMG.view.main.NavigationTree', {
    extend: 'Ext.list.Tree',
    xtype: 'navigationtree',

    select: function (path) {
        var me = this;
        var item = me.getStore().findRecord('path', path, 0, false, true, true);
        me.setSelection(item);
    },

    animation: false,
    expanderOnly: true,
    expanderFirst: false,
    store: 'NavigationStore',
    ui: 'pve-nav',
});
Ext.define('pmg-rule-list', {
    extend: 'Ext.data.Model',
    fields: [
        'id',
        'name',
        { name: 'active', type: 'boolean' },
        { name: 'direction', type: 'integer' },
        { name: 'priority', type: 'integer' },
    ],
    idProperty: 'id',
});

Ext.define('PMG.RulesConfiguration', {
    extend: 'Ext.container.Container',
    xtype: 'pmgRuleConfiguration',

    layout: 'border',
    border: false,
    defaults: {
        border: false,
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        selectedRuleChange: function (grid, selected, eOpts) {
            let me = this;
            let infoPanel = me.lookupReference('infopanel');
            let baseurl = '';

            if (selected.length > 0) {
                baseurl = `/config/ruledb/rules/${selected[0].data.id}`;
            }

            infoPanel.getController().setBaseUrl(baseurl);
        },

        editIconClick: function (gridView, rowindex, colindex, column, e, record) {
            let me = this;
            me.showEditWindow(gridView, record);
        },

        showEditWindow: function (gridView, record) {
            let _me = this;
            let win = Ext.create('PMG.RuleEditor', {
                url: `/api2/extjs/config/ruledb/rules/${record.data.id}/config`,
                listeners: {
                    destroy: function () {
                        gridView.getStore().load();
                    },
                },
            });
            win.load();
            win.show();
        },

        toggleIconClick: function (gridView, rowindex, colindex, column, e, record) {
            let _me = this;
            Proxmox.Utils.API2Request({
                url: `/config/ruledb/rules/${record.data.id}/config`,
                params: {
                    active: record.data.active ? 0 : 1,
                },
                method: 'PUT',
                callback: function () {
                    gridView.getStore().load();
                },
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        reload: function () {
            let me = this;
            me.lookupReference('rulegrid').getStore().load();
        },

        addRule: function () {
            let me = this;
            let win = Ext.create('PMG.RuleEditor', {
                url: '/api2/extjs/config/ruledb/rules/',
                method: 'POST',
                isCreate: true,
                listeners: {
                    destroy: function () {
                        me.lookupReference('rulegrid').getStore().load();
                    },
                },
            });
            win.load();
            win.show();
        },

        onFactoryDefaults: function () {
            let me = this;

            Ext.Msg.confirm(
                gettext('Confirm'),
                gettext('Reset rule database to factory defaults?'),
                function (button) {
                    if (button !== 'yes') {
                        return;
                    }
                    Proxmox.Utils.API2Request({
                        url: '/config/ruledb',
                        method: 'POST',
                        waitMsgTarget: me.getView(),
                        callback: function () {
                            me.reload();
                        },
                        failure: function (response, opts) {
                            Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                        },
                    });
                },
            );
        },

        init: function (view) {
            let grid = this.lookupReference('rulegrid');
            Proxmox.Utils.monStoreErrors(grid, grid.getStore(), true);
        },

        control: {
            'grid[reference=rulegrid]': {
                itemdblclick: 'showEditWindow',
                selectionchange: 'selectedRuleChange',
            },
            'button[reference=addButton]': {
                click: 'addRule',
            },
        },
    },

    viewModel: {
        data: {
            selectedRule: undefined,
            baseUrl: '/config/ruledb/rules',
        },
    },

    items: [
        {
            xtype: 'grid',
            layout: 'fit',
            title: 'Rules',
            reference: 'rulegrid',
            region: 'center',

            bind: {
                selection: '{selectedRule}',
            },

            dockedItems: {
                xtype: 'toolbar',
                reference: 'mytb',
                items: [
                    {
                        xtype: 'button',
                        text: gettext('Add'),
                        iconCls: 'fa fa-plus-circle',
                        reference: 'addButton',
                    },
                    {
                        xtype: 'proxmoxStdRemoveButton',
                        text: gettext('Remove'),
                        iconCls: 'fa fa-minus-circle',
                        reference: 'removeButton',
                        callback: 'reload',
                        getRecordName: function (rec) {
                            return rec.data.name;
                        },
                        bind: {
                            baseurl: '{baseUrl}',
                        },
                    },
                    '->',
                    {
                        text: gettext('Factory Defaults'),
                        handler: 'onFactoryDefaults',
                    },
                ],
            },

            viewConfig: {
                getRowClass: function (record, rowIndex) {
                    return record.get('active') ? 'enabled' : 'disabled';
                },
            },

            store: {
                autoLoad: true,
                model: 'pmg-rule-list',
                reference: 'rulesStore',
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/config/ruledb/rules',
                },
                sorters: [
                    {
                        property: 'priority',
                        direction: 'DESC',
                    },
                    {
                        property: 'name',
                    },
                ],
            },

            sortableColumns: false,
            columns: [
                {
                    text: 'Active',
                    dataIndex: 'active',
                    hidden: true,
                },
                {
                    text: 'Name',
                    dataIndex: 'name',
                    flex: 1,
                    renderer: Ext.htmlEncode,
                },
                {
                    text: 'Priority',
                    dataIndex: 'priority',
                },
                {
                    text: 'Direction',
                    dataIndex: 'direction',
                    renderer: PMG.Utils.format_rule_direction,
                },
                {
                    text: '',
                    xtype: 'actioncolumn',
                    align: 'center',
                    width: 70,
                    items: [
                        {
                            iconCls: 'fa fa-fw fa-pencil',
                            tooltip: 'Edit',
                            handler: 'editIconClick',
                        },
                        {
                            getClass: function (val, meta, rec) {
                                return (
                                    'fa fa-fw fa-' +
                                    (rec.get('active') ? 'toggle-on' : 'toggle-off')
                                );
                            },
                            getTip: function (val, meta, rec) {
                                return rec.get('active') ? 'Deactivate' : 'Activate';
                            },
                            handler: 'toggleIconClick',
                        },
                    ],
                },
            ],
        },
        {
            region: 'east',
            reference: 'infopanel',
            xtype: 'pmgRuleInfo',
            split: true,
            width: 440,
        },
    ],
});
Ext.define('PMG.SystemOptions', {
    extend: 'Proxmox.grid.ObjectGrid',
    xtype: 'pmgSystemOptions',

    monStoreErrors: true,
    interval: 5000,
    cwidth1: 200,

    url: '/api2/json/config/admin',
    editorConfig: {
        url: '/api2/extjs/config/admin',
        onlineHelp: 'pmgconfig_systemconfig',
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        onEdit: function () {
            let view = this.getView();
            view.run_editor();
        },
    },

    tbar: [
        {
            text: gettext('Edit'),
            xtype: 'proxmoxButton',
            disabled: true,
            handler: 'onEdit',
        },
    ],

    listeners: {
        itemdblclick: 'onEdit',
    },

    add_proxy_row: function (name, text, opts) {
        let me = this;

        opts = opts || {};
        me.rows = me.rows || {};

        me.rows[name] = {
            required: true,
            defaultValue: Proxmox.Utils.noneText,
            header: text,
            renderer: Ext.htmlEncode,
            editor: {
                xtype: 'proxmoxWindowEdit',
                onlineHelp: 'pmgconfig_systemconfig',
                subject: text,
                items: {
                    xtype: 'proxmoxtextfield',
                    vtype: 'HttpProxy',
                    name: name,
                    deleteEmpty: true,
                    emptyText: Proxmox.Utils.noneText,
                    labelWidth: Proxmox.Utils.compute_min_label_width(text, opts.labelWidth),
                    fieldLabel: text,
                },
            },
        };
    },

    initComponent: function () {
        let me = this;

        me.add_boolean_row('dailyreport', gettext('Send daily admin reports'), {
            defaultValue: 1,
        });

        me.add_boolean_row('advfilter', gettext('Use advanced statistic filters'), {
            defaultValue: 0,
        });

        me.add_integer_row('statlifetime', gettext('User statistic lifetime (days)'), {
            minValue: 1,
            defaultValue: 7,
            deleteEmpty: true,
        });

        me.add_text_row('email', gettext('Administrator EMail'), {
            deleteEmpty: true,
            defaultValue: Proxmox.Utils.noneText,
            renderer: Ext.htmlEncode,
        });

        me.add_text_row('admin-mail-from', gettext("'From:' for Admin Mail"), {
            deleteEmpty: true,
            defaultValue: Proxmox.Utils.noneText,
            renderer: Ext.htmlEncode,
        });

        me.add_proxy_row('http_proxy', gettext('HTTP proxy'));

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
        me.on('destroy', me.rstore.stopUpdate);
    },
});
Ext.define('PMG.SubscriptionKeyEdit', {
    extend: 'Proxmox.window.Edit',

    title: gettext('Upload Subscription Key'),
    width: 300,
    autoLoad: true,

    onlineHelp: 'getting_help',

    items: {
        xtype: 'proxmoxtextfield',
        trimValue: true,
        name: 'key',
        value: '',
        fieldLabel: gettext('Subscription Key'),
    },
});

Ext.define('PMG.Subscription', {
    extend: 'Proxmox.grid.ObjectGrid',
    xtype: 'pmgSubscription',

    title: gettext('Subscription'),

    border: false,

    onlineHelp: 'getting_help',

    viewConfig: {
        enableTextSelection: true,
    },

    showReport: function () {
        let me = this;

        const getReportFileName = function () {
            let now = Ext.Date.format(new Date(), 'D-d-F-Y-G-i');
            return Proxmox.NodeName + '-report-' + now + '.txt';
        };

        let view = Ext.createWidget('component', {
            itemId: 'system-report-view',
            scrollable: true,
            style: {
                'white-space': 'pre',
                'font-family': 'monospace',
                padding: '5px',
            },
        });

        let reportWindow = Ext.create('Ext.window.Window', {
            title: gettext('System Report'),
            width: 1024,
            height: 600,
            layout: 'fit',
            modal: true,
            buttons: [
                '->',
                {
                    text: gettext('Download'),
                    handler: function () {
                        let fileContent = Ext.String.htmlDecode(
                            reportWindow.getComponent('system-report-view').html,
                        );
                        let fileName = getReportFileName();

                        // Internet Explorer
                        if (window.navigator.msSaveOrOpenBlob) {
                            navigator.msSaveOrOpenBlob(new Blob([fileContent]), fileName);
                        } else {
                            let element = document.createElement('a');
                            element.setAttribute(
                                'href',
                                'data:text/plain;charset=utf-8,' + encodeURIComponent(fileContent),
                            );
                            element.setAttribute('download', fileName);
                            element.style.display = 'none';
                            document.body.appendChild(element);
                            element.click();
                            document.body.removeChild(element);
                        }
                    },
                },
            ],
            items: view,
        });

        Proxmox.Utils.API2Request({
            url: '/api2/extjs/nodes/' + Proxmox.NodeName + '/report',
            method: 'GET',
            waitMsgTarget: me,
            failure: function (response) {
                Ext.Msg.alert(gettext('Error'), response.htmlStatus);
            },
            success: function (response) {
                let report = Ext.htmlEncode(response.result.data);
                reportWindow.show();
                view.update(report);
            },
        });
    },

    initComponent: function () {
        let me = this;

        const reload = function () {
            me.rstore.load();
        };

        let baseurl = `/nodes/${Proxmox.NodeName}/subscription`;

        let render_status = function (value) {
            let message = me.getObjectValue('message');
            if (message) {
                return value + ': ' + message;
            }
            return value;
        };

        let rows = {
            productname: {
                header: gettext('Type'),
            },
            key: {
                header: gettext('Subscription Key'),
            },
            status: {
                header: gettext('Status'),
                renderer: render_status,
            },
            message: {
                visible: false,
            },
            serverid: {
                header: gettext('Server ID'),
            },
            sockets: {
                header: gettext('Sockets'),
            },
            checktime: {
                header: gettext('Last checked'),
                renderer: Proxmox.Utils.render_timestamp,
            },
            nextduedate: {
                header: gettext('Next due date'),
            },
        };

        Ext.apply(me, {
            url: '/api2/json' + baseurl,
            cwidth1: 170,
            tbar: [
                {
                    text: gettext('Upload Subscription Key'),
                    handler: function () {
                        let win = Ext.create('PMG.SubscriptionKeyEdit', {
                            url: '/api2/extjs/' + baseurl,
                            autoShow: true,
                        });
                        win.on('destroy', reload);
                    },
                },
                {
                    text: gettext('Remove Subscription'),
                    xtype: 'proxmoxStdRemoveButton',
                    confirmMsg: gettext('Are you sure to remove the subscription key?'),
                    baseurl: baseurl,
                    dangerous: true,
                    selModel: false,
                    callback: reload,
                },
                {
                    text: gettext('Check'),
                    handler: function () {
                        Proxmox.Utils.API2Request({
                            params: { force: 1 },
                            url: baseurl,
                            method: 'POST',
                            waitMsgTarget: me,
                            failure: function (response, opts) {
                                Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                            },
                            callback: reload,
                        });
                    },
                },
                '-',
                {
                    text: gettext('System Report'),
                    handler: function () {
                        Proxmox.Utils.checked_command(function () {
                            me.showReport();
                        });
                    },
                },
            ],
            rows: rows,
        });

        me.callParent();

        reload();
    },
});
Ext.define('PMG.BackupConfiguration', {
    extend: 'Ext.tab.Panel',
    alias: 'widget.pmgBackupConfiguration',

    title: gettext('Backup'),

    border: false,
    defaults: { border: false },

    items: [
        {
            itemId: 'local',
            title: gettext('Local Backup/Restore'),
            xtype: 'pmgBackupRestore',
            iconCls: 'fa fa-folder',
        },
        {
            itemId: 'proxmoxbackupserver',
            title: 'Proxmox Backup Server',
            xtype: 'pmgPBSConfig',
            iconCls: 'fa fa-floppy-o',
        },
    ],
});
Ext.define('pmg-backup-list', {
    extend: 'Ext.data.Model',
    fields: [
        'filename',
        { type: 'integer', name: 'size' },
        { type: 'date', dateFormat: 'timestamp', name: 'timestamp' },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/nodes/' + Proxmox.NodeName + '/backup',
    },
    idProperty: 'filename',
});

Ext.define('PMG.RestoreWindow', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgRestoreWindow',
    onlineHelp: 'chapter_pmgbackup',

    showProgress: true,
    title: gettext('Restore'),
    isCreate: true,
    method: 'POST',
    submitText: gettext('Restore'),
    fieldDefaults: {
        labelWidth: 150,
    },

    initComponent: function () {
        let me = this;

        me.items = [
            {
                xtype: 'proxmoxcheckbox',
                name: 'config',
                fieldLabel: gettext('System Configuration'),
            },
            {
                xtype: 'proxmoxcheckbox',
                name: 'database',
                value: 1,
                uncheckedValue: 0,
                fieldLabel: gettext('Rule Database'),
                listeners: {
                    change: function (field, value) {
                        field.nextSibling('field[name=statistic]').setDisabled(!value);
                    },
                },
            },
            {
                xtype: 'proxmoxcheckbox',
                name: 'statistic',
                fieldLabel: gettext('Statistic'),
            },
        ];

        let restorePath;
        if (me.filename) {
            restorePath = `backup/${encodeURIComponent(me.filename)}`;
        } else if (me.backup_time) {
            restorePath = `pbs/${me.remote}/snapshot/${me.backup_id}/${me.backup_time}`;
        } else {
            throw 'neither filename nor snapshot given';
        }
        me.url = `/nodes/${Proxmox.NodeName}/${restorePath}`;

        me.callParent();
    },
});

Ext.define('PMG.BackupWindow', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgBackupWindow',
    onlineHelp: 'chapter_pmgbackup',

    showProgress: true,
    title: gettext('Backup'),
    isCreate: true,
    method: 'POST',
    submitText: gettext('Backup'),
    fieldDefaults: {
        labelWidth: 150,
    },
    showTaskViewer: true,
    items: [
        {
            xtype: 'proxmoxcheckbox',
            name: 'statistic',
            value: 1,
            uncheckedValue: 0,
            fieldLabel: gettext('Include Statistics'),
        },
    ],
});

Ext.define('PMG.BackupRestore', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgBackupRestore',

    title: gettext('Backup') + '/' + gettext('Restore'),

    controller: {
        xclass: 'Ext.app.ViewController',

        createBackup: function () {
            let view = this.getView();
            Ext.create('PMG.BackupWindow', {
                url: '/nodes/' + Proxmox.NodeName + '/backup',
                taskDone: () => view.store.load(),
            }).show();
        },

        onRestore: function () {
            let view = this.getView();
            let rec = view.getSelection()[0];

            if (!(rec && rec.data && rec.data.filename)) {
                return;
            }

            Ext.create('PMG.RestoreWindow', {
                filename: rec.data.filename,
            }).show();
        },

        onAfterRemove: function (btn, res) {
            let view = this.getView();
            view.store.load();
        },
    },

    tbar: [
        {
            text: gettext('Backup Now'),
            handler: 'createBackup',
        },
        '-',
        {
            xtype: 'proxmoxButton',
            text: gettext('Restore'),
            handler: 'onRestore',
            disabled: true,
        },
        {
            xtype: 'proxmoxStdRemoveButton',
            baseurl: '/nodes/' + Proxmox.NodeName + '/backup',
            reference: 'removeBtn',
            callback: 'onAfterRemove',
            waitMsgTarget: true,
        },
    ],

    store: {
        autoLoad: true,
        model: 'pmg-backup-list',
        sorters: [
            {
                property: 'timestamp',
                direction: 'DESC',
            },
        ],
    },

    columns: [
        {
            header: gettext('Filename'),
            width: 300,
            sortable: true,
            renderer: Ext.htmlEncode,
            dataIndex: 'filename',
        },
        {
            xtype: 'datecolumn',
            header: gettext('Time'),
            width: 150,
            format: 'Y-m-d H:i',
            sortable: true,
            dataIndex: 'timestamp',
        },
        {
            header: gettext('Size'),
            width: 100,
            sortable: true,
            renderer: Proxmox.Utils.render_size,
            dataIndex: 'size',
        },
        {
            header: gettext('Download'),
            renderer: function (filename) {
                return (
                    "<a class='download' href='" +
                    '/api2/json/nodes/' +
                    Proxmox.NodeName +
                    '/backup/' +
                    encodeURIComponent(filename) +
                    "'><i class='fa fa-fw fa-download'</i></a>"
                );
            },
            dataIndex: 'filename',
        },
    ],
});
Ext.define('PMG.PBSConfig', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgPBSConfig',

    controller: {
        xclass: 'Ext.app.ViewController',

        callRestore: function (grid, record) {
            let remote = this.getViewModel().get('remote');
            Ext.create('PMG.RestoreWindow', {
                remote: remote,
                backup_id: record.data['backup-id'],
                backup_time: record.data['backup-time'],
            }).show();
        },

        restoreSnapshot: function (button) {
            let me = this;
            let view = me.lookup('snapshotsGrid');
            let record = view.getSelection()[0];
            me.callRestore(view, record);
        },

        runBackup: function (button) {
            let me = this;
            let view = me.lookup('snapshotsGrid');
            let remote = me.getViewModel().get('remote');
            Ext.create('PMG.BackupWindow', {
                url: `/nodes/${Proxmox.NodeName}/pbs/${remote}/snapshot`,
                taskDone: () => view.getStore().load(),
            }).show();
        },

        reload: function (grid) {
            let me = this;
            let selection = grid.getSelection();
            me.showInfo(grid, selection);
        },

        showInfo: function (grid, selected) {
            let me = this;
            let viewModel = me.getViewModel();
            if (selected[0]) {
                let remote = selected[0].data.remote;
                viewModel.set('selected', true);
                viewModel.set('remote', remote);

                // set grid stores and load them
                let remstore = me.lookup('snapshotsGrid').getStore();
                remstore
                    .getProxy()
                    .setUrl(`/api2/json/nodes/${Proxmox.NodeName}/pbs/${remote}/snapshot`);
                remstore.load();

                let scheduleStore = me.lookup('schedulegrid').rstore;
                scheduleStore
                    .getProxy()
                    .setUrl(`/api2/json/nodes/${Proxmox.NodeName}/pbs/${remote}/timer`);
                scheduleStore.load();
            } else {
                viewModel.set('selected', false);
            }
        },
        reloadSnapshots: function () {
            let me = this;
            let grid = me.lookup('grid');
            let selection = grid.getSelection();
            me.showInfo(grid, selection);
        },
        init: function (view) {
            let me = this;
            me.lookup('grid').relayEvents(view, ['activate']);

            let remoteGrid = me.lookup('grid');
            view.mon(remoteGrid.store, 'load', function (store, r, success, o) {
                if (success) {
                    remoteGrid.getSelectionModel().select(0);
                }
            });

            let snapshotGrid = me.lookup('snapshotsGrid');
            let schedulegrid = me.lookup('schedulegrid');

            Proxmox.Utils.monStoreErrors(snapshotGrid, snapshotGrid.getStore(), true);
            Proxmox.Utils.monStoreErrors(schedulegrid, schedulegrid.getStore(), true);
        },

        control: {
            'grid[reference=grid]': {
                selectionchange: 'showInfo',
                load: 'reload',
            },
            'grid[reference=snapshotsGrid]': {
                itemdblclick: 'restoreSnapshot',
            },
        },
    },

    viewModel: {
        data: {
            remote: '',
            selected: false,
        },
    },

    layout: 'border',

    items: [
        {
            xtype: 'pmgPBSConfigGrid',
            reference: 'grid',
            title: gettext('Remote'),
            hidden: false,
            region: 'center',
            minHeight: 130,
            border: false,
        },
        {
            xtype: 'proxmoxObjectGrid',
            region: 'south',
            reference: 'schedulegrid',
            title: gettext('Schedule'),
            height: 155,
            border: false,
            hidden: true,
            emptyText: gettext('No schedule setup.'),
            tbar: [
                {
                    text: gettext('Set Schedule'),
                    handler: function () {
                        let me = this;
                        let remote = me.lookupViewModel().get('remote');
                        let win = Ext.createWidget('pmgPBSScheduleEdit', {
                            remote: remote,
                            autoShow: true,
                        });
                        win.on('destroy', () => me.up('grid').rstore.load());
                    },
                },
                {
                    xtype: 'proxmoxStdRemoveButton',
                    baseurl: `/nodes/${Proxmox.NodeName}/pbs/`,
                    callback: function () {
                        this.up('grid').rstore.load();
                    },
                    text: gettext('Remove Schedule'),
                    selModel: false,
                    confirmMsg: function (_rec) {
                        let me = this;
                        let remote = me.lookupViewModel().get('remote');
                        return Ext.String.format(
                            gettext('Are you sure you want to remove the schedule for {0}'),
                            `'${remote}'`,
                        );
                    },
                    getUrl: function (_rec) {
                        let remote = this.lookupViewModel().get('remote');
                        return `${this.baseurl}/${remote}/timer`;
                    },
                },
                '->',
                {
                    text: gettext('Reload'),
                    iconCls: 'fa fa-refresh',
                    handler: function () {
                        this.up('grid').rstore.load();
                    },
                },
            ],
            bind: {
                title: Ext.String.format(gettext("Schedule on '{0}'"), '{remote}'),
                hidden: '{!selected}',
            },
            url: '/', // hack, obj. grid is a bit dumb..
            rows: {
                schedule: {
                    text: gettext('Schedule'),
                    required: true,
                    defaultValue: gettext('None'),
                },
                delay: {
                    text: gettext('Delay'),
                },
                'next-run': {
                    text: gettext('Next Run'),
                },
            },
        },
        {
            xtype: 'grid',
            region: 'south',
            reference: 'snapshotsGrid',
            height: '50%',
            border: false,
            split: true,
            hidden: true,
            emptyText: gettext('No backups on remote'),
            tbar: [
                {
                    text: gettext('Backup Now'),
                    handler: 'runBackup',
                },
                '-',
                {
                    xtype: 'proxmoxButton',
                    text: gettext('Restore'),
                    handler: 'restoreSnapshot',
                    disabled: true,
                },
                {
                    xtype: 'proxmoxStdRemoveButton',
                    text: gettext('Forget Snapshot'),
                    disabled: true,
                    getUrl: function (rec) {
                        let me = this;
                        let remote = me.lookupViewModel().get('remote');
                        let snapshot = `${rec.data['backup-id']}/${rec.data['backup-time']}`;
                        return `/nodes/${Proxmox.NodeName}/pbs/${remote}/snapshot/${snapshot}`;
                    },
                    confirmMsg: function (rec) {
                        let _me = this;
                        let snapshot = `${rec.data['backup-id']}/${rec.data['backup-time']}`;
                        return Ext.String.format(
                            gettext('Are you sure you want to forget snapshot {0}'),
                            `'${snapshot}'`,
                        );
                    },
                    callback: 'reloadSnapshots',
                },
                '->',
                {
                    text: gettext('Reload'),
                    iconCls: 'fa fa-refresh',
                    handler: function () {
                        this.up('grid').store.load();
                    },
                },
            ],
            store: {
                fields: ['backup-id', 'backup-time', 'size', 'ctime', 'encrypted'],
                proxy: { type: 'proxmox' },
                sorters: [
                    {
                        property: 'backup-time',
                        direction: 'DESC',
                    },
                ],
            },
            bind: {
                title: Ext.String.format(gettext("Backup snapshots on '{0}'"), '{remote}'),
                hidden: '{!selected}',
            },
            columns: [
                {
                    text: 'Group ID',
                    dataIndex: 'backup-id',
                    flex: 1,
                },
                {
                    text: 'Time',
                    dataIndex: 'backup-time',
                    width: 180,
                },
                {
                    text: 'Size',
                    dataIndex: 'size',
                    renderer: Proxmox.Utils.render_size,
                    flex: 1,
                },
                {
                    text: 'Encrypted',
                    dataIndex: 'encrypted',
                    hidden: true, // FIXME: actually return from API
                    renderer: Proxmox.Utils.format_boolean,
                    flex: 1,
                },
            ],
        },
    ],
});
Ext.define('PMG.PBSInputPanel', {
    extend: 'Ext.tab.Panel',
    xtype: 'pmgPBSInputPanel',
    mixins: ['Proxmox.Mixin.CBind'],

    bodyPadding: 10,
    remoteId: undefined,

    cbindData: function (initialConfig) {
        let me = this;
        me.isCreate = initialConfig.isCreate || !initialConfig.remoteId;
        return {};
    },

    items: [
        {
            xtype: 'inputpanel',
            title: gettext('Backup Server'),
            onGetValues: function (values) {
                values.disable = values.enable ? 0 : 1;
                delete values.enable;
                return values;
            },
            column1: [
                {
                    xtype: 'pmxDisplayEditField',
                    name: 'remote',
                    cbind: {
                        editable: '{isCreate}',
                    },
                    fieldLabel: gettext('ID'),
                    allowBlank: false,
                },
                {
                    xtype: 'pmxDisplayEditField',
                    name: 'server',
                    vtype: 'DnsOrIp',
                    fieldLabel: gettext('Server'),
                    cbind: { editable: '{isCreate}' },
                    allowBlank: false,
                },
                {
                    xtype: 'pmxDisplayEditField',
                    name: 'datastore',
                    fieldLabel: 'Datastore',
                    cbind: { editable: '{isCreate}' },
                    allowBlank: false,
                },
                {
                    xtype: 'pmxDisplayEditField',
                    name: 'namespace',
                    fieldLabel: gettext('Namespace'),
                    cbind: { editable: '{isCreate}' },
                    emptyText: gettext('Root'),
                },
            ],
            column2: [
                {
                    xtype: 'pmxDisplayEditField',
                    name: 'username',
                    fieldLabel: gettext('Username'),
                    emptyText: gettext('Example') + ': admin@pbs',
                    cbind: { editable: '{isCreate}' },
                    regex: /\S+@\w+/,
                    regexText: gettext('Example') + ': admin@pbs',
                    allowBlank: false,
                },
                {
                    xtype: 'pmxDisplayEditField',
                    editable: true, // FIXME: set to false if (!create && user == token)
                    editConfig: {
                        xtype: 'proxmoxtextfield',
                    },
                    inputType: 'password',
                    name: 'password',
                    cbind: {
                        allowBlank: '{!isCreate}',
                        emptyText: (get) => (get('isCreate') ? '' : gettext('Unchanged')),
                    },
                    fieldLabel: gettext('Password'),
                },
                {
                    xtype: 'proxmoxKVComboBox',
                    name: 'notify',
                    fieldLabel: gettext('Notify'),
                    comboItems: [
                        ['always', gettext('Always')],
                        ['error', gettext('Errors')],
                        ['never', gettext('Never')],
                    ],
                    deleteEmpty: false,
                    emptyText: gettext('Never'),
                },
                {
                    xtype: 'proxmoxcheckbox',
                    name: 'enable',
                    checked: true,
                    uncheckedValue: 0,
                    fieldLabel: gettext('Enable'),
                },
            ],
            columnB: [
                {
                    xtype: 'proxmoxcheckbox',
                    name: 'include-statistics',
                    checked: true,
                    uncheckedValue: 0,
                    fieldLabel: gettext('Statistics'),
                    boxLabel: gettext('Include in Backup'),
                },
                {
                    xtype: 'proxmoxtextfield',
                    name: 'fingerprint',
                    fieldLabel: gettext('Fingerprint'),
                    emptyText: gettext(
                        'Server certificate SHA-256 fingerprint, required for self-signed certificates',
                    ),
                    regex: /[A-Fa-f0-9]{2}(:[A-Fa-f0-9]{2}){31}/,
                    regexText: gettext('Example') + ': AB:CD:EF:...',
                    allowBlank: true,
                },
            ],
        },
        {
            xtype: 'inputpanel',
            title: gettext('Prune Options'),
            defaults: {
                // set nested, else we'd only set the defaults for the two column containers
                defaults: {
                    minValue: 1,
                    labelWidth: 100,
                    allowBlank: true,
                },
            },
            column1: [
                {
                    xtype: 'proxmoxintegerfield',
                    fieldLabel: gettext('Keep Last'),
                    name: 'keep-last',
                    cbind: { deleteEmpty: '{!isCreate}' },
                },
                {
                    xtype: 'proxmoxintegerfield',
                    fieldLabel: gettext('Keep Daily'),
                    name: 'keep-daily',
                    cbind: { deleteEmpty: '{!isCreate}' },
                },
                {
                    xtype: 'proxmoxintegerfield',
                    fieldLabel: gettext('Keep Monthly'),
                    name: 'keep-monthly',
                    cbind: { deleteEmpty: '{!isCreate}' },
                },
            ],
            column2: [
                {
                    xtype: 'proxmoxintegerfield',
                    fieldLabel: gettext('Keep Hourly'),
                    name: 'keep-hourly',
                    cbind: { deleteEmpty: '{!isCreate}' },
                },
                {
                    xtype: 'proxmoxintegerfield',
                    fieldLabel: gettext('Keep Weekly'),
                    name: 'keep-weekly',
                    cbind: { deleteEmpty: '{!isCreate}' },
                },
                {
                    xtype: 'proxmoxintegerfield',
                    fieldLabel: gettext('Keep Yearly'),
                    name: 'keep-yearly',
                    cbind: { deleteEmpty: '{!isCreate}' },
                },
            ],
        },
    ],
});

Ext.define('PMG.PBSEdit', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgPBSEdit',
    onlineHelp: 'pmgbackup_pbs_remotes',

    subject: 'Proxmox Backup Server',
    isAdd: true,

    bodyPadding: 0,

    initComponent: function () {
        let me = this;

        me.isCreate = !me.remoteId;

        me.method = 'POST';
        me.url = '/api2/extjs/config/pbs';
        if (!me.isCreate) {
            me.url += `/${me.remoteId}`;
            me.method = 'PUT';
        }

        me.items = [
            {
                xtype: 'pmgPBSInputPanel',
                isCreate: me.isCreate,
                remoteId: me.remoteId,
            },
        ];

        me.callParent();

        if (!me.isCreate) {
            me.load({
                success: function (response, options) {
                    let values = response.result.data;

                    values.enable = values.disable ? 0 : 1;
                    me.setValues(values);
                },
            });
        }
    },
});
Ext.define('PMG.PBSScheduleEdit', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgPBSScheduleEdit',
    onlineHelp: 'pmgbackup_pbs_schedule',

    isAdd: true,
    isCreate: true,
    submitText: gettext('Set'),

    method: 'POST',
    title: gettext('Configure Scheduled Backup'),
    autoLoad: true,
    items: [
        {
            xtype: 'proxmoxKVComboBox',
            name: 'schedule',
            fieldLabel: gettext('Schedule'),
            comboItems: [
                ['hourly', 'hourly'],
                ['daily', 'daily'],
                ['weekly', 'weekly'],
                ['monthly', 'monthly'],
            ],
            editable: true,
            deleteEmpty: false,
            emptyText: gettext('daily'),
        },
        {
            xtype: 'proxmoxKVComboBox',
            name: 'delay',
            fieldLabel: gettext('Random Delay'),
            comboItems: [
                ['0s', gettext('No Delay')],
                ['15 minutes', '15 Minutes'],
                ['6 hours', '6 hours'],
            ],
            editable: true,
            deleteEmpty: false,
            emptyText: gettext('5 Minutes'),
        },
    ],
    initComponent: function () {
        let me = this;

        me.url = `/nodes/${Proxmox.NodeName}/pbs/${me.remote}/timer`;
        me.callParent();
    },
});

Ext.define('pmg-pbs-config', {
    extend: 'Ext.data.Model',
    fields: ['remote', 'server', 'datastore', 'username', 'disabled'],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/config/pbs',
    },
    idProperty: 'remote',
});

Ext.define('PMG.PBSConfigGrid', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgPBSConfigGrid',

    controller: {
        xclass: 'Ext.app.ViewController',

        run_editor: function () {
            let me = this;
            let view = me.getView();
            let rec = view.getSelection()[0];
            if (!rec) {
                return;
            }

            let win = Ext.createWidget('pmgPBSEdit', {
                remoteId: rec.data.remote,
                autoLoad: true,
                autoShow: true,
            });
            win.on('destroy', me.reload, me);
        },

        newRemote: function () {
            let me = this;
            let win = Ext.createWidget('pmgPBSEdit', {});
            win.on('destroy', me.reload, me);
            win.show();
        },

        reload: function () {
            let me = this;
            let view = me.getView();
            view.getStore().load();
        },

        init: function (view) {
            let _me = this;
            Proxmox.Utils.monStoreErrors(view, view.getStore(), true);
        },
    },

    store: {
        model: 'pmg-pbs-config',
        sorters: [
            {
                property: 'remote',
                direction: 'ASC',
            },
        ],
    },

    tbar: [
        {
            text: gettext('Add Remote'),
            handler: 'newRemote',
        },
        '-',
        {
            xtype: 'proxmoxButton',
            text: gettext('Edit'),
            disabled: true,
            handler: 'run_editor',
        },
        {
            xtype: 'proxmoxStdRemoveButton',
            baseurl: '/config/pbs',
            callback: 'reload',
        },
        '->',
        {
            text: gettext('Reload'),
            iconCls: 'fa fa-refresh',
            handler: 'reload',
        },
    ],

    listeners: {
        itemdblclick: 'run_editor',
        activate: 'reload',
    },

    columns: [
        {
            header: gettext('Remote'),
            sortable: true,
            dataIndex: 'remote',
            flex: 2,
        },
        {
            header: gettext('Server'),
            sortable: true,
            dataIndex: 'server',
            flex: 2,
        },
        {
            header: gettext('Datastore'),
            sortable: true,
            dataIndex: 'datastore',
            flex: 1,
        },
        {
            header: gettext('User ID'),
            sortable: true,
            dataIndex: 'username',
            flex: 1,
        },
        {
            header: gettext('Encryption'),
            width: 80,
            sortable: true,
            hidden: true, // for now
            dataIndex: 'encryption-key',
            renderer: Proxmox.Utils.format_boolean,
        },
        {
            header: gettext('Enabled'),
            width: 80,
            sortable: true,
            dataIndex: 'disable',
            renderer: Proxmox.Utils.format_neg_boolean,
        },
    ],
});
Ext.define('PMG.SystemConfiguration', {
    extend: 'Ext.tab.Panel',
    xtype: 'pmgSystemConfiguration',

    title: gettext('Configuration') + ': ' + gettext('System'),
    border: false,
    scrollable: true,
    defaults: { border: false },
    items: [
        {
            title: gettext('Network/Time'),
            itemId: 'network',
            iconCls: 'fa fa-exchange',
            xtype: 'panel',
            layout: {
                type: 'vbox',
                align: 'stretch',
                multi: true,
            },
            bodyPadding: '0 0 10 0',
            defaults: {
                collapsible: true,
                animCollapse: false,
                margin: '10 10 0 10',
            },
            items: [
                {
                    title: gettext('Time'),
                    xtype: 'proxmoxNodeTimeView',
                    nodename: Proxmox.NodeName,
                },
                {
                    title: gettext('DNS'),
                    xtype: 'proxmoxNodeDNSView',
                    nodename: Proxmox.NodeName,
                },
                {
                    flex: 1,
                    minHeight: 200,
                    title: gettext('Interfaces'),
                    xtype: 'proxmoxNodeNetworkView',
                    types: ['bond'],
                    nodename: Proxmox.NodeName,
                    showApplyBtn: true,
                },
            ],
        },
        {
            xtype: 'pmgSystemOptions',
            itemId: 'options',
            title: gettext('Options'),
            iconCls: 'fa fa-cogs',
        },
    ],

    initComponent: function () {
        var me = this;

        me.callParent();

        var networktime = me.getComponent('network');
        Ext.Array.forEach(networktime.query(), function (item) {
            item.relayEvents(networktime, ['activate', 'deactivate', 'destroy']);
        });
    },
});
Ext.define('PMG.form.DNSBLSitesGrid', {
    extend: 'Ext.grid.Panel',
    alias: 'widget.pmgDnsblSitesGrid',

    mixins: ['Ext.form.field.Field'],

    allowBlank: true,
    selectAll: false,
    isFormField: true,
    deleteEmpty: false,
    selModel: 'checkboxmodel',

    config: {
        deleteEmpty: false,
    },

    emptyText: gettext('No Sites defined'),
    viewConfig: {
        deferEmptyText: false,
    },

    setValue: function (value) {
        let me = this;
        let sites = value ?? '';
        if (!sites) {
            me.getStore().removeAll();
            me.checkChange();
            return me;
        }
        let matcher = /^(.*?)(?:=(.*?))?(?:\*(.*))?$/;
        let entries = (sites.split(/[;, ]/) || [])
            .filter((s) => !!s)
            .map((entry) => {
                let [, site, filter, weight] = matcher.exec(entry);
                return {
                    site,
                    filter,
                    weight,
                };
            });
        me.getStore().setData(entries);
        me.checkChange();
        return me;
    },

    getValue: function () {
        let me = this;
        let values = [];
        me.getStore().each((rec) => {
            let val = rec.data.site;
            if (rec.data.filter) {
                val += `=${rec.data.filter}`;
            }
            if (rec.data.weight) {
                val += `*${rec.data.weight}`;
            }
            values.push(val);
        });
        return values.join(';');
    },

    getErrors: function (value) {
        let me = this;
        let emptySite = false;
        me.getStore().each((rec) => {
            if (!rec.data.site) {
                emptySite = true;
            }
        });
        let errors = [];
        if (emptySite) {
            errors.push(gettext('Site must not be empty.'));
        }
        return errors;
    },

    // override framework function to implement deleteEmpty behaviour
    getSubmitData: function () {
        let me = this,
            data = null,
            val;
        if (!me.disabled && me.submitValue) {
            val = me.getValue();
            if (val !== null && val !== '') {
                data = {};
                data[me.getName()] = val;
            } else if (me.getDeleteEmpty()) {
                data = {};
                data.delete = me.getName();
            }
        }
        return data;
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        addLine: function () {
            let me = this;
            me.getView().getStore().add({
                site: '',
                filter: '',
                weight: '',
            });
        },

        removeSelection: function () {
            let me = this;
            let view = me.getView();
            let selection = view.getSelection();
            if (selection === undefined) {
                return;
            }

            selection.forEach((sel) => {
                view.getStore().remove(sel);
            });
            view.checkChange();
        },

        fieldChange: function (field, newValue, oldValue) {
            let me = this;
            let view = me.getView();
            let rec = field.getWidgetRecord();
            if (!rec) {
                return;
            }
            let column = field.getWidgetColumn();
            rec.set(column.dataIndex, newValue);
            view.checkChange();
        },
    },

    tbar: [
        {
            text: gettext('Add'),
            handler: 'addLine',
        },
        {
            xtype: 'proxmoxButton',
            text: gettext('Remove'),
            handler: 'removeSelection',
            disabled: true,
        },
    ],

    columns: [
        {
            header: gettext('Site'),
            dataIndex: 'site',
            xtype: 'widgetcolumn',
            widget: {
                xtype: 'proxmoxtextfield',
                isFormField: false,
                allowBlank: false,
                listeners: {
                    change: 'fieldChange',
                },
            },
            flex: 1,
        },
        {
            header: gettext('Filter'),
            xtype: 'widgetcolumn',
            flex: 1,
            dataIndex: 'filter',
            widget: {
                xtype: 'proxmoxtextfield',
                emptyText: Proxmox.Utils.noneText,
                allowBlank: true,
                isFormField: false,
                listeners: {
                    change: 'fieldChange',
                },
            },
        },
        {
            header: gettext('Weight'),
            xtype: 'widgetcolumn',
            flex: 1,
            dataIndex: 'weight',
            widget: {
                xtype: 'proxmoxintegerfield',
                emptyText: '1',
                allowBlank: true,
                isFormField: false,
                listeners: {
                    change: 'fieldChange',
                },
            },
        },
    ],

    store: {
        listeners: {
            update: function () {
                this.commitChanges();
            },
        },
    },

    initComponent: function () {
        let me = this;
        me.callParent();
        me.initField();
    },
});
Ext.define('PMG.MailProxyRelaying', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgMailProxyRelaying'],

    monStoreErrors: true,

    initComponent: function () {
        var me = this;

        me.add_text_row('relay', gettext('Default Relay'), {
            deleteEmpty: true,
            defaultValue: Proxmox.Utils.noneText,
            renderer: Ext.htmlEncode,
            onlineHelp: 'pmgconfig_mailproxy_relaying',
        });

        me.add_integer_row('relayport', gettext('Relay Port'), {
            defaultValue: 25,
            deleteEmpty: true,
            minValue: 1,
            maxValue: 65535,
            onlineHelp: 'pmgconfig_mailproxy_relaying',
        });

        me.add_combobox_row('relayprotocol', gettext('Relay Protocol'), {
            defaultValue: 'smtp',
            comboItems: [
                ['smtp', 'SMTP'],
                ['lmtp', 'LMTP'],
            ],
            onlineHelp: 'pmgconfig_mailproxy_relaying',
        });

        me.add_boolean_row('relaynomx', gettext('Disable MX lookup (SMTP)'), {
            onlineHelp: 'pmgconfig_mailproxy_relaying',
        });

        me.rows.smarthost = {
            required: true,
            multiKey: ['smarthost', 'smarthostport'],
            header: gettext('Smarthost'),
            renderer: function () {
                var host = me.getObjectValue('smarthost', undefined);
                var port = me.getObjectValue('smarthostport', undefined);
                var result = '';
                if (host) {
                    if (port) {
                        if (host.match(Proxmox.Utils.IP6_match)) {
                            result = '[' + host + ']:' + port;
                        } else {
                            result = host + ':' + port;
                        }
                    } else {
                        result = host;
                    }
                }
                if (result === '') {
                    result = Proxmox.Utils.noneText;
                }
                return result;
            },
            editor: {
                xtype: 'proxmoxWindowEdit',
                onlineHelp: 'pmgconfig_mailproxy_relaying',
                subject: gettext('Smarthost'),
                fieldDefaults: {
                    labelWidth: 100,
                },
                items: [
                    {
                        xtype: 'proxmoxtextfield',
                        name: 'smarthost',
                        deleteEmpty: true,
                        emptyText: Proxmox.Utils.noneText,
                        fieldLabel: gettext('Smarthost'),
                    },
                    {
                        xtype: 'proxmoxintegerfield',
                        name: 'smarthostport',
                        deleteEmpty: true,
                        minValue: 1,
                        maxValue: 65535,
                        emptyText: Proxmox.Utils.defaultText,
                        fieldLabel: gettext('Port'),
                    },
                ],
            },
            onlineHelp: 'pmgconfig_mailproxy_relaying',
        };

        me.rows.smarthostport = { visible: false };

        var baseurl = '/config/mail';

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        Ext.apply(me, {
            tbar: [
                {
                    text: gettext('Edit'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        me.run_editor();
                    },
                    selModel: me.selModel,
                },
            ],
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_mailproxy_relaying',
            },
            interval: 5000,
            cwidth1: 200,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('PMG.MailProxyPorts', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgMailProxyPorts'],

    url: '/api2/json/config/mail',

    monStoreErrors: true,

    editorConfig: {
        url: '/api2/extjs/config/mail',
        onlineHelp: 'pmgconfig_mailproxy_ports',
    },

    interval: 5000,

    cwidth1: 200,

    controller: {
        xclass: 'Ext.app.ViewController',

        onEdit: function () {
            this.getView().run_editor();
        },
    },

    listeners: {
        itemdblclick: 'onEdit',
    },

    tbar: [
        {
            text: gettext('Edit'),
            xtype: 'proxmoxButton',
            disabled: true,
            handler: 'onEdit',
        },
    ],

    initComponent: function () {
        var me = this;

        me.add_integer_row('ext_port', gettext('External SMTP Port'), {
            defaultValue: 25,
            deleteEmpty: true,
            minValue: 1,
            maxValue: 65535,
        });

        me.add_integer_row('int_port', gettext('Internal SMTP Port'), {
            defaultValue: 26,
            deleteEmpty: true,
            minValue: 1,
            maxValue: 65535,
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('PMG.MailProxyOptions', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgMailProxyOptions'],

    monStoreErrors: true,

    initComponent: function () {
        var me = this;

        me.add_integer_row('maxsize', gettext('Message Size (bytes)'), {
            defaultValue: 1024 * 1024 * 10,
            minValue: 1024,
            deleteEmpty: true,
        });

        me.add_boolean_row('rejectunknown', gettext('Reject Unknown Clients'));

        me.add_boolean_row('rejectunknownsender', gettext('Reject Unknown Senders'));

        me.add_boolean_row('helotests', gettext('SMTP HELO checks'));

        var render_verifyreceivers = function (value) {
            if (value === undefined || value === '__default__') {
                return Proxmox.Utils.noText;
            }
            return Proxmox.Utils.yesText + ' (' + value + ')';
        };

        me.add_combobox_row('verifyreceivers', gettext('Verify Receivers'), {
            renderer: render_verifyreceivers,
            defaultValue: '__default__',
            deleteEmpty: true,
            comboItems: [
                ['__default__', render_verifyreceivers('__default__')],
                ['450', render_verifyreceivers('450')],
                ['550', render_verifyreceivers('550')],
            ],
        });

        me.add_boolean_row('greylist', gettext('Use Greylisting for IPv4'), {
            defaultValue: 1,
        });

        me.add_integer_row('greylistmask4', gettext('Netmask for Greylisting IPv4'), {
            defaultValue: 24,
            minValue: 0,
            maxValue: 32,
        });

        me.add_boolean_row('greylist6', gettext('Use Greylisting for IPv6'), {
            defaultValue: 0,
        });

        me.add_integer_row('greylistmask6', gettext('Netmask for Greylisting IPv6'), {
            defaultValue: 64,
            minValue: 0,
            maxValue: 128,
        });

        me.add_boolean_row('spf', gettext('Use SPF'), { defaultValue: 1 });

        me.add_boolean_row('hide_received', gettext('Hide Internal Hosts'));

        me.add_integer_row('dwarning', gettext('Delay Warning Time (hours)'), {
            defaultValue: 4,
            minValue: 0,
        });

        me.add_integer_row('conn_count_limit', gettext('Client Connection Count Limit'), {
            defaultValue: 50,
            minValue: 0,
            maxValue: 65535,
        });

        me.add_integer_row('conn_rate_limit', gettext('Client Connection Rate Limit'), {
            defaultValue: 0,
            minValue: 0,
        });

        me.add_integer_row('message_rate_limit', gettext('Client Message Rate Limit'), {
            defaultValue: 0,
            minValue: 0,
        });

        me.add_text_row('banner', gettext('SMTPD Banner'), {
            deleteEmpty: true,
            defaultValue: 'ESMTP Proxmox',
            renderer: Ext.htmlEncode,
        });

        me.add_boolean_row('ndr_on_block', gettext('Send NDR on Blocked E-Mails'));
        // FIXME allow to pass onlineHelp to ObjectGrid's add_xyz_row..
        // onlineHelp: 'pmgconfig_mailproxy_before_after_queue',
        me.rows.ndr_on_block.editor.onlineHelp = 'pmgconfig_mailproxy_before_after_queue';

        me.add_boolean_row('before_queue_filtering', gettext('Before Queue Filtering'));

        me.add_integer_row('dnsbl_threshold', gettext('DNSBL Threshold'), {
            deleteEmpty: true,
            defaultValue: 1,
            minValue: 0,
        });

        me.rows.dnsbl_sites = {
            required: true,
            header: gettext('DNSBL Sites'),
            renderer: function () {
                return (me.getObjectValue('dnsbl_sites') ?? '')
                    .split(/[;, ]/)
                    .filter((s) => !!s)
                    .map((site) => Ext.htmlEncode(site))
                    .join('<br>');
            },
            editor: {
                xtype: 'proxmoxWindowEdit',
                subject: gettext('DNSBL Sites'),
                fieldDefaults: {
                    labelWidth: 100,
                },
                width: 600,
                height: 400,
                items: [
                    {
                        xtype: 'pmgDnsblSitesGrid',
                        name: 'dnsbl_sites',
                        deleteEmpty: true,
                    },
                ],
            },
        };

        var baseurl = '/config/mail';

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        Ext.apply(me, {
            tbar: [
                {
                    text: gettext('Edit'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        me.run_editor();
                    },
                    selModel: me.selModel,
                },
            ],
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_mailproxy_options',
            },
            interval: 5000,
            cwidth1: 200,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('PMG.MailProxyTLS', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgMailProxyTLS'],

    monStoreErrors: true,

    initComponent: function () {
        var me = this;

        me.add_boolean_row('tls', gettext('Enable TLS'));

        me.add_boolean_row('tlslog', gettext('Enable TLS Logging'));

        me.add_boolean_row('tlsheader', gettext('Add TLS received header'));

        var baseurl = '/config/mail';

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        Ext.apply(me, {
            tbar: [
                {
                    text: gettext('Edit'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        me.run_editor();
                    },
                    selModel: me.selModel,
                },
            ],
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_mailproxy_tls',
            },
            interval: 5000,
            cwidth1: 200,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('PMG.MailProxyTLSPanel', {
    extend: 'Ext.panel.Panel',
    alias: 'widget.pmgMailProxyTLSPanel',

    layout: {
        type: 'vbox',
        align: 'stretch',
    },

    bodyPadding: '0 0 10 0',
    defaults: {
        collapsible: true,
        animCollapse: false,
        margin: '10 10 0 10',
    },

    initComponent: function () {
        var me = this;

        var tlsSettings = Ext.create('PMG.MailProxyTLS', {
            title: gettext('Settings'),
        });

        var tlsDestinations = Ext.create('PMG.MailProxyTLSDestinations', {
            title: gettext('TLS Destination Policy'),
            flex: 1,
        });

        const tlsInboundDomains = Ext.create('PMG.MailProxyTLSInboundDomains', {
            title: gettext('TLS Inbound Domains'),
            flex: 1,
        });

        me.items = [tlsSettings, tlsDestinations, tlsInboundDomains];

        me.callParent();

        tlsSettings.relayEvents(me, ['activate', 'deactivate', 'destroy']);
        tlsDestinations.relayEvents(me, ['activate', 'deactivate', 'destroy']);
        tlsInboundDomains.relayEvents(me, ['activate', 'deactivate', 'destroy']);
    },
});
Ext.define('pmg-tls-policy', {
    extend: 'Ext.data.Model',
    fields: ['destination', 'policy'],
    idProperty: 'destination',
    proxy: {
        type: 'proxmox',
        url: '/api2/json/config/tlspolicy',
    },
    sorters: {
        property: 'destination',
        direction: 'ASC',
    },
});

Ext.define('PMG.TLSDestinationEdit', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgTLSDestinationEdit',
    onlineHelp: 'pmgconfig_mailproxy_tls',

    subject: gettext('TLS Policy'),
    initComponent: function () {
        var me = this;

        var isCreate = !Ext.isDefined(me.destination);

        var url = '/api2/extjs/config/tlspolicy' + (isCreate ? '' : '/' + me.destination);
        var method = isCreate ? 'POST' : 'PUT';
        var text = isCreate ? 'Create' : 'Edit';

        var items = [
            {
                xtype: isCreate ? 'proxmoxtextfield' : 'displayfield',
                name: 'destination',
                fieldLabel: gettext('Destination'),
            },
            {
                xtype: 'proxmoxKVComboBox',
                name: 'policy',
                fieldLabel: gettext('Policy'),
                deleteEmpty: false,
                comboItems: [
                    ['none', 'none'],
                    ['may', 'may'],
                    ['encrypt', 'encrypt'],
                    ['dane', 'dane'],
                    ['dane-only', 'dane-only'],
                    ['fingerprint', 'fingerprint'],
                    ['verify', 'verify'],
                    ['secure', 'secure'],
                ],
                allowBlank: true,
                value: 'encrypt',
            },
        ];

        Ext.apply(me, {
            url: url,
            method: method,
            items: items,
            text: text,
        });

        me.callParent();
    },
});

Ext.define('PMG.MailProxyTLSDestinations', {
    extend: 'Ext.grid.GridPanel',
    alias: ['widget.pmgMailProxyTLSDestinations'],

    viewConfig: {
        trackOver: false,
    },
    columns: [
        {
            header: gettext('Destination'),
            width: 200,
            sortable: true,
            dataIndex: 'destination',
        },
        {
            header: gettext('Policy'),
            sortable: false,
            dataIndex: 'policy',
            flex: 1,
        },
    ],

    initComponent: function () {
        let me = this;

        let rstore = Ext.create('Proxmox.data.UpdateStore', {
            model: 'pmg-tls-policy',
            storeid: 'pmg-mailproxy-tls-store-' + ++Ext.idSeed,
        });

        let store = Ext.create('Proxmox.data.DiffStore', { rstore: rstore });

        let reload = () => rstore.load();

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        var run_editor = function () {
            var rec = me.selModel.getSelection()[0];
            if (!rec) {
                return;
            }

            var win = Ext.createWidget('pmgTLSDestinationEdit', {
                destination: rec.data.destination,
            });

            win.load();
            win.on('destroy', reload);
            win.show();
        };

        let tbar = [
            {
                text: gettext('Create'),
                handler: () =>
                    Ext.createWidget('pmgTLSDestinationEdit', {
                        autoLoad: true,
                        autoShow: true,
                        listeners: {
                            destroy: () => reload(),
                        },
                    }),
            },
            '-',
            {
                xtype: 'proxmoxButton',
                disabled: true,
                text: gettext('Edit'),
                handler: run_editor,
            },
            {
                xtype: 'proxmoxStdRemoveButton',
                baseurl: '/config/tlspolicy',
                callback: reload,
                waitMsgTarget: me,
            },
        ];

        Proxmox.Utils.monStoreErrors(me, store, true);

        Ext.apply(me, {
            store: store,
            tbar: tbar,
            run_editor: run_editor,
            listeners: {
                itemdblclick: run_editor,
                activate: reload,
            },
        });

        me.on('activate', rstore.startUpdate);
        me.on('destroy', rstore.stopUpdate);
        me.on('deactivate', rstore.stopUpdate);
        me.callParent();
    },
});
Ext.define('pmg-tls-inbound-domains', {
    extend: 'Ext.data.Model',
    fields: ['domain'],
    idProperty: 'domain',
    proxy: {
        type: 'proxmox',
        url: '/api2/json/config/tls-inbound-domains',
    },
    sorters: {
        property: 'domain',
        direction: 'ASC',
    },
});

Ext.define('PMG.TLSInboundDomainsEdit', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgTLSInboundDomainsEdit',
    onlineHelp: 'pmgconfig_mailproxy_tls',

    subject: gettext('TLS Inbound domains'),
    url: '/api2/extjs/config/tls-inbound-domains',
    method: 'POST',

    items: [
        {
            xtype: 'proxmoxtextfield',
            name: 'domain',
            fieldLabel: gettext('Domain'),
        },
    ],
});

Ext.define('PMG.MailProxyTLSInboundDomains', {
    extend: 'Ext.grid.GridPanel',
    alias: ['widget.pmgMailProxyTLSInboundDomains'],

    viewConfig: {
        trackOver: false,
    },

    columns: [
        {
            header: gettext('Domain'),
            flex: 1,
            sortable: true,
            dataIndex: 'domain',
        },
    ],

    initComponent: function () {
        const me = this;

        const rstore = Ext.create('Proxmox.data.UpdateStore', {
            model: 'pmg-tls-inbound-domains',
            storeid: 'pmg-mailproxy-tls-inbound-domains-store-' + ++Ext.idSeed,
        });

        const store = Ext.create('Proxmox.data.DiffStore', { rstore: rstore });
        const reload = () => rstore.load();
        me.selModel = Ext.create('Ext.selection.RowModel', {});
        Proxmox.Utils.monStoreErrors(me, store, true);

        Ext.apply(me, {
            store,
            tbar: [
                {
                    text: gettext('Create'),
                    handler: () => {
                        Ext.createWidget('pmgTLSInboundDomainsEdit', {
                            autoShow: true,
                            listeners: {
                                destroy: reload,
                            },
                        });
                    },
                },
                {
                    xtype: 'proxmoxStdRemoveButton',
                    baseurl: '/config/tls-inbound-domains',
                    callback: reload,
                    waitMsgTarget: me,
                },
            ],
            listeners: {
                activate: rstore.startUpdate,
                destroy: rstore.stopUpdate,
                deactivate: rstore.stopUpdate,
            },
        });

        me.callParent();
    },
});
Ext.define('pmg-transport', {
    extend: 'Ext.data.Model',
    fields: [
        'domain',
        'host',
        'protocol',
        { name: 'port', type: 'integer' },
        { name: 'use_mx', type: 'boolean' },
        'comment',
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/config/transport',
    },
    idProperty: 'domain',
});

Ext.define('PMG.Transport', {
    extend: 'Ext.grid.GridPanel',
    alias: ['widget.pmgTransport'],

    initComponent: function () {
        let me = this;

        let store = new Ext.data.Store({
            model: 'pmg-transport',
            sorters: {
                property: 'domain',
                direction: 'ASC',
            },
        });
        Proxmox.Utils.monStoreErrors(me, store, true);
        let reload = () => store.load();

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        let run_editor = function () {
            let rec = me.selModel.getSelection()[0];
            if (!rec) {
                return;
            }

            let win = Ext.createWidget('pmgTransportEditor', {
                url: '/api2/extjs/config/transport/' + rec.data.domain,
                method: 'PUT',
            });
            win.load();
            win.on('destroy', reload);
            win.show();
        };

        Ext.apply(me, {
            store: store,
            tbar: [
                {
                    text: gettext('Create'),
                    handler: () =>
                        Ext.createWidget('pmgTransportEditor', {
                            autoShow: true,
                            method: 'POST',
                            url: '/api2/extjs/config/transport',
                            isCreate: true,
                            listeners: {
                                destroy: () => reload(),
                            },
                        }),
                },
                '-',
                {
                    xtype: 'proxmoxButton',
                    text: gettext('Edit'),
                    disabled: true,
                    selModel: me.selModel,
                    handler: run_editor,
                },
                {
                    xtype: 'proxmoxStdRemoveButton',
                    selModel: me.selModel,
                    baseurl: '/config/transport',
                    callback: reload,
                    waitMsgTarget: me,
                },
                '->',
                {
                    xtype: 'pmgFilterField',
                    filteredFields: ['domain', 'host', 'port', 'protocol', 'comment'],
                },
            ],
            viewConfig: {
                trackOver: false,
            },
            columns: [
                {
                    header: gettext('Relay Domain'),
                    width: 200,
                    dataIndex: 'domain',
                },
                {
                    header: gettext('Host'),
                    width: 200,
                    dataIndex: 'host',
                },
                {
                    header: gettext('Protocol'),
                    width: 200,
                    dataIndex: 'protocol',
                },
                {
                    header: gettext('Port'),
                    width: 80,
                    dataIndex: 'port',
                },
                {
                    header: gettext('Use MX'),
                    width: 80,
                    renderer: Proxmox.Utils.format_boolean,
                    dataIndex: 'use_mx',
                },
                {
                    header: gettext('Comment'),
                    renderer: Ext.String.htmlEncode,
                    dataIndex: 'comment',
                    flex: 1,
                },
            ],
            listeners: {
                itemdblclick: run_editor,
                activate: reload,
            },
        });

        me.callParent();
    },
});

Ext.define('PMG.TransportEditor', {
    extend: 'Proxmox.window.Edit',
    alias: 'widget.pmgTransportEditor',
    mixins: ['Proxmox.Mixin.CBind'],

    cbindData: (cfg) => ({
        domainXType: cfg.method === 'POST' ? 'proxmoxtextfield' : 'displayfield',
    }),

    viewModel: {
        data: {
            proto: 'smtp',
        },
        formulas: {
            protoIsSMTP: (get) => get('proto') === 'smtp',
        },
    },
    onlineHelp: 'pmgconfig_mailproxy_transports',
    subject: gettext('Transport'),

    items: [
        {
            xtype: 'displayfield',
            cbind: {
                xtype: '{domainXType}',
            },
            name: 'domain',
            fieldLabel: gettext('Relay Domain'),
        },
        {
            xtype: 'textfield',
            name: 'host',
            fieldLabel: gettext('Host'),
        },
        {
            xtype: 'proxmoxKVComboBox',
            name: 'protocol',
            fieldLabel: gettext('Protocol'),
            deleteEmpty: false,
            comboItems: [
                ['smtp', 'SMTP'],
                ['lmtp', 'LMTP'],
            ],
            allowBlank: true,
            value: 'smtp',
            bind: {
                value: '{proto}',
            },
        },
        {
            xtype: 'proxmoxintegerfield',
            name: 'port',
            value: 25,
            minValue: 1,
            maxValue: 65535,
            fieldLabel: gettext('Port'),
        },
        {
            xtype: 'proxmoxcheckbox',
            name: 'use_mx',
            checked: true,
            bind: {
                disabled: '{!protoIsSMTP}',
                hidden: '{!protoIsSMTP}',
            },
            uncheckedValue: 0,
            fieldLabel: gettext('Use MX'),
        },
        {
            xtype: 'textfield',
            name: 'comment',
            fieldLabel: gettext('Comment'),
        },
    ],
});
Ext.define('pmg-mynetworks', {
    extend: 'Ext.data.Model',
    fields: ['cidr', 'comment'],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/config/mynetworks',
    },
    idProperty: 'cidr',
});

Ext.define('PMG.MyNetworks', {
    extend: 'Ext.grid.GridPanel',
    alias: ['widget.pmgMyNetworks'],

    initComponent: function () {
        let me = this;

        let store = new Ext.data.Store({
            model: 'pmg-mynetworks',
            sorters: {
                property: 'cidr',
                direction: 'ASC',
            },
        });
        let reload = () => store.load();

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        let run_editor = function () {
            let rec = me.selModel.getSelection()[0];
            if (!rec) {
                return;
            }
            Ext.createWidget('proxmoxWindowEdit', {
                autoShow: true,
                autoLoad: true,
                url: '/api2/extjs/config/mynetworks/' + rec.data.cidr,
                onlineHelp: 'pmgconfig_mailproxy_networks',
                method: 'PUT',
                subject: gettext('Trusted Network'),
                items: [
                    {
                        xtype: 'displayfield',
                        name: 'cidr',
                        fieldLabel: gettext('CIDR'),
                    },
                    {
                        xtype: 'textfield',
                        name: 'comment',
                        fieldLabel: gettext('Comment'),
                    },
                ],
                listeners: {
                    destroy: () => reload(),
                },
            });
        };

        let tbar = [
            {
                text: gettext('Create'),
                handler: () =>
                    Ext.createWidget('proxmoxWindowEdit', {
                        autoShow: true,
                        method: 'POST',
                        url: '/api2/extjs/config/mynetworks',
                        onlineHelp: 'pmgconfig_mailproxy_networks',
                        isCreate: true,
                        subject: gettext('Trusted Network'),
                        items: [
                            {
                                xtype: 'proxmoxtextfield',
                                name: 'cidr',
                                fieldLabel: gettext('CIDR'),
                            },
                            {
                                xtype: 'proxmoxtextfield',
                                name: 'comment',
                                fieldLabel: gettext('Comment'),
                            },
                        ],
                        listeners: {
                            destroy: () => reload(),
                        },
                    }),
            },
            '-',
            {
                xtype: 'proxmoxButton',
                text: gettext('Edit'),
                disabled: true,
                selModel: me.selModel,
                handler: run_editor,
            },
            {
                xtype: 'proxmoxStdRemoveButton',
                selModel: me.selModel,
                baseurl: '/config/mynetworks',
                callback: reload,
                waitMsgTarget: me,
            },
            '->',
            {
                xtype: 'pmgFilterField',
                filteredFields: ['cidr', 'comment'],
            },
        ];

        Proxmox.Utils.monStoreErrors(me, store, true);

        Ext.apply(me, {
            store: store,
            tbar: tbar,
            run_editor: run_editor,
            viewConfig: {
                trackOver: false,
            },
            columns: [
                {
                    header: gettext('Trusted Network'),
                    width: 200,
                    sortable: true,
                    dataIndex: 'cidr',
                },
                {
                    header: gettext('Comment'),
                    sortable: false,
                    renderer: Ext.String.htmlEncode,
                    dataIndex: 'comment',
                    flex: 1,
                },
            ],
            listeners: {
                itemdblclick: run_editor,
                activate: reload,
            },
        });

        me.callParent();
    },
});
Ext.define('pmg-domains', {
    extend: 'Ext.data.Model',
    fields: ['domain', 'comment'],
    idProperty: 'domain',
});

Ext.define('PMG.RelayDomains', {
    extend: 'Ext.grid.GridPanel',
    alias: ['widget.pmgRelayDomains'],

    baseurl: '/config/domains',
    domain_desc: gettext('Relay Domain'),

    onlineHelp: 'pmgconfig_mailproxy_relay_domains',

    initComponent: function () {
        var me = this;

        var store = new Ext.data.Store({
            model: 'pmg-domains',
            sorters: {
                property: 'domain',
                direction: 'ASC',
            },
            proxy: {
                type: 'proxmox',
                url: '/api2/json' + me.baseurl,
            },
        });

        var reload = function () {
            store.load();
        };

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        var run_editor = function () {
            var rec = me.selModel.getSelection()[0];
            if (!rec) {
                return;
            }

            var config = {
                url: '/api2/extjs' + me.baseurl + '/' + rec.data.domain,
                onlineHelp: me.onlineHelp,
                method: 'PUT',
                subject: me.domain_desc,
                items: [
                    {
                        xtype: 'displayfield',
                        name: 'domain',
                        fieldLabel: me.domain_desc,
                    },
                    {
                        xtype: 'textfield',
                        name: 'comment',
                        fieldLabel: gettext('Comment'),
                    },
                ],
            };

            var win = Ext.createWidget('proxmoxWindowEdit', config);

            win.load();
            win.on('destroy', reload);
            win.show();
        };

        let tbar = [
            {
                text: gettext('Create'),
                handler: () =>
                    Ext.createWidget('proxmoxWindowEdit', {
                        autoShow: true,
                        method: 'POST',
                        url: '/api2/extjs' + me.baseurl,
                        onlineHelp: me.onlineHelp,
                        isCreate: true,
                        subject: gettext('Relay Domain'),
                        items: [
                            {
                                xtype: 'proxmoxtextfield',
                                name: 'domain',
                                fieldLabel: me.domain_desc,
                            },
                            {
                                xtype: 'proxmoxtextfield',
                                name: 'comment',
                                fieldLabel: gettext('Comment'),
                            },
                        ],
                        listeners: {
                            destroy: () => reload(),
                        },
                    }),
            },
            '-',
            {
                xtype: 'proxmoxButton',
                text: gettext('Edit'),
                disabled: true,
                selModel: me.selModel,
                handler: run_editor,
            },
            {
                xtype: 'proxmoxStdRemoveButton',
                selModel: me.selModel,
                baseurl: me.baseurl,
                callback: reload,
                waitMsgTarget: me,
            },
            '->',
            {
                xtype: 'pmgFilterField',
                filteredFields: ['domain', 'comment'],
            },
        ];

        Proxmox.Utils.monStoreErrors(me, store, true);

        Ext.apply(me, {
            store: store,
            tbar: tbar,
            run_editor: run_editor,
            viewConfig: {
                trackOver: false,
            },
            columns: [
                {
                    header: me.domain_desc,
                    width: 200,
                    sortable: true,
                    dataIndex: 'domain',
                },
                {
                    header: gettext('Comment'),
                    sortable: false,
                    renderer: Ext.String.htmlEncode,
                    dataIndex: 'comment',
                    flex: 1,
                },
            ],
            listeners: {
                itemdblclick: run_editor,
                activate: reload,
            },
        });

        me.callParent();
    },
});
Ext.define('PMG.DKIMEnableEdit', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgDKIMEnableEdit',

    items: [
        {
            xtype: 'displayfield',
            userCls: 'pmx-hint',
            value: gettext('You need to create a Selector before enabling DKIM Signing'),
            hidden: true,
        },
        {
            xtype: 'proxmoxcheckbox',
            name: 'dkim_sign',
            uncheckedValue: 0,
            defaultValue: 0,
            checked: false,
            deleteDefaultValue: false,
            fieldLabel: gettext('Sign Outgoing Mails'),
        },
    ],
    listeners: {
        show: function () {
            var me = this;
            var disablefn = function (errormsg) {
                Ext.Msg.alert(gettext('Error'), errormsg);
                me.down('displayfield').setVisible(true);
                me.down('proxmoxcheckbox').setDisabled(true);
                me.close();
            };

            Proxmox.Utils.API2Request({
                url: '/config/dkim/selector',
                method: 'GET',
                failure: function (response, opts) {
                    disablefn(response.htmlStatus);
                },
                success: function (response, opts) {
                    if (!Ext.isDefined(response.result.data.record)) {
                        disablefn(
                            gettext('Could not read private key - please create a selector first!'),
                        );
                    }
                },
            });
        },
    },
});

Ext.define('PMG.SelectorViewer', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgDKIMSelectorView',

    url: '/config/dkim/selector',
    title: gettext('Selector'),

    width: 800,
    resizable: false,

    items: [
        {
            xtype: 'displayfield',
            fieldLabel: gettext('Selector'),
            name: 'selector',
        },
        {
            xtype: 'displayfield',
            fieldLabel: gettext('Key Size'),
            name: 'keysize',
        },
        {
            xtype: 'textarea',
            editable: false,
            height: 220,
            fieldLabel: gettext('DNS TXT Record'),
            name: 'record',
            value: 'Could not read private key!',
        },
    ],

    initComponent: function () {
        var me = this;

        me.callParent();

        // hide OK/Reset button, because we just want to show data
        me.down('toolbar[dock=bottom]').setVisible(false);
    },
});

Ext.define('PMG.SelectorList', {
    extend: 'Ext.form.ComboBox',
    xtype: 'pmgDKIMSelectorList',

    queryMode: 'local',
    store: {
        fields: ['selector'],
        filterOnLoad: true,
        proxy: {
            type: 'proxmox',
            url: '/api2/json/config/dkim/selectors',
        },
        autoLoad: true,
        sorters: [
            {
                property: 'selector',
                direction: 'ASC',
            },
        ],
    },

    valueField: 'selector',
    displayField: 'selector',
    allowBlank: false,
});

Ext.define('PMG.DKIMSettings', {
    extend: 'Proxmox.grid.ObjectGrid',
    xtype: 'pmgDKIM',

    monStoreErrors: true,

    dkimdomainTextHash: {
        envelope: gettext('Envelope'),
        header: gettext('Header'),
    },

    initComponent: function () {
        var me = this;

        me.rows = {};
        var enable_sign_text = gettext('Enable DKIM Signing');
        me.rows.dkim_sign = {
            required: true,
            defaultValue: 0,
            header: enable_sign_text,
            renderer: Proxmox.Utils.format_boolean,
            editor: {
                xtype: 'pmgDKIMEnableEdit',
                subject: enable_sign_text,
            },
        };

        var selector_text = gettext('Selector');
        me.rows.dkim_selector = {
            required: true,
            header: selector_text,
            editor: {
                xtype: 'proxmoxWindowEdit',
                subject: selector_text,
                isCreate: true,
                method: 'POST',
                url: '/config/dkim/selector',
                submitText: gettext('Update'),
                items: [
                    {
                        xtype: 'displayfield',
                        name: 'warning',
                        userCls: 'pmx-hint',
                        value: gettext(
                            'Warning: You need to update the _domainkey DNS records of all signed domains!',
                        ),
                    },
                    {
                        xtype: 'pmgDKIMSelectorList',
                        fieldLabel: selector_text,
                        name: 'selector',
                    },
                    {
                        xtype: 'proxmoxKVComboBox',
                        fieldLabel: gettext('Key Size'),
                        name: 'keysize',
                        value: '2048',
                        allowBlank: false,
                        deleteEmpty: false,
                        required: true,
                        comboItems: [
                            ['1024', '1024'],
                            ['2048', '2048'],
                            ['4096', '4096'],
                            ['8192', '8192'],
                        ],
                    },
                    {
                        xtype: 'proxmoxcheckbox',
                        name: 'force',
                        fieldLabel: gettext('Overwrite existing file'),
                    },
                ],
            },
        };

        let render_dkimdomain = (value) => me.dkimdomainTextHash[value] || value;
        me.add_combobox_row('dkim-use-domain', gettext('Signing Domain Source'), {
            renderer: render_dkimdomain,
            defaultValue: 'envelope',
            comboItems: [
                ['envelope', render_dkimdomain('envelope')],
                ['header', render_dkimdomain('header')],
            ],
        });

        me.add_boolean_row('dkim_sign_all_mail', gettext('Sign all Outgoing Mail'));

        var baseurl = '/config/admin';

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        Ext.apply(me, {
            tbar: [
                {
                    text: gettext('View DNS Record'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        var win = Ext.create('PMG.SelectorViewer', {});
                        win.load();
                        win.show();
                    },
                    selModel: me.selModel,
                },
                {
                    text: gettext('Edit'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        me.run_editor();
                    },
                    selModel: me.selModel,
                },
            ],
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_mailproxy_dkim',
            },
            interval: 5000,
            cwidth1: 200,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('PMG.DKIMDomains', {
    extend: 'PMG.RelayDomains',
    alias: ['widget.pmgDKIMDomains'],

    baseurl: '/config/dkim/domains',
    domain_desc: gettext('Sign Domain'),
    onlineHelp: 'pmgconfig_mailproxy_dkim',
});

Ext.define('PMG.MailProxyDKIMPanel', {
    extend: 'Ext.panel.Panel',
    alias: 'widget.pmgMailProxyDKIMPanel',

    layout: {
        type: 'vbox',
        align: 'stretch',
    },

    bodyPadding: '0 0 10 0',
    defaults: {
        collapsible: true,
        animCollapse: false,
        margin: '10 10 0 10',
    },

    initComponent: function () {
        var me = this;

        var DKIMSettings = Ext.create('PMG.DKIMSettings', {
            title: gettext('Settings'),
        });

        var DKIMDomains = Ext.create('PMG.DKIMDomains', {
            title: gettext('Sign Domains'),
            flex: 1,
        });

        me.items = [DKIMSettings, DKIMDomains];

        me.callParent();

        DKIMSettings.relayEvents(me, ['activate', 'deactivate', 'destroy']);
        DKIMDomains.relayEvents(me, ['activate', 'deactivate', 'destroy']);
    },
});
Ext.define('PMG.MailProxyConfiguration', {
    extend: 'Ext.tab.Panel',
    alias: 'widget.pmgMailProxyConfiguration',

    title: gettext('Configuration') + ': ' + gettext('Mail Proxy'),

    border: false,
    defaults: { border: false },

    items: [
        {
            itemId: 'relaying',
            title: gettext('Relaying'),
            xtype: 'pmgMailProxyRelaying',
        },
        {
            itemId: 'relaydomains',
            title: gettext('Relay Domains'),
            xtype: 'pmgRelayDomains',
        },
        {
            itemId: 'ports',
            title: gettext('Ports'),
            xtype: 'pmgMailProxyPorts',
        },
        {
            itemId: 'options',
            title: gettext('Options'),
            xtype: 'pmgMailProxyOptions',
        },
        {
            itemId: 'transports',
            title: gettext('Transports'),
            xtype: 'pmgTransport',
        },
        {
            itemId: 'networks',
            title: gettext('Networks'),
            xtype: 'pmgMyNetworks',
        },
        {
            itemId: 'tls',
            title: gettext('TLS'),
            xtype: 'pmgMailProxyTLSPanel',
        },
        {
            itemId: 'dkim',
            title: gettext('DKIM'),
            xtype: 'pmgMailProxyDKIMPanel',
        },
        {
            itemId: 'welcomelist',
            title: gettext('Welcomelist'),
            xtype: 'pmgObjectGroup',
            hideGroupInfo: true,
            showDirection: true,
            otype_list: [1000, 1009, 1001, 1007, 1002, 1008, 1003, 1004],
            baseurl: '/config/welcomelist',
        },
    ],
});
Ext.define('PMG.SpamDetectorLanguagesInputPanel', {
    extend: 'Proxmox.panel.InputPanel',
    alias: 'widget.pmgSpamDetectorLanguagesInputPanel',

    languages: [
        ['af', 'Afrikaans'],
        ['am', 'Amharic'],
        ['ar', 'Arabic'],
        ['be', 'Byelorussian'],
        ['bg', 'Bulgarian'],
        ['bs', 'Bosnian'],
        ['ca', 'Catalan'],
        ['cs', 'Czech'],
        ['cy', 'Welsh'],
        ['da', 'Danish'],
        ['de', 'German'],
        ['el', 'Greek'],
        ['en', 'English'],
        ['eo', 'Esperanto'],
        ['es', 'Spanish'],
        ['et', 'Estonian'],
        ['eu', 'Basque'],
        ['fa', 'Persian'],
        ['fi', 'Finnish'],
        ['fr', 'French'],
        ['fy', 'Frisian'],
        ['ga', 'Irish'],
        ['gd', 'Scottish'],
        ['he', 'Hebrew'],
        ['hi', 'Hindi'],
        ['hr', 'Croatian'],
        ['hu', 'Hungarian'],
        ['hy', 'Armenian'],
        ['id', 'Indonesian'],
        ['is', 'Icelandic'],
        ['it', 'Italian'],
        ['ja', 'Japanese'],
        ['ka', 'Georgian'],
        ['ko', 'Korean'],
        ['la', 'Latin'],
        ['lt', 'Lithuanian'],
        ['lv', 'Latvian'],
        ['mr', 'Marathi'],
        ['ms', 'Malay'],
        ['ne', 'Nepali'],
        ['nl', 'Dutch'],
        ['no', 'Norwegian'],
        ['pl', 'Polish'],
        ['pt', 'Portuguese'],
        ['qu', 'Quechua'],
        ['Rhaeto', 'Romance'],
        ['ro', 'Romanian'],
        ['ru', 'Russian'],
        ['sa', 'Sanskrit'],
        ['sco', 'Scots'],
        ['sk', 'Slovak'],
        ['sl', 'Slovenian'],
        ['sq', 'Albanian'],
        ['sr', 'Serbian'],
        ['sv', 'Swedish'],
        ['sw', 'Swahili'],
        ['ta', 'Tamil'],
        ['th', 'Thai'],
        ['tl', 'Tagalog'],
        ['tr', 'Turkish'],
        ['uk', 'Ukrainian'],
        ['vi', 'Vietnamese'],
        ['yi', 'Yiddish'],
        ['zh', 'Chinese'],
    ],

    onGetValues: function (values) {
        if (!values.languages) {
            values.delete = 'languages';
        } else if (Ext.isArray(values.languages)) {
            values.languages = values.languages.join(' ');
        }

        return values;
    },

    initComponent: function () {
        var me = this;

        me.column1 = [];
        me.column2 = [];
        me.column3 = [];
        me.column4 = [];

        var i, len;
        for (i = 0, len = me.languages.length; i < len; i++) {
            let config = {
                xtype: 'checkboxfield',
                inputValue: me.languages[i][0],
                boxLabel: me.languages[i][1],
                name: 'languages',
            };
            if (i % 4 === 0) {
                me.column1.push(config);
            } else if (i % 4 === 1) {
                me.column2.push(config);
            } else if (i % 4 === 2) {
                me.column3.push(config);
            } else if (i % 4 === 3) {
                me.column4.push(config);
            }
        }

        me.callParent();
    },
});

Ext.define('PMG.SpamDetectorLanguages', {
    extend: 'Proxmox.window.Edit',
    onlineHelp: 'pmgconfig_spamdetector',

    subject: 'Languages',

    items: [
        {
            xtype: 'pmgSpamDetectorLanguagesInputPanel',
        },
    ],

    initComponent: function () {
        let me = this;

        me.callParent();

        me.load({
            success: function (response, options) {
                let value = response.result.data.languages || '';
                let languages = value.split(/[ ,;]+/);
                me.setValues({ languages: languages });
            },
        });
    },
});
Ext.define('PMG.SpamDetectorOptions', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgSpamDetectorOptions'],

    monStoreErrors: true,

    initComponent: function () {
        var me = this;

        me.add_boolean_row('use_awl', gettext('Use auto-welcomelists'), {
            defaultValue: 0,
        });

        me.add_boolean_row('use_bayes', gettext('Use Bayesian filter'), {
            defaultValue: 0,
        });

        me.add_boolean_row('rbl_checks', gettext('Use RBL checks'), {
            defaultValue: 1,
        });

        me.add_boolean_row('use_razor', gettext('Use Razor2 checks'), {
            defaultValue: 1,
        });

        me.add_boolean_row('extract_text', gettext('Extract Text from Attachments'));

        me.add_integer_row('maxspamsize', gettext('Max Spam Size (bytes)'), {
            defaultValue: 256 * 1024,
            minValue: 64,
            deleteEmpty: true,
        });

        me.rows.languages = {
            required: true,
            header: gettext('Languages'),
            editor: 'PMG.SpamDetectorLanguages',
            renderer: (value) => value || 'all',
        };

        me.add_integer_row('bounce_score', gettext('Backscatter Score'), {
            defaultValue: 0,
            minValue: 0,
            maxValue: 1000,
            deleteEmpty: true,
        });

        me.add_integer_row('clamav_heuristic_score', gettext('Heuristic Score'), {
            defaultValue: 3,
            minValue: 0,
            maxValue: 1000,
            deleteEmpty: true,
        });

        var baseurl = '/config/spam';

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        Ext.apply(me, {
            tbar: [
                {
                    text: gettext('Edit'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        me.run_editor();
                    },
                    selModel: me.selModel,
                },
            ],
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_spamdetector',
            },
            interval: 5000,
            cwidth1: 200,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('PMG.SpamQuarantineOptions', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgSpamQuarantineOptions'],

    monStoreErrors: true,

    authmodeTextHash: {
        ticket: 'Ticket',
        ldap: 'LDAP',
        ldapticket: 'LDAP or Ticket',
    },

    reportstyleTextHash: {
        none: gettext('No Reports'),
        short: gettext('Short'),
        verbose: gettext('Verbose'),
        custom: gettext('Custom'),
    },

    initComponent: function () {
        let me = this;

        me.add_integer_row('lifetime', gettext('Lifetime (days)'), {
            minValue: 1,
            defaultValue: 7,
            deleteEmpty: true,
        });

        let render_authmode = (value) => me.authmodeTextHash[value] || value;
        me.add_combobox_row('authmode', gettext('Authentication mode'), {
            defaultValue: 'ticket',
            renderer: render_authmode,
            comboItems: [
                ['ticket', render_authmode('ticket')],
                ['ldap', render_authmode('ldap')],
                ['ldapticket', render_authmode('ldapticket')],
            ],
        });

        let render_reportstyle = (value) => me.reportstyleTextHash[value] || value;
        me.add_combobox_row('reportstyle', gettext('User Spamreport Style'), {
            defaultValue: 'verbose',
            renderer: render_reportstyle,
            comboItems: [
                ['none', render_reportstyle('none')],
                ['short', render_reportstyle('short')],
                ['verbose', render_reportstyle('verbose')],
                ['custom', render_reportstyle('custom')],
            ],
        });

        me.add_text_row('hostname', gettext('Quarantine Host'), {
            deleteEmpty: true,
            defaultValue: Proxmox.Utils.noneText,
            renderer: Ext.htmlEncode,
        });
        me.add_integer_row('port', gettext('Quarantine port'), {
            deleteEmpty: true,
            defaultValue: Proxmox.Utils.defaultText,
        });
        me.add_text_row('mailfrom', gettext("EMail 'From:'"), {
            deleteEmpty: true,
            defaultValue: Proxmox.Utils.noneText,
            renderer: Ext.htmlEncode,
        });
        me.add_boolean_row('viewimages', gettext('View images'), {
            defaultValue: 1,
        });
        me.add_boolean_row('allowhrefs', gettext('Allow HREFs'), {
            defaultValue: 1,
        });

        let baseurl = '/config/spamquar';

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        Ext.apply(me, {
            tbar: [
                {
                    text: gettext('Edit'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        me.run_editor();
                    },
                    selModel: me.selModel,
                },
            ],
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_spamdetector_quarantine',
            },
            interval: 5000,
            cwidth1: 200,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('pmg-spamassassin-database', {
    extend: 'Ext.data.Model',
    fields: [
        'channel',
        'version',
        'update_version',
        { name: 'update_avail', type: 'boolean' },
        { name: 'last_updated', type: 'date', dateFormat: 'timestamp' },
    ],
    idProperty: 'channel',
});

Ext.define('PMG.SpamDetectorStatusGrid', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgSpamDetectorStatus',

    title: gettext('Status'),

    viewConfig: {
        trackOver: false,
    },
    columns: [
        {
            header: gettext('Channel'),
            sortable: true,
            flex: 1,
            dataIndex: 'channel',
        },
        {
            header: gettext('Last Update'),
            sortable: true,
            flex: 2,
            dataIndex: 'last_updated',
        },
        {
            header: gettext('Version'),
            flex: 1,
            sortable: true,
            dataIndex: 'version',
        },
        {
            header: gettext('Update Available'),
            flex: 1,
            sortable: true,
            dataIndex: 'update_avail',
            renderer: function (value, metaData, record) {
                if (!value) {
                    return Proxmox.Utils.noText;
                } else {
                    return Proxmox.Utils.yesText + ' (' + record.data.update_version + ')';
                }
            },
        },
    ],

    listeners: {
        activate: function () {
            var me = this;
            me.store.load();
        },
    },

    tbar: [
        {
            text: gettext('Update Now'),
            handler: function () {
                let view = this.up('grid');
                Proxmox.Utils.API2Request({
                    url: '/nodes/' + Proxmox.NodeName + '/spamassassin/rules',
                    method: 'POST',
                    failure: function (response) {
                        Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                    },
                    success: function (response) {
                        const upid = response.result.data;

                        let win = Ext.create('Proxmox.window.TaskViewer', {
                            upid: upid,
                            autoShow: true,
                        });
                        view.mon(win, 'close', () => view.store.load());
                    },
                });
            },
        },
    ],

    initComponent: function () {
        var me = this;

        me.store = Ext.create('Ext.data.Store', {
            model: 'pmg-spamassassin-database',
            proxy: {
                type: 'proxmox',
                url: '/api2/json/nodes/' + Proxmox.NodeName + '/spamassassin/rules',
            },
            sorters: {
                property: 'name',
                direction: 'ASC',
            },
        });

        me.callParent();

        Proxmox.Utils.monStoreErrors(me.getView(), me.store, true);
    },
});
Ext.define('PMG.SpamDetectorConfiguration', {
    extend: 'Ext.tab.Panel',
    alias: 'widget.pmgSpamDetectorConfiguration',

    title: gettext('Configuration') + ': ' + gettext('Spam Detector'),

    border: false,
    defaults: { border: false },

    items: [
        {
            title: gettext('Options'),
            itemId: 'options',
            xtype: 'pmgSpamDetectorOptions',
        },
        {
            title: gettext('Quarantine'),
            itemId: 'quarantine',
            xtype: 'pmgSpamQuarantineOptions',
        },
        {
            title: gettext('Status'),
            itemId: 'status',
            xtype: 'pmgSpamDetectorStatus',
        },
        {
            title: gettext('Custom Scores'),
            itemId: 'scores',
            xtype: 'pmgSpamDetectorCustomScores',
        },
    ],
});
Ext.define('pmg-sa-custom', {
    extend: 'Ext.data.Model',
    fields: ['name', 'score', 'comment', 'digest'],
    idProperty: 'name',
});

Ext.define('PMG.SpamDetectorCustomScores', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgSpamDetectorCustomScores',

    layout: 'border',

    viewModel: {
        data: {
            applied: true,
            changetext: '',
            digest: null,
        },
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        reload: function () {
            let me = this;
            let vm = me.getViewModel();
            let grid = me.lookup('grid');

            Proxmox.Utils.API2Request({
                url: '/config/customscores',
                failure: function (response, opts) {
                    grid.getStore().loadData({});
                    Proxmox.Utils.setErrorMask(grid, response.htmlStatus);
                    vm.set('digest', null);
                    vm.set('applied', true);
                    vm.set('changetext', '');
                },
                success: function (response, opts) {
                    let data = response.result.data;
                    let digestel = data.pop(); // last element is digest
                    let changes = response.result.changes;
                    grid.getStore().loadData(data);

                    vm.set('digest', digestel.digest);
                    vm.set('applied', !changes);
                    vm.set('changetext', `<pre>${changes || ''}</pre>`);
                },
            });
        },

        revert: function () {
            let me = this;
            let vm = me.getViewModel();
            let grid = me.lookup('grid');

            Proxmox.Utils.API2Request({
                url: '/config/customscores',
                method: 'DELETE',
                param: {
                    digest: vm.get('digest'),
                },
                failure: function (response, opts) {
                    grid.getStore().loadData({});
                    Proxmox.Utils.setErrorMask(grid, response.htmlStatus);
                    vm.set('digest', null);
                    vm.set('applied', true);
                    vm.set('changetext', '');
                },
                success: () => {
                    me.reload();
                },
            });
        },

        restart: function () {
            var me = this;
            var vm = this.getViewModel();

            Ext.createWidget('proxmoxWindowEdit', {
                method: 'PUT',
                url: '/api2/extjs/config/customscores',
                isCreate: true,
                submitText: gettext('Apply'),
                showProgress: true,
                taskDone: () => {
                    me.reload();
                },

                title: gettext('Apply Custom Scores'),
                onlineHelp: 'pmgconfig_spamdetector_customscores',

                items: [
                    {
                        xtype: 'proxmoxcheckbox',
                        name: 'restart-daemon',
                        boxLabel: gettext('Restart pmg-smtp-filter to activate changes.'),
                        labelWidth: 150,
                        checked: true,
                    },
                    {
                        xtype: 'hiddenfield',
                        name: 'digest',
                        value: vm.get('digest'),
                    },
                ],
            }).show();
        },

        create_custom: function () {
            var me = this;
            var vm = this.getViewModel();

            Ext.createWidget('proxmoxWindowEdit', {
                autoShow: true,
                method: 'POST',
                url: '/api2/extjs/config/customscores',
                isCreate: true,
                subject: gettext('Custom Rule Score'),
                onlineHelp: 'pmgconfig_spamdetector_customscores',
                items: [
                    {
                        xtype: 'proxmoxtextfield',
                        name: 'name',
                        allowBlank: false,
                        fieldLabel: gettext('Name'),
                    },
                    {
                        xtype: 'numberfield',
                        name: 'score',
                        allowBlank: false,
                        fieldLabel: gettext('Score'),
                    },

                    {
                        xtype: 'proxmoxtextfield',
                        name: 'comment',
                        fieldLabel: gettext('Comment'),
                    },
                    {
                        xtype: 'hiddenfield',
                        name: 'digest',
                        value: vm.get('digest'),
                    },
                ],
                listeners: {
                    destroy: () => me.reload(),
                },
            });
        },

        run_editor: function () {
            let me = this;
            let vm = me.getViewModel();
            let grid = me.lookup('grid');
            let rec = grid.getSelection()[0];
            if (!rec) {
                return;
            }

            Ext.createWidget('proxmoxWindowEdit', {
                autoShow: true,
                autoLoad: true,
                url: '/api2/extjs/config/customscores/' + rec.data.name,
                method: 'PUT',
                subject: gettext('Custom Rule Score'),
                onlineHelp: 'pmgconfig_spamdetector_customscores',
                items: [
                    {
                        xtype: 'displayfield',
                        name: 'name',
                        fieldLabel: gettext('Name'),
                    },
                    {
                        xtype: 'numberfield',
                        name: 'score',
                        allowBlank: false,
                        fieldLabel: gettext('Score'),
                    },

                    {
                        xtype: 'proxmoxtextfield',
                        name: 'comment',
                        fieldLabel: gettext('Comment'),
                    },
                    {
                        xtype: 'hiddenfield',
                        name: 'digest',
                        value: vm.get('digest'),
                    },
                ],
                listeners: {
                    destroy: () => me.reload(),
                },
            });
        },
    },

    listeners: {
        activate: 'reload',
    },

    defaults: {
        border: 0,
    },

    items: [
        {
            xtype: 'gridpanel',
            region: 'center',
            reference: 'grid',

            store: {
                model: 'pmg-sa-custom',
                proxy: {
                    type: 'proxmox',
                    url: '/api2/json/config/customscores',
                },
                sorters: {
                    property: 'name',
                },
            },

            tbar: [
                {
                    text: gettext('Create'),
                    handler: 'create_custom',
                },
                '-',
                {
                    xtype: 'proxmoxButton',
                    text: gettext('Edit'),
                    disabled: true,
                    handler: 'run_editor',
                },
                {
                    xtype: 'proxmoxStdRemoveButton',
                    getUrl: function (rec) {
                        let digest = this.up('grid').digest;
                        let url = `/config/customscores/${rec.getId()}`;
                        if (digest) {
                            url += `?digest=${digest}`;
                        }
                        return url;
                    },
                    callback: 'reload',
                },
            ],

            viewConfig: {
                trackOver: false,
            },

            columns: [
                {
                    header: gettext('Name'),
                    width: 200,
                    sortable: true,
                    dataIndex: 'name',
                },
                {
                    header: gettext('Score'),
                    width: 200,
                    sortable: true,
                    dataIndex: 'score',
                },
                {
                    header: gettext('Comment'),
                    sortable: false,
                    renderer: Ext.String.htmlEncode,
                    dataIndex: 'comment',
                    flex: 1,
                },
            ],

            listeners: {
                itemdblclick: 'run_editor',
            },
        },
        {
            xtype: 'panel',
            bodyPadding: 5,
            region: 'south',
            autoScroll: true,
            flex: 0.5,
            hidden: true,
            bind: {
                hidden: '{applied}',
                html: '{changetext}',
            },
            reference: 'changes',
            tbar: [
                {
                    text: gettext('Revert'),
                    handler: 'revert',
                    disabled: true,
                    bind: {
                        disabled: '{applied}',
                    },
                },
                '-',
                {
                    text: gettext('Apply Custom Scores'),
                    handler: 'restart',
                    disabled: true,
                    bind: {
                        disabled: '{applied}',
                    },
                },
                '->',
                `<b style="font-weight: 600">${gettext('Pending changes')}</b>`,
                '->',
            ],
            split: true,
        },
    ],
});
Ext.define('PMG.VirusDetectorOptions', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgVirusDetectorOptions'],

    monStoreErrors: true,

    initComponent: function () {
        var me = this;

        me.add_boolean_row(
            'archiveblockencrypted',
            gettext('Block encrypted archives and documents'),
        );

        me.add_integer_row('archivemaxrec', gettext('Max recursion'), {
            minValue: 1,
            defaultValue: 5,
            deleteEmpty: true,
        });

        me.add_integer_row('archivemaxfiles', gettext('Max files'), {
            minValue: 0,
            defaultValue: 1000,
            deleteEmpty: true,
        });

        me.add_integer_row('archivemaxsize', gettext('Max file size'), {
            minValue: 1000 * 1000,
            defaultValue: 25 * 1000 * 1000,
            deleteEmpty: true,
        });

        me.add_integer_row('maxscansize', gettext('Max scan size'), {
            minValue: 1000 * 1000,
            defaultValue: 100 * 1000 * 1000,
            deleteEmpty: true,
        });

        me.add_integer_row('maxcccount', gettext('Max credit card numbers'), {
            minValue: 0,
            defaultValue: 0,
            deleteEmpty: true,
        });

        var baseurl = '/config/clamav';

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        Ext.apply(me, {
            tbar: [
                {
                    text: gettext('Edit'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        me.run_editor();
                    },
                    selModel: me.selModel,
                },
            ],
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_clamav_options',
            },
            interval: 5000,
            cwidth1: 270,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('PMG.VirusQuarantineOptions', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgVirusQuarantineOptions'],

    monStoreErrors: true,

    initComponent: function () {
        var me = this;

        me.add_integer_row('lifetime', gettext('Lifetime (days)'), {
            minValue: 1,
            defaultValue: 7,
            deleteEmpty: true,
        });

        me.add_boolean_row('viewimages', gettext('View images'), {
            defaultValue: 1,
        });

        me.add_boolean_row('allowhrefs', gettext('Allow HREFs'), {
            defaultValue: 1,
        });

        var baseurl = '/config/virusquar';

        me.selModel = Ext.create('Ext.selection.RowModel', {});

        Ext.apply(me, {
            tbar: [
                {
                    text: gettext('Edit'),
                    xtype: 'proxmoxButton',
                    disabled: true,
                    handler: function () {
                        me.run_editor();
                    },
                    selModel: me.selModel,
                },
            ],
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_clamav_quarantine',
            },
            interval: 5000,
            cwidth1: 200,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});
Ext.define('pmg-virus-list', {
    extend: 'Ext.data.Model',
    fields: [
        'id',
        'envelope_sender',
        'from',
        'sender',
        'receiver',
        'subject',
        { type: 'integer', name: 'bytes' },
        { type: 'string', name: 'virusname' },
        { type: 'date', dateFormat: 'timestamp', name: 'time' },
        {
            type: 'string',
            name: 'day',
            convert: function (v, rec) {
                return Ext.Date.format(rec.get('time'), 'Y-m-d');
            },
            depends: ['time'],
        },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/quarantine/virus',
    },
    idProperty: 'id',
});

Ext.define('PMG.VirusQuarantine', {
    extend: 'Ext.container.Container',
    xtype: 'pmgVirusQuarantine',

    border: false,
    layout: { type: 'border' },

    defaults: { border: false },

    viewModel: {
        parent: null,
        data: {
            mailid: '',
        },
        formulas: {
            downloadMailURL: (get) =>
                '/api2/json/quarantine/download?mailid=' + encodeURIComponent(get('mailid')),
        },
    },
    controller: 'quarantine',

    items: [
        {
            title: gettext('Virus Quarantine'),
            xtype: 'pmgQuarantineList',
            emptyText: gettext('No data in database'),
            selModel: 'checkboxmodel',
            quarantineType: 'virus',
            reference: 'list',
            region: 'west',
            width: 500,
            split: true,
            collapsible: false,
            store: {
                model: 'pmg-virus-list',
                groupField: 'day',
                groupDir: 'DESC',
                sorters: [
                    {
                        property: 'time',
                        direction: 'DESC',
                    },
                ],
            },

            columns: [
                {
                    header: `${gettext('Sender')}/${gettext('Receiver')}/${gettext('Subject')}`,
                    dataIndex: 'subject',
                    renderer: PMG.Utils.render_sender_receiver,
                    flex: 1,
                },
                {
                    header: gettext('Virus'),
                    dataIndex: 'virusname',
                    align: 'right',
                    width: 70,
                },
                {
                    header: gettext('Size') + ' (KB)',
                    renderer: (v) => Ext.Number.toFixed(v / 1024, 0),
                    dataIndex: 'bytes',
                    align: 'right',
                    width: 90,
                },
                {
                    header: gettext('Date'),
                    dataIndex: 'day',
                    hidden: true,
                },
                {
                    xtype: 'datecolumn',
                    header: gettext('Time'),
                    dataIndex: 'time',
                    format: 'H:i:s',
                },
            ],
        },
        {
            title: gettext('Selected Mail'),
            border: false,
            region: 'center',
            layout: 'fit',
            split: true,
            reference: 'preview',
            disabled: true,
            dockedItems: [
                {
                    xtype: 'toolbar',
                    dock: 'top',
                    overflowHandler: 'scroller',
                    style: {
                        // docked items have set the bottom with to 0px with '! important'
                        // but we still want one here, so we can remove the borders of the grids
                        'border-bottom-width': '1px ! important',
                    },
                    items: [
                        {
                            xtype: 'button',
                            reference: 'raw',
                            text: gettext('Toggle Raw'),
                            enableToggle: true,
                            iconCls: 'fa fa-file-code-o',
                        },
                        {
                            xtype: 'tbseparator',
                            reference: 'themeCheckSep',
                        },
                        {
                            xtype: 'proxmoxcheckbox',
                            reference: 'themeCheck',
                            checked: true,
                            boxLabel: gettext('Dark-mode filter'),
                            iconCls: 'fa fa-paint-brush',
                        },
                        '->',
                        {
                            xtype: 'button',
                            reference: 'download',
                            text: gettext('Download'),
                            setDownload: function (id) {
                                this.el.dom.download = id + '.eml';
                            },
                            bind: {
                                href: '{downloadMailURL}',
                                download: '{mailid}',
                            },
                            iconCls: 'fa fa-download',
                        },
                        {
                            reference: 'deliver',
                            text: gettext('Deliver'),
                            iconCls: 'fa fa-paper-plane-o info-blue',
                            handler: 'btnHandler',
                        },
                        {
                            reference: 'delete',
                            text: gettext('Delete'),
                            iconCls: 'fa fa-trash-o critical',
                            handler: 'btnHandler',
                        },
                    ],
                },
                {
                    xtype: 'pmgMailInfo',
                    hidden: true,
                    reference: 'mailinfo',
                    border: false,
                },
                {
                    xtype: 'pmgAttachmentGrid',
                    reference: 'attachmentlist',
                    showDownloads: false,
                    border: false,
                    dock: 'bottom',
                },
            ],
        },
    ],
});
Ext.define('pmg-attachment-list', {
    extend: 'Ext.data.Model',
    fields: [
        'id',
        'envelope_sender',
        'from',
        'sender',
        'receiver',
        'subject',
        { type: 'integer', name: 'bytes' },
        { type: 'date', dateFormat: 'timestamp', name: 'time' },
        {
            type: 'string',
            name: 'day',
            convert: function (v, rec) {
                return Ext.Date.format(rec.get('time'), 'Y-m-d');
            },
            depends: ['time'],
        },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/quarantine/attachment',
    },
    idProperty: 'id',
});

Ext.define('PMG.AttachmentQuarantine', {
    extend: 'Ext.container.Container',
    xtype: 'pmgAttachmentQuarantine',

    border: false,
    layout: { type: 'border' },

    defaults: { border: false },

    viewModel: {
        parent: null,
        data: {
            mailid: '',
        },
        formulas: {
            downloadMailURL: (get) =>
                '/api2/json/quarantine/download?mailid=' + encodeURIComponent(get('mailid')),
        },
    },
    controller: 'quarantine',
    items: [
        {
            title: gettext('Attachment Quarantine'),
            xtype: 'pmgQuarantineList',
            emptyText: gettext('No data in database'),
            selModel: 'checkboxmodel',
            quarantineType: 'attachment',
            reference: 'list',
            region: 'west',
            width: 500,
            split: true,
            collapsible: false,
            store: {
                model: 'pmg-attachment-list',
                groupField: 'day',
                groupDir: 'DESC',
                sorters: [
                    {
                        property: 'time',
                        direction: 'DESC',
                    },
                ],
            },

            columns: [
                {
                    header: `${gettext('Sender')}/${gettext('Receiver')}/${gettext('Subject')}`,
                    dataIndex: 'subject',
                    renderer: PMG.Utils.render_sender_receiver,
                    flex: 1,
                },
                {
                    header: gettext('Size') + ' (KB)',
                    renderer: (v) => Ext.Number.toFixed(v / 1024, 0),
                    dataIndex: 'bytes',
                    align: 'right',
                    width: 90,
                },
                {
                    header: gettext('Date'),
                    dataIndex: 'day',
                    hidden: true,
                },
                {
                    xtype: 'datecolumn',
                    header: gettext('Time'),
                    dataIndex: 'time',
                    format: 'H:i:s',
                },
            ],
        },
        {
            title: gettext('Selected Mail'),
            border: false,
            region: 'center',
            layout: 'fit',
            split: true,
            reference: 'preview',
            disabled: true,
            dockedItems: [
                {
                    xtype: 'toolbar',
                    overflowHandler: 'scroller',
                    dock: 'top',
                    items: [
                        {
                            xtype: 'button',
                            reference: 'raw',
                            text: gettext('Toggle Raw'),
                            enableToggle: true,
                            iconCls: 'fa fa-file-code-o',
                        },
                        {
                            xtype: 'tbseparator',
                            reference: 'themeCheckSep',
                        },
                        {
                            xtype: 'proxmoxcheckbox',
                            reference: 'themeCheck',
                            checked: true,
                            boxLabel: gettext('Dark-mode filter'),
                            iconCls: 'fa fa-paint-brush',
                        },
                        '->',
                        {
                            xtype: 'button',
                            reference: 'download',
                            text: gettext('Download'),
                            setDownload: function (id) {
                                this.el.dom.download = id + '.eml';
                            },
                            bind: {
                                href: '{downloadMailURL}',
                                download: '{mailid}',
                            },
                            iconCls: 'fa fa-download',
                        },
                        {
                            reference: 'deliver',
                            text: gettext('Deliver'),
                            iconCls: 'fa fa-paper-plane-o info-blue',
                            handler: 'btnHandler',
                        },
                        {
                            reference: 'delete',
                            text: gettext('Delete'),
                            iconCls: 'fa fa-trash-o critical',
                            handler: 'btnHandler',
                        },
                    ],
                },
                {
                    xtype: 'pmgMailInfo',
                    hidden: true,
                    reference: 'mailinfo',
                },
                {
                    xtype: 'pmgAttachmentGrid',
                    reference: 'attachmentlist',
                    dock: 'bottom',
                    collapsible: false,
                },
            ],
        },
    ],
});
Ext.define('PMG.grid.AttachmentGrid', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgAttachmentGrid',
    mixins: ['Proxmox.Mixin.CBind'],

    showDownloads: true,

    title: gettext('Attachments'),
    iconCls: 'fa fa-paperclip',

    minHeight: 50,
    maxHeight: 250,
    scrollable: true,

    collapsible: true,
    titleCollapse: true,

    tbar: [
        '->',
        {
            xtype: 'checkbox',
            boxLabel: gettext('Show All Parts'),
            boxLabelAlgign: 'before',
            listeners: {
                change: function (checkBox, value) {
                    let store = this.up('pmgAttachmentGrid').getStore();
                    store.clearFilter();
                    if (!value) {
                        store.filter({
                            property: 'content-disposition',
                            value: 'attachment',
                        });
                    }
                },
            },
        },
    ],

    store: {
        autoDestroy: true,
        fields: ['name', 'content-type', 'size', 'content-disposition'],
        proxy: {
            type: 'proxmox',
        },
        filters: {
            property: 'content-disposition',
            value: 'attachment',
        },
    },

    controller: {
        xclass: 'Ext.app.ViewController',
        init: function (view) {
            view.store.on('load', this.onLoad, this);
        },
        onLoad: function (store, records, success) {
            let me = this;
            let view = me.getView();
            if (!success) {
                view.updateTitleStats(-1);
                return;
            }
            let attachments = records.filter(
                ({ data }) => data['content-disposition'] === 'attachment',
            );
            let totalSize = attachments.reduce((sum, { data }) => sum + data.size, 0);
            view.updateTitleStats(attachments.length, totalSize);
        },
    },

    updateTitleStats: function (count, totalSize) {
        let me = this;
        let title;
        if (count > 0) {
            title = Ext.String.format(gettext('{0} Attachments'), count);
            title += ` (${Proxmox.Utils.format_size(totalSize)})`;
            if (me.collapsible) {
                me.expand();
            }
        } else {
            title = gettext('No Attachments');
            if (me.collapsible) {
                me.collapse();
            }
        }
        me.setTitle(title);
    },

    setID: function (rec) {
        var me = this;
        if (!rec || !rec.data || !rec.data.id) {
            me.getStore().removeAll();
            return;
        }
        var url = '/api2/json/quarantine/listattachments?id=' + rec.data.id;
        me.mailid = rec.data.id;
        me.store.proxy.setUrl(url);
        me.store.load();
    },

    emptyText: gettext('No Attachments'),

    download: function () {
        Ext.Msg.alert(arguments);
    },

    columns: [
        {
            text: gettext('Filename'),
            dataIndex: 'name',
            flex: 1,
        },
        {
            text: gettext('Filetype'),
            dataIndex: 'content-type',
            renderer: PMG.Utils.render_filetype,
            flex: 1,
        },
        {
            text: gettext('Size'),
            renderer: Proxmox.Utils.render_size,
            dataIndex: 'size',
            flex: 1,
        },
        {
            header: gettext('Download'),
            cbind: {
                hidden: '{!showDownloads}',
            },
            renderer: function (value, mD, rec) {
                var me = this;
                let url = `/api2/json/quarantine/download?mailid=${me.mailid}&attachmentid=${rec.data.id}`;
                return `<a target='_blank' class='download' download='${rec.data.name}' href='${url}'>
		    <i class='fa fa-fw fa-download'</i>
		</a>`;
            },
        },
    ],
});
Ext.define('PMG.ClamAVDatabaseConfig', {
    extend: 'Proxmox.grid.ObjectGrid',
    alias: ['widget.pmgClamAVDatabaseConfig'],

    monStoreErrors: true,

    initComponent: function () {
        var me = this;

        me.add_text_row('dbmirror', gettext('Database Mirror'), {
            deleteEmpty: true,
            defaultValue: 'database.clamav.net',
            renderer: Ext.htmlEncode,
        });

        me.add_boolean_row('scriptedupdates', gettext('Incremental Download'), {
            defaultValue: 0,
        });

        var baseurl = '/config/clamav';

        Ext.apply(me, {
            url: '/api2/json' + baseurl,
            editorConfig: {
                url: '/api2/extjs' + baseurl,
                onlineHelp: 'pmgconfig_clamav',
            },
            interval: 5000,
            cwidth1: 200,
            listeners: {
                itemdblclick: me.run_editor,
            },
        });

        me.callParent();

        me.on('activate', me.rstore.startUpdate);
        me.on('destroy', me.rstore.stopUpdate);
        me.on('deactivate', me.rstore.stopUpdate);
    },
});

Ext.define('pmg-clamav-database', {
    extend: 'Ext.data.Model',
    fields: ['name', 'type', 'build_time', 'version', { name: 'nsigs', type: 'integer' }],
    idProperty: 'name',
});

Ext.define('PMG.ClamAVDatabaseStatus', {
    extend: 'Ext.grid.GridPanel',
    alias: ['widget.pmgClamAVDatabaseStatus'],

    title: gettext('Status'),

    reload: function () {
        var me = this;

        me.store.load();
    },

    initComponent: function () {
        var me = this;

        me.store = new Ext.data.Store({
            model: 'pmg-clamav-database',
            proxy: {
                type: 'proxmox',
                url: '/api2/json/nodes/' + Proxmox.NodeName + '/clamav/database',
            },
            sorters: {
                property: 'name',
                direction: 'ASC',
            },
        });

        Ext.apply(me, {
            viewConfig: {
                trackOver: false,
            },
            columns: [
                {
                    header: gettext('Name'),
                    sortable: true,
                    flex: 1,
                    dataIndex: 'name',
                },
                {
                    header: gettext('Build time'),
                    sortable: true,
                    flex: 2,
                    dataIndex: 'build_time',
                },
                {
                    header: gettext('Version'),
                    flex: 1,
                    sortable: true,
                    dataIndex: 'version',
                },
                {
                    header: gettext('Signatures'),
                    flex: 1,
                    sortable: true,
                    dataIndex: 'nsigs',
                },
            ],
            listeners: {
                activate: me.reload,
            },
        });

        me.callParent();

        Proxmox.Utils.monStoreErrors(me.getView(), me.store, true);
    },
});

Ext.define('PMG.ClamAVDatabase', {
    extend: 'Ext.panel.Panel',
    alias: ['widget.pmgClamAVDatabase'],

    layout: { type: 'vbox', align: 'stretch' },

    initComponent: function () {
        var me = this;

        var selModel = Ext.create('Ext.selection.RowModel', {});
        var editPanel = Ext.create('PMG.ClamAVDatabaseConfig', {
            border: false,
            xtype: 'pmgClamAVDatabaseConfig',
            selModel: selModel,
        });

        var statusPanel = Ext.create('PMG.ClamAVDatabaseStatus', {
            border: false,
            flex: 1,
        });

        var update_command = function () {
            Proxmox.Utils.API2Request({
                url: '/nodes/' + Proxmox.NodeName + '/clamav/database',
                method: 'POST',
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
                success: function (response, opts) {
                    var upid = response.result.data;

                    var win = Ext.create('Proxmox.window.TaskViewer', {
                        upid: upid,
                    });
                    win.show();
                    me.mon(win, 'close', function () {
                        statusPanel.reload();
                    });
                },
            });
        };

        me.tbar = [
            {
                text: gettext('Edit'),
                xtype: 'proxmoxButton',
                disabled: true,
                handler: function () {
                    editPanel.run_editor();
                },
                selModel: selModel,
            },
            {
                text: gettext('Update now'),
                handler: update_command,
            },
        ];

        me.items = [editPanel, statusPanel];

        me.callParent();

        editPanel.relayEvents(me, ['activate', 'deactivate', 'destroy']);
        statusPanel.relayEvents(me, ['activate', 'deactivate', 'destroy']);
    },
});
Ext.define('PMG.VirusDetectorConfiguration', {
    extend: 'Ext.tab.Panel',
    alias: 'widget.pmgVirusDetectorConfiguration',

    title: gettext('Configuration') + ': ' + gettext('Virus Detector'),

    border: false,
    defaults: { border: false },

    items: [
        {
            title: gettext('Options'),
            itemId: 'options',
            xtype: 'pmgVirusDetectorOptions',
        },
        {
            title: gettext('ClamAV'),
            itemId: 'clamav',
            xtype: 'pmgClamAVDatabase',
        },
        {
            title: gettext('Quarantine'),
            itemId: 'quarantine',
            xtype: 'pmgVirusQuarantineOptions',
        },
    ],
});
Ext.define('pmg-ldap-config', {
    extend: 'Ext.data.Model',
    fields: [
        'profile',
        'server1',
        'server2',
        'comment',
        'mode',
        'binddn',
        'bindpw',
        'basedn',
        'groupbasedn',
        'filter',
        'accountattr',
        'mailattr',
        { name: 'port', type: 'integer' },
        { name: 'gcount', type: 'integer' },
        { name: 'mcount', type: 'integer' },
        { name: 'ucount', type: 'integer' },
        { name: 'disable', type: 'boolean' },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/config/ldap',
    },
    idProperty: 'profile',
});

Ext.define('PMG.LDAPInputPanel', {
    extend: 'Proxmox.panel.InputPanel',
    alias: 'widget.pmgLDAPInputPanel',

    profileId: undefined,

    onGetValues: function (values) {
        var _me = this;

        values.disable = values.enable ? 0 : 1;
        delete values.enable;

        // do not send empty password
        if (values.bindpw === '') {
            delete values.bindpw;
        }

        return values;
    },

    initComponent: function () {
        var me = this;

        me.column1 = [
            {
                xtype: me.profileId ? 'displayfield' : 'textfield',
                fieldLabel: gettext('Profile Name'),
                value: me.profileId || '',
                name: 'profile',
                vtype: 'StorageId',
                allowBlank: false,
            },
            {
                xtype: 'proxmoxKVComboBox',
                name: 'mode',
                comboItems: [
                    ['ldap', PMG.Utils.format_ldap_protocol('ldap')],
                    ['ldaps', PMG.Utils.format_ldap_protocol('ldaps')],
                    ['ldap+starttls', PMG.Utils.format_ldap_protocol('ldap+starttls')],
                ],
                value: 'ldap',
                fieldLabel: gettext('Protocol'),
                listeners: {
                    change: function (cb, value) {
                        var isldap = value === 'ldap';
                        me.down('field[name=verify]').setVisible(!isldap);
                    },
                },
            },
            {
                xtype: 'proxmoxcheckbox',
                name: 'verify',
                fieldLabel: gettext('Verify Certificate'),
                hidden: true,
                uncheckedValue: 0,
                value: 1,
                checked: 1,
            },
            {
                xtype: 'textfield',
                fieldLabel: gettext('Server'),
                allowBlank: false,
                vtype: 'DnsOrIp',
                name: 'server1',
            },
            {
                xtype: 'proxmoxtextfield',
                fieldLabel: gettext('Server'),
                allowBlank: true,
                deleteEmpty: !me.isCreate,
                vtype: 'DnsOrIp',
                name: 'server2',
            },
            {
                xtype: 'proxmoxintegerfield',
                name: 'port',
                emptyText: gettext('Default'),
                deleteEmpty: !me.isCreate,
                minValue: 1,
                maxValue: 65535,
                fieldLabel: gettext('Port'),
            },
            {
                xtype: 'textfield',
                name: 'binddn',
                allowBlank: true,
                fieldLabel: gettext('User name'),
            },
            {
                xtype: 'textfield',
                inputType: 'password',
                allowBlank: true,
                emptyText: gettext('Unchanged'),
                name: 'bindpw',
                fieldLabel: gettext('Password'),
            },
        ];

        me.column2 = [
            {
                xtype: 'proxmoxcheckbox',
                name: 'enable',
                checked: true,
                uncheckedValue: 0,
                fieldLabel: gettext('Enable'),
            },
            {
                xtype: 'proxmoxtextfield',
                allowBlank: true,
                deleteEmpty: !me.isCreate,
                name: 'basedn',
                fieldLabel: gettext('Base DN'),
            },
            {
                xtype: 'proxmoxtextfield',
                allowBlank: true,
                deleteEmpty: !me.isCreate,
                name: 'groupbasedn',
                fieldLabel: gettext('Base DN for Groups'),
            },
            {
                xtype: 'proxmoxtextfield',
                allowBlank: true,
                deleteEmpty: !me.isCreate,
                name: 'mailattr',
                fieldLabel: gettext('EMail attribute name(s)'),
            },
            {
                xtype: 'proxmoxtextfield',
                allowBlank: true,
                deleteEmpty: !me.isCreate,
                name: 'accountattr',
                fieldLabel: gettext('Account attribute name'),
            },
            {
                xtype: 'proxmoxtextfield',
                allowBlank: true,
                deleteEmpty: !me.isCreate,
                name: 'filter',
                fieldLabel: gettext('LDAP filter'),
            },
            {
                xtype: 'proxmoxtextfield',
                allowBlank: true,
                deleteEmpty: !me.isCreate,
                name: 'groupclass',
                fieldLabel: gettext('Group objectclass'),
            },
        ];

        me.columnB = [
            {
                xtype: 'textfield',
                fieldLabel: gettext('Comment'),
                allowBlank: true,
                name: 'comment',
            },
        ];

        me.callParent();
    },
});

Ext.define('PMG.LDAPEdit', {
    extend: 'Proxmox.window.Edit',
    alias: 'widget.pmgLDAPEdit',
    onlineHelp: 'pmgconfig_ldap',

    subject: 'LDAP Profile',
    isAdd: true,

    initComponent: function () {
        var me = this;

        me.isCreate = !me.profileId;

        if (me.isCreate) {
            me.url = '/api2/extjs/config/ldap';
            me.method = 'POST';
        } else {
            me.url = '/api2/extjs/config/ldap/' + me.profileId + '/config';
            me.method = 'PUT';
        }

        var ipanel = Ext.create('PMG.LDAPInputPanel', {
            isCreate: me.isCreate,
            profileId: me.profileId,
        });

        me.items = [ipanel];

        me.fieldDefaults = {
            labelWidth: 150,
        };

        me.callParent();

        if (!me.isCreate) {
            me.load({
                success: function (response, options) {
                    var values = response.result.data;

                    values.enable = values.disable ? 0 : 1;
                    values.verify = !!values.verify;
                    ipanel.setValues(values);
                },
            });
        }
    },
});

Ext.define('PMG.LDAPUserGrid', {
    extend: 'Ext.grid.Panel',
    xtype: 'pmgLDAPUserGrid',

    emptyText: gettext('No data in database'),
    store: {
        autoDestroy: true,
        fields: ['dn', 'account', 'pmail'],
        proxy: { type: 'proxmox' },
        sorters: ['dn'],
    },
    columns: [
        {
            text: 'DN',
            dataIndex: 'dn',
            flex: 1,
        },
        {
            text: gettext('Account'),
            dataIndex: 'account',
            flex: 1,
        },
        {
            text: gettext('Primary E-Mail'),
            dataIndex: 'pmail',
            flex: 1,
        },
    ],

    initComponent: function () {
        var me = this;
        me.callParent();
        if (me.url) {
            me.getStore().getProxy().setUrl(me.url);
            me.getStore().load();
        }
    },
});

Ext.define('PMG.LDAPConfig', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgLDAPConfig',

    controller: {
        xclass: 'Ext.app.ViewController',

        openUserList: function (grid, record) {
            var name = this.getViewModel().get('name');
            Ext.create('Ext.window.Window', {
                title: Ext.String.format(gettext("Users of '{0}'"), record.data.dn),
                modal: true,
                width: 600,
                height: 400,
                layout: 'fit',
                items: [
                    {
                        xtype: 'pmgLDAPUserGrid',
                        border: false,
                        url:
                            '/api2/json/config/ldap/' +
                            name +
                            '/groups/' +
                            encodeURIComponent(record.data.gid),
                    },
                ],
            }).show();
        },

        showUsers: function (button) {
            var me = this;
            var view = me.lookup('groupgrid');
            var record = view.getSelection()[0];
            me.openUserList(view, record);
        },

        openUserMails: function (grid, record) {
            var name = this.getViewModel().get('name');
            Ext.create('Ext.window.Window', {
                title: Ext.String.format(gettext("E-Mail addresses of '{0}'"), record.data.dn),
                modal: true,
                width: 600,
                height: 400,
                layout: 'fit',
                items: [
                    {
                        xtype: 'grid',
                        border: false,
                        store: {
                            autoLoad: true,
                            field: ['email', 'primary'],
                            proxy: {
                                type: 'proxmox',
                                url:
                                    '/api2/json/config/ldap/' +
                                    name +
                                    '/users/' +
                                    encodeURIComponent(record.data.pmail),
                            },
                        },
                        columns: [{ dataIndex: 'email', text: gettext('E-Mail address'), flex: 1 }],
                    },
                ],
            }).show();
        },

        showEmails: function (button) {
            var me = this;
            var view = me.lookup('usergrid');
            var record = view.getSelection()[0];
            me.openUserMails(view, record);
        },

        reload: function (grid) {
            var me = this;
            var selection = grid.getSelection();
            me.showInfo(grid, selection);
        },

        showInfo: function (grid, selected) {
            var me = this;
            var viewModel = me.getViewModel();
            if (selected[0]) {
                let name = selected[0].data.profile;
                viewModel.set('selected', true);
                viewModel.set('name', name);

                // set grid stores and load them
                let gstore = me.lookup('groupgrid').getStore();
                let ustore = me.lookup('usergrid').getStore();
                gstore.getProxy().setUrl('/api2/json/config/ldap/' + name + '/groups');
                ustore.getProxy().setUrl('/api2/json/config/ldap/' + name + '/users');
                gstore.load();
                ustore.load();
            } else {
                viewModel.set('selected', false);
            }
        },

        init: function (view) {
            var me = this;
            me.lookup('grid').relayEvents(view, ['activate']);
            var groupgrid = me.lookup('groupgrid');
            var usergrid = me.lookup('usergrid');

            Proxmox.Utils.monStoreErrors(groupgrid, groupgrid.getStore(), true);
            Proxmox.Utils.monStoreErrors(usergrid, usergrid.getStore(), true);
        },

        control: {
            'grid[reference=grid]': {
                selectionchange: 'showInfo',
                load: 'reload',
            },
            'grid[reference=groupgrid]': {
                itemdblclick: 'openUserList',
            },
            'grid[reference=usergrid]': {
                itemdblclick: 'openUserMails',
            },
        },
    },

    viewModel: {
        data: {
            name: '',
            selected: false,
        },
    },

    layout: 'border',

    items: [
        {
            region: 'center',
            reference: 'grid',
            xtype: 'pmgLDAPConfigGrid',
            border: false,
        },
        {
            xtype: 'tabpanel',
            reference: 'data',
            hidden: true,
            height: '50%',
            border: false,
            split: true,
            region: 'south',
            bind: {
                hidden: '{!selected}',
            },
            items: [
                {
                    xtype: 'grid',
                    reference: 'groupgrid',
                    border: false,
                    emptyText: gettext('No data in database'),
                    tbar: [
                        {
                            xtype: 'proxmoxButton',
                            text: gettext('Show Users'),
                            handler: 'showUsers',
                            disabled: true,
                        },
                    ],
                    store: {
                        fields: ['dn', 'gid'],
                        proxy: { type: 'proxmox' },
                        sorters: ['dn'],
                    },
                    bind: {
                        title: Ext.String.format(gettext("Groups of '{0}'"), '{name}'),
                    },
                    columns: [
                        {
                            text: 'DN',
                            dataIndex: 'dn',
                            flex: 1,
                        },
                    ],
                },
                {
                    xtype: 'pmgLDAPUserGrid',
                    reference: 'usergrid',
                    border: false,
                    tbar: [
                        {
                            xtype: 'proxmoxButton',
                            text: gettext('Show E-Mail addresses'),
                            handler: 'showEmails',
                            disabled: true,
                        },
                    ],
                    bind: {
                        title: Ext.String.format(gettext("Users of '{0}'"), '{name}'),
                    },
                },
            ],
        },
    ],
});

Ext.define('PMG.LDAPConfigGrid', {
    extend: 'Ext.grid.GridPanel',
    alias: 'widget.pmgLDAPConfigGrid',

    controller: {
        xclass: 'Ext.app.ViewController',

        run_editor: function () {
            var me = this;
            var view = me.getView();
            var rec = view.getSelection()[0];
            if (!rec) {
                return;
            }

            var win = Ext.createWidget('pmgLDAPEdit', {
                profileId: rec.data.profile,
            });
            win.on('destroy', me.reload, me);
            win.load();
            win.show();
        },

        newProfile: function () {
            var me = this;
            var win = Ext.createWidget('pmgLDAPEdit', {});
            win.on('destroy', me.reload, me);
            win.show();
        },

        reload: function () {
            let view = this.getView();
            view.getStore().load();
            view.fireEvent('load', view);
        },

        sync: function () {
            var me = this;
            var view = me.getView();
            var rec = view.getSelection()[0];
            Proxmox.Utils.API2Request({
                url: '/config/ldap/' + rec.data.profile + '/sync',
                method: 'POST',
                waitMsgTarget: view,
                callback: function () {
                    me.reload();
                },
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        init: function (view) {
            var _me = this;
            Proxmox.Utils.monStoreErrors(view, view.getStore(), true);
        },
    },

    store: {
        model: 'pmg-ldap-config',
        sorters: [
            {
                property: 'profile',
                direction: 'ASC',
            },
        ],
    },

    tbar: [
        {
            text: gettext('Create'),
            handler: 'newProfile',
        },
        '-',
        {
            xtype: 'proxmoxButton',
            text: gettext('Edit'),
            disabled: true,
            handler: 'run_editor',
        },
        {
            xtype: 'proxmoxStdRemoveButton',
            baseurl: '/config/ldap',
            callback: 'reload',
        },
        '-',
        {
            xtype: 'proxmoxButton',
            text: gettext('Synchronize'),
            enableFn: function (rec) {
                return !rec.data.disable;
            },
            disabled: true,
            handler: 'sync',
        },
    ],

    listeners: {
        itemdblclick: 'run_editor',
        activate: 'reload',
    },

    columns: [
        {
            header: gettext('Profile Name'),
            sortable: true,
            width: 120,
            dataIndex: 'profile',
        },
        {
            header: gettext('Protocol'),
            sortable: true,
            dataIndex: 'mode',
            renderer: PMG.Utils.format_ldap_protocol,
        },
        {
            header: gettext('Server'),
            sortable: true,
            dataIndex: 'server1',
            renderer: function (value, metaData, rec) {
                if (rec.data.server2) {
                    return value + '<br>' + rec.data.server2;
                }
                return value;
            },
        },
        {
            header: gettext('Enabled'),
            width: 80,
            sortable: true,
            dataIndex: 'disable',
            renderer: Proxmox.Utils.format_neg_boolean,
        },
        {
            header: gettext('Comment'),
            sortable: false,
            renderer: Ext.String.htmlEncode,
            dataIndex: 'comment',
            flex: 1,
        },
        {
            header: gettext('Accounts'),
            width: 80,
            sortable: true,
            dataIndex: 'ucount',
        },
        {
            header: gettext('Addresses'),
            width: 80,
            sortable: true,
            dataIndex: 'mcount',
        },
        {
            header: gettext('Groups'),
            width: 80,
            sortable: true,
            dataIndex: 'gcount',
        },
    ],
});
Ext.define('PMG.UserEdit', {
    extend: 'Proxmox.window.Edit',
    alias: 'widget.pmgUserEdit',
    mixins: ['Proxmox.Mixin.CBind'],
    onlineHelp: 'pmgconfig_localuser',

    userid: undefined,

    isAdd: true,

    subject: gettext('User'),

    fieldDefaults: { labelWidth: 120 },

    cbindData: function (initialConfig) {
        var me = this;

        var userid = initialConfig.userid;
        var baseurl = '/api2/extjs/access/users';

        me.isCreate = !userid;
        me.url = userid ? baseurl + '/' + userid : baseurl;
        me.method = userid ? 'PUT' : 'POST';
        me.autoLoad = !!userid;

        return {
            useridXType: userid ? 'displayfield' : 'textfield',
            isSuperUser: userid === 'root@pam',
        };
    },

    viewModel: {
        data: {
            realm: 'pmg',
        },
        formulas: {
            maySetPassword: function (get) {
                let realm = get('realm');

                let view = this.getView();
                let realmStore = view.down('pmxRealmComboBox').getStore();
                if (realmStore.isLoaded()) {
                    let rec = realmStore.findRecord('realm', realm, 0, false, true, true);
                    return rec.data.type === 'pmg' && view.isCreate;
                } else {
                    return view.isCreate;
                }
            },
        },
    },

    items: {
        xtype: 'inputpanel',
        column1: [
            {
                xtype: 'textfield',
                name: 'username',
                fieldLabel: gettext('User name'),
                renderer: Ext.htmlEncode,
                allowBlank: false,
                cbind: {
                    submitValue: '{isCreate}',
                    xtype: '{useridXType}',
                },
            },
            {
                xtype: 'textfield',
                inputType: 'password',
                fieldLabel: gettext('Password'),
                minLength: 8,
                allowBlank: false,
                name: 'password',
                listeners: {
                    change: function (field) {
                        field.next().validate();
                    },
                    blur: function (field) {
                        field.next().validate();
                    },
                },
                bind: {
                    hidden: '{!maySetPassword}',
                    disabled: '{!maySetPassword}',
                },
            },
            {
                xtype: 'textfield',
                inputType: 'password',
                fieldLabel: gettext('Confirm password'),
                name: 'verifypassword',
                vtype: 'password',
                initialPassField: 'password',
                allowBlank: false,
                submitValue: false,
                bind: {
                    hidden: '{!maySetPassword}',
                    disabled: '{!maySetPassword}',
                },
            },
            {
                xtype: 'pmgRoleSelector',
                name: 'role',
                allowBlank: false,
                fieldLabel: gettext('Role'),
                cbind: {
                    disabled: '{isSuperUser}',
                },
            },
            {
                xtype: 'datefield',
                name: 'expire',
                emptyText: Proxmox.Utils.neverText,
                format: 'Y-m-d',
                submitFormat: 'U',
                fieldLabel: gettext('Expire'),
                cbind: {
                    disabled: '{isSuperUser}',
                },
            },
            {
                xtype: 'proxmoxcheckbox',
                fieldLabel: gettext('Enabled'),
                name: 'enable',
                uncheckedValue: 0,
                defaultValue: 1,
                checked: true,
                cbind: {
                    disabled: '{isSuperUser}',
                },
            },
        ],

        column2: [
            {
                xtype: 'pmxRealmComboBox',
                reference: 'realmfield',
                name: 'realm',
                baseUrl: '/access/auth-realm',
                bind: {
                    value: '{realm}',
                },
                cbind: {
                    disabled: '{!isCreate}',
                    hidden: '{!isCreate}',
                },
            },
            {
                xtype: 'proxmoxtextfield',
                name: 'firstname',
                fieldLabel: gettext('First Name'),
                cbind: {
                    deleteEmpty: '{!isCreate}',
                },
            },
            {
                xtype: 'proxmoxtextfield',
                name: 'lastname',
                fieldLabel: gettext('Last Name'),
                cbind: {
                    deleteEmpty: '{!isCreate}',
                },
            },
            {
                xtype: 'proxmoxtextfield',
                name: 'email',
                fieldLabel: gettext('E-Mail'),
                vtype: 'proxmoxMail',
                cbind: {
                    deleteEmpty: '{!isCreate}',
                },
            },
        ],

        columnB: [
            {
                xtype: 'proxmoxtextfield',
                name: 'comment',
                fieldLabel: gettext('Comment'),
                cbind: {
                    disabled: '{isSuperUser}',
                    deleteEmpty: '{!isCreate}',
                },
            },
            {
                xtype: 'proxmoxtextfield',
                name: 'keys',
                fieldLabel: gettext('Key IDs'),
                cbind: {
                    deleteEmpty: '{!isCreate}',
                },
            },
        ],
    },

    getValues: function (dirtyOnly) {
        var me = this;

        var values = me.callParent(arguments);

        // hack: ExtJS datefield does not submit 0, so we need to set that
        if (!values.expire) {
            values.expire = 0;
        }

        if (me.isCreate) {
            values.userid = values.username + '@' + values.realm;
        }

        delete values.username;

        if (!values.password) {
            delete values.password;
        }

        return values;
    },

    setValues: function (values) {
        var me = this;

        if (Ext.isDefined(values.expire)) {
            if (values.expire) {
                values.expire = new Date(values.expire * 1000);
            } else {
                // display 'never' instead of '1970-01-01'
                values.expire = null;
            }
        }

        me.callParent([values]);
    },
});
Ext.define('pmg-users', {
    extend: 'Ext.data.Model',
    fields: [
        'userid',
        'firstname',
        'lastname',
        'email',
        'comment',
        'role',
        'keys',
        'realm',
        'totp-lock',
        { type: 'boolean', name: 'enable' },
        { type: 'date', dateFormat: 'timestamp', name: 'expire' },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/access/users',
    },
    idProperty: 'userid',
});

Ext.define('PMG.UserView', {
    extend: 'Ext.grid.GridPanel',
    alias: 'widget.pmgUserView',

    store: {
        autoLoad: true,
        model: 'pmg-users',
        sorters: [
            {
                property: 'realm',
                direction: 'ASC',
            },
            {
                property: 'userid',
                direction: 'ASC',
            },
        ],
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        init: function (view) {
            Proxmox.Utils.monStoreErrors(view, view.store);
        },

        renderUsername: function (userid) {
            return Ext.htmlEncode(userid.match(/^(.+)(@[^@]+)$/)[1]);
        },

        renderFullName: function (firstname, metaData, record) {
            var first = firstname || '';
            var last = record.data.lastname || '';
            return Ext.htmlEncode(first + ' ' + last);
        },

        onAdd: function () {
            var view = this.getView();

            var win = Ext.create('PMG.UserEdit', {});
            win.on('destroy', function () {
                view.reload();
            });
            win.show();
        },

        onEdit: function () {
            var view = this.getView();

            var rec = view.selModel.getSelection()[0];

            var win = Ext.create('PMG.UserEdit', {
                userid: rec.data.userid,
            });
            win.on('destroy', function () {
                view.reload();
            });
            win.show();
        },

        onPassword: function (btn, event, rec) {
            var view = this.getView();

            var win = Ext.create('Proxmox.window.PasswordEdit', {
                userid: rec.data.userid,
            });
            win.on('destroy', function () {
                view.reload();
            });
            win.show();
        },

        onAfterRemove: function (btn, res) {
            var view = this.getView();
            view.reload();
        },

        onUnlockTfa: function (btn, event, rec) {
            let me = this;
            let view = me.getView();
            Ext.Msg.confirm(
                Ext.String.format(gettext('Unlock TFA authentication for {0}'), rec.data.userid),
                gettext(
                    "Locked 2nd factors can happen if the user's password was leaked. Are you sure you want to unlock the user?",
                ),
                function (btn_response) {
                    if (btn_response === 'yes') {
                        Proxmox.Utils.API2Request({
                            url: `/access/users/${rec.data.userid}/unlock-tfa`,
                            waitMsgTarget: view,
                            method: 'PUT',
                            failure: function (response, options) {
                                Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                            },
                            success: function (response, options) {
                                view.reload();
                            },
                        });
                    }
                },
            );
        },
    },

    listeners: {
        scope: 'controller',
        itemdblclick: 'onEdit',
    },

    tbar: [
        {
            text: gettext('Add'),
            reference: 'addBtn',
            handler: 'onAdd',
        },
        '-',
        {
            xtype: 'proxmoxButton',
            text: gettext('Edit'),
            disabled: true,
            handler: 'onEdit',
        },
        {
            xtype: 'proxmoxStdRemoveButton',
            baseurl: '/access/users',
            reference: 'removeBtn',
            callback: 'onAfterRemove',
            waitMsgTarget: true,
        },
        {
            xtype: 'proxmoxButton',
            text: gettext('Password'),
            disabled: true,
            handler: 'onPassword',
        },
        '-',
        {
            xtype: 'proxmoxButton',
            text: gettext('Unlock TFA'),
            handler: 'onUnlockTfa',
            disabled: true,
            enableFn: ({ data }) =>
                data['totp-locked'] || data['tfa-locked-until'] > new Date().getTime() / 1000,
        },
    ],

    columns: [
        {
            header: gettext('User name'),
            width: 200,
            sortable: true,
            renderer: 'renderUsername',
            dataIndex: 'userid',
        },
        {
            header: gettext('Realm'),
            width: 100,
            sortable: true,
            dataIndex: 'realm',
        },
        {
            header: gettext('Role'),
            width: 150,
            sortable: true,
            renderer: PMG.Utils.format_user_role,
            dataIndex: 'role',
        },
        {
            header: gettext('Enabled'),
            width: 80,
            sortable: true,
            renderer: Proxmox.Utils.format_boolean,
            dataIndex: 'enable',
        },
        {
            header: gettext('Expire'),
            width: 80,
            sortable: true,
            renderer: Proxmox.Utils.format_expire,
            dataIndex: 'expire',
        },
        {
            header: gettext('Name'),
            width: 150,
            sortable: true,
            renderer: 'renderFullName',
            dataIndex: 'firstname',
        },
        {
            header: gettext('TFA Lock'),
            width: 120,
            sortable: true,
            dataIndex: 'totp-locked',
            renderer: function (v, metaData, record) {
                let locked_until = record.data['tfa-locked-until'];
                if (locked_until !== undefined) {
                    let now = new Date().getTime() / 1000;
                    if (locked_until > now) {
                        return gettext('Locked');
                    }
                }

                if (record.data['totp-locked']) {
                    return gettext('TOTP Locked');
                }

                return Proxmox.Utils.noText;
            },
        },
        {
            header: gettext('Comment'),
            sortable: false,
            renderer: Ext.String.htmlEncode,
            dataIndex: 'comment',
            flex: 1,
        },
    ],

    reload: function () {
        var me = this;

        me.store.load();
    },
});
Ext.define('PMG.OIDCInputPanel', {
    extend: 'Proxmox.panel.InputPanel',
    xtype: 'pmgAuthOIDCPanel',
    mixins: ['Proxmox.Mixin.CBind'],
    onlineHelp: 'user_oidc',

    type: 'oidc',

    viewModel: {
        data: {
            roleSource: '__default__',
            autocreate: 0,
        },
        formulas: {
            hideFixedRoleAssignment: function (get) {
                return get('roleSource') !== 'fixed' || !get('autocreate');
            },
            hideClaimRoleAssignment: function (get) {
                return get('roleSource') !== 'from-claim' || !get('autocreate');
            },
        },
    },

    onGetValues: function (values) {
        let me = this;

        if (me.isCreate && !me.useTypeInUrl) {
            values.type = me.type;
        }

        let autocreateRoleAssignment = {};
        if (values.source) {
            autocreateRoleAssignment.source = values.source;
        }
        if (values.source === 'fixed') {
            autocreateRoleAssignment['fixed-role'] = values['fixed-role'];
        } else if (values.source === 'from-claim') {
            autocreateRoleAssignment['role-claim'] = values['role-claim'];
        }
        values['autocreate-role-assignment'] =
            Proxmox.Utils.printPropertyString(autocreateRoleAssignment);
        Proxmox.Utils.delete_if_default(values, 'autocreate-role-assignment', '', me.isCreate);

        delete values.source;
        delete values['fixed-role'];
        delete values['role-claim'];

        return values;
    },

    setValues: function (values) {
        let autocreateRoleAssignment = Proxmox.Utils.parsePropertyString(
            values['autocreate-role-assignment'],
        );

        values.source = autocreateRoleAssignment.source ?? '__default__';

        if (autocreateRoleAssignment.source === 'fixed') {
            values['fixed-role'] = autocreateRoleAssignment['fixed-role'];
        }
        if (autocreateRoleAssignment.source === 'from-claim') {
            values['role-claim'] = autocreateRoleAssignment['role-claim'];
        }

        this.callParent(arguments);
    },

    columnT: [
        {
            xtype: 'textfield',
            name: 'issuer-url',
            fieldLabel: gettext('Issuer URL'),
            allowBlank: false,
        },
    ],

    column1: [
        {
            xtype: 'pmxDisplayEditField',
            name: 'realm',
            cbind: {
                value: '{realm}',
                editable: '{isCreate}',
            },
            fieldLabel: gettext('Realm'),
            allowBlank: false,
        },
        {
            xtype: 'proxmoxcheckbox',
            fieldLabel: gettext('Default realm'),
            name: 'default',
            value: 0,
            cbind: {
                deleteEmpty: '{!isCreate}',
            },
            autoEl: {
                tag: 'div',
                'data-qtip': gettext('Set realm as default for login'),
            },
        },
        {
            xtype: 'proxmoxtextfield',
            fieldLabel: gettext('Client ID'),
            name: 'client-id',
            allowBlank: false,
        },
        {
            xtype: 'proxmoxtextfield',
            fieldLabel: gettext('Client Key'),
            cbind: {
                deleteEmpty: '{!isCreate}',
            },
            name: 'client-key',
        },
    ],

    column2: [
        {
            xtype: 'pmxDisplayEditField',
            name: 'username-claim',
            fieldLabel: gettext('Username Claim'),
            editConfig: {
                xtype: 'proxmoxKVComboBox',
                editable: true,
                comboItems: [
                    ['__default__', Proxmox.Utils.defaultText],
                    ['sub', gettext('sub (subject)')],
                    ['preferred_username', gettext('preferred_username')],
                ],
            },
            cbind: {
                value: (get) => (get('isCreate') ? '__default__' : Proxmox.Utils.defaultText),
                deleteEmpty: '{!isCreate}',
                editable: '{isCreate}',
            },
        },
        {
            xtype: 'proxmoxtextfield',
            name: 'scopes',
            fieldLabel: gettext('Scopes'),
            emptyText: `${Proxmox.Utils.defaultText} (email profile)`,
            submitEmpty: false,
            cbind: {
                deleteEmpty: '{!isCreate}',
            },
        },
        {
            xtype: 'proxmoxKVComboBox',
            name: 'prompt',
            fieldLabel: gettext('Prompt'),
            editable: true,
            emptyText: gettext('Auth-Provider Default'),
            comboItems: [
                ['__default__', gettext('Auth-Provider Default')],
                ['none', 'none'],
                ['login', 'login'],
                ['consent', 'consent'],
                ['select_account', 'select_account'],
            ],
            cbind: {
                deleteEmpty: '{!isCreate}',
            },
        },
    ],

    columnB: [
        {
            xtype: 'proxmoxtextfield',
            name: 'comment',
            fieldLabel: gettext('Comment'),
            cbind: {
                deleteEmpty: '{!isCreate}',
            },
        },
        {
            xtype: 'displayfield',
            value: gettext('Autocreate Options'),
        },
        {
            xtype: 'proxmoxcheckbox',
            fieldLabel: gettext('Autocreate Users'),
            name: 'autocreate',
            bind: {
                value: '{autocreate}',
            },
            cbind: {
                deleteEmpty: '{!isCreate}',
            },
        },
        {
            xtype: 'proxmoxKVComboBox',
            name: 'source',
            fieldLabel: gettext('Source for Role Assignment'),
            allowBlank: false,
            deleteEmpty: false,
            comboItems: [
                [
                    '__default__',
                    Proxmox.Utils.defaultText +
                        ' (' +
                        gettext('All auto-created users get audit role') +
                        ')',
                ],
                ['fixed', gettext('Fixed role for all auto-created users')],
                ['from-claim', gettext('Get role from OIDC claim')],
            ],
            bind: {
                value: '{roleSource}',
                disabled: '{!autocreate}',
                hidden: '{!autocreate}',
            },
        },
        {
            xtype: 'pmgRoleSelector',
            name: 'fixed-role',
            allowBlank: false,
            deleteEmpty: false,
            fieldLabel: gettext('Fixed Role'),
            bind: {
                disabled: '{hideFixedRoleAssignment}',
                hidden: '{hideFixedRoleAssignment}',
            },
        },
        {
            xtype: 'proxmoxtextfield',
            name: 'role-claim',
            allowBlank: false,
            deleteEmpty: false,
            fieldLabel: gettext('Role Claim'),
            bind: {
                disabled: '{hideClaimRoleAssignment}',
                hidden: '{hideClaimRoleAssignment}',
            },
        },
    ],

    advancedColumnB: [
        {
            xtype: 'proxmoxtextfield',
            name: 'acr-values',
            fieldLabel: gettext('ACR Values'),
            submitEmpty: false,
            cbind: {
                deleteEmpty: '{!isCreate}',
            },
        },
    ],
});
// TODO merge with the one from pbs in widget toolkit
Ext.define('PMG.WebauthnConfigEdit', {
    extend: 'Proxmox.window.Edit',
    alias: ['widget.pmgWebauthnConfigEdit'],

    subject: gettext('Webauthn'),
    url: '/api2/extjs/config/tfa/webauthn',
    autoLoad: true,

    width: 512,

    fieldDefaults: {
        labelWidth: 120,
    },

    setValues: function (values) {
        let me = this;

        me.relayingPartySet = values && typeof values.rp === 'string';

        me.callParent(arguments);
    },

    items: [
        {
            xtype: 'textfield',
            fieldLabel: gettext('Relying Party'),
            name: 'rp',
            allowBlank: false,
            listeners: {
                dirtychange: function (field, isDirty) {
                    let win = field.up('window');
                    let warningBox = win.down('box[id=rpChangeWarning]');
                    warningBox.setHidden(!win.relayingPartySet || !isDirty);
                },
            },
        },
        {
            xtype: 'textfield',
            fieldLabel: gettext('Origin'),
            name: 'origin',
            allowBlank: false,
        },
        {
            xtype: 'textfield',
            fieldLabel: 'ID',
            name: 'id',
            allowBlank: false,
        },
        {
            xtype: 'container',
            layout: 'hbox',
            items: [
                {
                    xtype: 'box',
                    flex: 1,
                },
                {
                    xtype: 'button',
                    text: gettext('Auto-fill'),
                    iconCls: 'fa fa-fw fa-pencil-square-o',
                    handler: function (button, ev) {
                        let panel = this.up('panel');
                        panel.down('field[name=rp]').setValue(document.location.hostname);
                        panel.down('field[name=origin]').setValue(document.location.origin);
                        panel.down('field[name=id]').setValue(document.location.hostname);
                    },
                },
            ],
        },
        {
            xtype: 'box',
            html:
                `<span class='pmx-hint'>${gettext('Note:')}</span> ` +
                gettext('WebAuthn requires using a trusted certificate.'),
        },
        {
            xtype: 'box',
            id: 'rpChangeWarning',
            hidden: true,
            padding: '5 0 0 0',
            html:
                '<i class="fa fa-exclamation-triangle warning"></i> ' +
                gettext('Changing the Relying Party may break existing webAuthn TFA entries.'),
        },
    ],
});

Ext.define('PMG.TFAView', {
    extend: 'Proxmox.panel.TfaView',
    alias: 'widget.pmgTFAView',

    initComponent: function () {
        let me = this;

        me.tbar.push('->', {
            text: gettext('WebAuthn'),
            itemId: 'webauthn',
            iconCls: 'fa fa-fw fa-cog',
            handler: () => Ext.create('PMG.WebauthnConfigEdit', { autoShow: true }),
        });

        me.callParent(arguments);
    },
});
Ext.define('PMG.FetchmailEdit', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgFetchmailEdit',
    onlineHelp: 'pmgconfig_fetchmail',

    userid: undefined,

    isAdd: true,

    subject: 'Fetchmail',

    fieldDefaults: { labelWidth: 120 },

    controller: {
        xclass: 'Ext.app.ViewController',

        onProtocolChange: function () {
            var protocol = this.lookupReference('protocol').getValue();
            var ssl = this.lookupReference('ssl').getValue();

            var port_field = this.lookupReference('port');
            if (protocol === 'pop3') {
                port_field.setValue(ssl ? 995 : 110);
            } else if (protocol === 'imap') {
                port_field.setValue(ssl ? 993 : 143);
            }
        },
    },

    items: {
        xtype: 'inputpanel',
        column1: [
            {
                xtype: 'textfield',
                name: 'server',
                fieldLabel: gettext('Server'),
                allowBlank: false,
            },
            {
                xtype: 'proxmoxKVComboBox',
                fieldLabel: gettext('Protocol'),
                name: 'protocol',
                reference: 'protocol',
                value: 'pop3',
                listeners: { change: 'onProtocolChange' },
                comboItems: [
                    ['pop3', 'pop3'],
                    ['imap', 'imap'],
                ],
            },
            {
                xtype: 'proxmoxintegerfield',
                name: 'port',
                reference: 'port',
                fieldLabel: gettext('Port'),
                value: 110,
                minValue: 1,
                maxValue: 65535,
                allowBlank: false,
            },
            {
                xtype: 'textfield',
                name: 'user',
                fieldLabel: gettext('Username'),
                allowBlank: false,
            },
            {
                xtype: 'textfield',
                name: 'pass',
                inputType: 'password',
                fieldLabel: gettext('Password'),
                allowBlank: false,
            },
            {
                xtype: 'textfield',
                name: 'target',
                fieldLabel: gettext('Deliver to'),
                allowBlank: false,
            },
        ],
        column2: [
            {
                xtype: 'proxmoxcheckbox',
                fieldLabel: gettext('Enabled'),
                name: 'enable',
                uncheckedValue: 0,
                checked: true,
            },
            {
                xtype: 'proxmoxintegerfield',
                name: 'interval',
                fieldLabel: gettext('Interval'),
                value: 1,
                minValue: 1,
                maxValue: 24 * 12 * 7,
                allowBlank: false,
            },
            {
                xtype: 'proxmoxcheckbox',
                fieldLabel: gettext('Use SSL'),
                listeners: { change: 'onProtocolChange' },
                name: 'ssl',
                reference: 'ssl',
                uncheckedValue: 0,
                checked: false,
            },
            {
                xtype: 'proxmoxcheckbox',
                fieldLabel: gettext('Keep old mails'),
                name: 'keep',
                uncheckedValue: 0,
                checked: false,
            },
        ],
    },
});
Ext.define('pmg-fetchmail-users', {
    extend: 'Ext.data.Model',
    fields: [
        'id',
        'protocol',
        'port',
        'server',
        'user',
        'pass',
        'target',
        'ssl',
        'keep',
        { type: 'integer', name: 'interval' },
        { type: 'boolean', name: 'enable' },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/config/fetchmail',
    },
    idProperty: 'id',
});

Ext.define('PMG.FetchmailView', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgFetchmailView',

    baseurl: '/api2/extjs/config/fetchmail',

    store: {
        autoDestroy: true,
        autoLoad: true,
        model: 'pmg-fetchmail-users',
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        init: function (view) {
            Proxmox.Utils.monStoreErrors(view, view.store, true);
        },

        onAdd: function () {
            var view = this.getView();

            var win = Ext.create('PMG.FetchmailEdit', {
                url: view.baseurl,
                method: 'POST',
            });
            win.on('destroy', function () {
                view.store.load();
            });
            win.show();
        },

        onEdit: function () {
            var view = this.getView();

            var rec = view.selModel.getSelection()[0];

            var win = Ext.create('PMG.FetchmailEdit', {
                userid: rec.data.id,
                url: view.baseurl + '/' + rec.data.id,
                method: 'PUT',
                autoLoad: true,
            });
            win.on('destroy', function () {
                view.store.load();
            });
            win.show();
        },

        onAfterRemove: function (btn, res) {
            var view = this.getView();
            view.store.load();
        },
    },

    tbar: [
        {
            text: gettext('Add'),
            reference: 'addBtn',
            handler: 'onAdd',
        },
        '-',
        {
            xtype: 'proxmoxButton',
            text: gettext('Edit'),
            disabled: true,
            handler: 'onEdit',
        },
        {
            xtype: 'proxmoxStdRemoveButton',
            baseurl: '/config/fetchmail',
            reference: 'removeBtn',
            callback: 'onAfterRemove',
            waitMsgTarget: true,
        },
    ],

    listeners: {
        //scope: 'controller',
        itemdblclick: 'onEdit',
    },

    columns: [
        {
            header: gettext('Server'),
            flex: 1,
            renderer: Ext.String.htmlEncode,
            dataIndex: 'server',
        },
        {
            header: gettext('Protocol'),
            dataIndex: 'protocol',
        },
        {
            header: gettext('User name'),
            flex: 1,
            renderer: Ext.String.htmlEncode,
            dataIndex: 'user',
        },
        {
            header: gettext('Deliver to'),
            flex: 1,
            renderer: Ext.String.htmlEncode,
            dataIndex: 'target',
        },
        {
            header: gettext('Enabled'),
            sortable: true,
            renderer: Proxmox.Utils.format_boolean,
            dataIndex: 'enable',
        },
        {
            header: gettext('Interval'),
            dataIndex: 'interval',
        },
    ],
});
Ext.define('PMG.UserManagement', {
    extend: 'Ext.tab.Panel',
    alias: 'widget.pmgUserManagement',

    title: gettext('Configuration') + ': ' + gettext('User Management'),

    border: false,
    defaults: { border: false },

    items: [
        {
            xtype: 'pmgUserView',
            title: gettext('Local'),
            itemId: 'local',
            iconCls: 'fa fa-user',
        },
        {
            xtype: 'pmgTFAView',
            title: 'Two Factor',
            itemId: 'tfa',
            iconCls: 'fa fa-key',
            issuerName: `Proxmox Mail Gateway - ${Proxmox.NodeName}`,
        },
        {
            xtype: 'pmgLDAPConfig',
            title: 'LDAP',
            itemId: 'ldap',
            iconCls: 'fa fa-address-book-o',
        },
        {
            xtype: 'pmgFetchmailView',
            title: 'Fetchmail',
            itemId: 'pop',
            iconCls: 'fa fa-reply-all',
        },
        {
            xtype: 'pmxAuthView',
            title: gettext('Realms'),
            itemId: 'realms',
            baseUrl: '/access/auth-realm',
            storeBaseUrl: '/access/auth-realm',
            showDefaultRealm: true,
            iconCls: 'fa fa-address-book-o',
        },
    ],
});
Ext.define('pmx-users', {
    extend: 'Ext.data.Model',
    fields: [
        'userid',
        'firstname',
        'lastname',
        'email',
        'comment',
        { type: 'boolean', name: 'enable' },
        { type: 'date', dateFormat: 'timestamp', name: 'expire' },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/access/users',
    },
    idProperty: 'userid',
});
Ext.define('PMG.ViewMailHeaders', {
    extend: 'Ext.window.Window',
    alias: 'widget.pmgViewMailHeaders',

    url: undefined,

    width: 600,

    height: 400,

    bodyPadding: 10,

    modal: true,

    layout: {
        type: 'vbox',
        align: 'stretch',
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        toggleRaw: function (field, newValue) {
            let me = this;
            let view = me.getView();
            view.raw = !newValue;
            me.loadData(view.url);
        },

        setData: function (data) {
            let view = this.getView();
            let panel = view.lookupReference('contentPanel');
            let from = data.match(/^FROM:\s*(.*\S)\s*$/im);
            if (from) {
                view.lookupReference('fromField').setValue(from[1]);
            }
            let to = data.match(/^TO:\s*(.*\S)\s*$/im);
            if (to) {
                view.lookupReference('toField').setValue(to[1]);
            }
            let subject = data.match(/^SUBJECT:\s*(.*\S)\s*$/im);
            if (subject) {
                view.lookupReference('subjectField').setValue(subject[1]);
            }
            panel.update(Ext.String.htmlEncode(data));
        },

        loadData: function (url) {
            let me = this;
            let view = me.getView();
            if (!view.raw) {
                url += '?decode-header=1';
            }
            Proxmox.Utils.API2Request({
                url,
                waitMsgTarget: view,
                method: 'GET',
                success: (response) => me.setData(response.result.data),
                failure: function (response, opts) {
                    view.destroy();
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        init: function (view) {
            let me = this;
            me.loadData(view.url);
        },
    },

    items: [
        {
            xtype: 'textfield',
            fieldLabel: gettext('From'),
            reference: 'fromField',
            focusable: false,
            exitable: false,
        },
        {
            xtype: 'textfield',
            fieldLabel: gettext('To'),
            reference: 'toField',
            focusable: false,
            exitable: false,
        },
        {
            xtype: 'textfield',
            fieldLabel: gettext('Subject'),
            reference: 'subjectField',
            focusable: false,
            exitable: false,
        },
        {
            xtype: 'container',
            layout: 'hbox',
            items: [
                {
                    xtype: 'displayfield',
                    fieldLabel: gettext('Header'),
                    flex: 1,
                },
                {
                    xtype: 'checkbox',
                    reference: 'raw',
                    boxLabel: gettext('Decode'),
                    value: true,
                    iconCls: 'fa fa-file-code-o',
                    handler: 'toggleRaw',
                },
            ],
        },
        {
            xtype: 'panel',
            bodyPadding: 5,
            reference: 'contentPanel',
            flex: 1,
            autoScroll: true,
            bodyStyle: 'white-space:pre',
        },
    ],
});
Ext.define('pmg-qshape', {
    extend: 'Ext.data.Model',
    fields: [
        'domain',
        { type: 'integer', name: 'total' },
        { type: 'integer', name: '5m' },
        { type: 'integer', name: '10m' },
        { type: 'integer', name: '20m' },
        { type: 'integer', name: '40m' },
        { type: 'integer', name: '80m' },
        { type: 'integer', name: '160m' },
        { type: 'integer', name: '320m' },
        { type: 'integer', name: '640m' },
        { type: 'integer', name: '1280m' },
        { type: 'integer', name: '1280m+' },
    ],
    idProperty: 'domain',
});

Ext.define('PMG.Postfix.QShape', {
    extend: 'Ext.grid.GridPanel',
    alias: 'widget.pmgPostfixQShape',

    nodename: undefined,

    store: {
        autoLoad: true,
        model: 'pmg-qshape',
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        init: function (view) {
            if (view.nodename) {
                view.setNodename(view.nodename);
            }
        },

        onFlush: function () {
            var view = this.getView();

            Proxmox.Utils.API2Request({
                url: '/api2/extjs/nodes/' + view.nodename + '/postfix/flush_queues',
                method: 'POST',
                waitMsgTarget: view,
                success: function (response, opts) {
                    view.store.load();
                },
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        onDeleteAll: function () {
            var view = this.getView();

            Proxmox.Utils.API2Request({
                url: '/api2/extjs/nodes/' + view.nodename + '/postfix/queue/deferred',
                method: 'DELETE',
                waitMsgTarget: view,
                success: function (response, opts) {
                    view.store.load();
                },
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        onDiscardVerifyDatabase: function () {
            var view = this.getView();

            Proxmox.Utils.API2Request({
                url: '/api2/extjs/nodes/' + view.nodename + '/postfix/discard_verify_cache',
                method: 'POST',
                waitMsgTarget: view,
                failure: function (response, opts) {
                    Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                },
            });
        },

        control: {
            '#': {
                activate: function () {
                    this.view.store.load(); // reload
                },
            },
        },
    },

    tbar: [
        {
            text: gettext('Flush Queue'),
            handler: 'onFlush',
        },
        {
            xtype: 'proxmoxButton',
            text: gettext('Delete all Messages'),
            dangerous: true,
            confirmMsg: 'Are you sure you want to delete all deferred mails?',
            selModel: null,
            handler: 'onDeleteAll',
        },
        {
            text: gettext('Discard address verification database'),
            handler: 'onDiscardVerifyDatabase',
        },
    ],

    columns: [
        {
            header: gettext('Domain'),
            width: 200,
            dataIndex: 'domain',
        },
        {
            header: gettext('Total'),
            width: 80,
            dataIndex: 'total',
        },
        {
            header: '5m',
            width: 80,
            dataIndex: '5m',
        },
        {
            header: '10m',
            width: 80,
            dataIndex: '10m',
        },
        {
            header: '20m',
            width: 80,
            dataIndex: '20m',
        },
        {
            header: '40m',
            width: 80,
            dataIndex: '40m',
        },
        {
            header: '80m',
            width: 80,
            dataIndex: '80m',
        },
        {
            header: '160m',
            width: 80,
            dataIndex: '160m',
        },
        {
            header: '320m',
            width: 80,
            dataIndex: '320m',
        },
        {
            header: '640m',
            width: 80,
            dataIndex: '640m',
        },
        {
            header: '1280m',
            width: 80,
            dataIndex: '1280m',
        },
        {
            header: '1280m+',
            width: 80,
            dataIndex: '1280m+',
        },
    ],

    setNodename: function (nodename) {
        var me = this;

        me.nodename = nodename;

        me.store.setProxy({
            type: 'proxmox',
            url: '/api2/json/nodes/' + me.nodename + '/postfix/qshape',
        });

        me.store.load();
    },
});
Ext.define('pmg-mailq', {
    extend: 'Ext.data.Model',
    fields: [
        'queue_id',
        'queue_name',
        { type: 'date', dateFormat: 'timestamp', name: 'arrival_time' },
        { type: 'integer', name: 'message_size' },
        'sender',
        'receiver',
        'reason',
    ],
    idProperty: 'queue_id',
});

Ext.define('PMG.Postfix.MailQueue', {
    extend: 'Ext.grid.GridPanel',
    alias: 'widget.pmgPostfixMailQueue',

    nodename: undefined,

    filter: undefined,

    queuename: 'deferred',

    selModel: {
        selType: 'checkboxmodel',
        mode: 'MULTI',
        showHeaderCheckbox: true,
    },

    store: {
        xclass: 'Ext.data.Store',
        model: 'pmg-mailq',
        remoteFilter: true,
        remoteSort: true,
        sorters: 'arrival_time',
        pageSize: 2000,
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        init: function (view) {
            if (view.nodename) {
                view.setNodename(view.nodename);
            }

            view.delayFilterTask = new Ext.util.DelayedTask(function () {
                view.filter = view.lookupReference('filter').getValue();
                view.updateProxy();
            });
        },

        onChangeFilter: function (f, v) {
            let view = this.getView();
            view.delayFilterTask.delay(500);
        },

        doAction: function (action, prompt) {
            let view = this.getView();
            let selection = view.getSelectionModel().getSelection();
            let ids = selection.map((r) => r.get('queue_id'));

            let do_action = function () {
                Proxmox.Utils.API2Request({
                    url: `/api2/extjs/nodes/${view.nodename}/postfix/queue/${view.queuename}`,
                    method: 'POST',
                    params: { action: action, ids: ids.join(';') },
                    waitMsgTarget: view,
                    success: () => {
                        view.selModel.deselectAll();
                        view.store.load();
                    },
                    failure: (response) => Ext.Msg.alert(gettext('Error'), response.htmlStatus),
                });
            };

            if (selection.length === 1 && action === 'deliver') {
                do_action(action, ids);
            } else {
                Ext.Msg.show({
                    title: gettext('Confirm'),
                    message: Ext.String.format(prompt, ids.length),
                    buttons: Ext.Msg.YESNO,
                    icon: Ext.Msg.INFO,
                    fn: function (btn) {
                        if (btn === 'yes') {
                            do_action(action, ids);
                        }
                    },
                });
            }
        },

        onFlush: function (button, event, rec) {
            this.doAction('deliver', gettext('Deliver {0} selected mails?'));
        },

        onRemove: function (button, event, rec) {
            this.doAction('delete', gettext('Delete {0} selected mails?'));
        },

        onHeaders: function (button, event, { data }) {
            let view = this.getView();

            Ext.create('PMG.ViewMailHeaders', {
                autoShow: true,
                title: view.title + ' : ' + data.queue_id,
                url: `/api2/extjs/nodes/${view.nodename}/postfix/queue/${view.queuename}/${data.queue_id}`,
            });
        },

        control: {
            '#': {
                activate: function () {
                    this.view.updateProxy(); // reload
                },
                itemdblclick: function (grid, rec, item, index, event) {
                    this.onHeaders(grid, event, rec);
                },
            },
            'field[reference=filter]': {
                change: 'onChangeFilter',
            },
        },
    },

    tbar: [
        {
            xtype: 'proxmoxButton',
            reference: 'headerBtn',
            enableFn: function (rec) {
                let grid = this.up('grid');
                if (!grid) {
                    return false;
                }
                return grid.getSelectionModel().getCount() === 1;
            },
            disabled: true,
            text: gettext('Headers'),
            handler: 'onHeaders',
        },
        {
            xtype: 'proxmoxButton',
            disabled: true,
            text: gettext('Flush'),
            handler: 'onFlush',
        },
        {
            xtype: 'proxmoxButton',
            disabled: true,
            text: gettext('Remove'),
            handler: 'onRemove',
        },
        {
            xtype: 'label',
            html: gettext('Filter') + ':',
        },
        {
            xtype: 'textfield',
            width: 300,
            reference: 'filter',
        },
    ],

    columns: [
        {
            header: gettext('Time'),
            flex: 2,
            renderer: Ext.util.Format.dateRenderer('Y-m-d H:i:s'),
            dataIndex: 'arrival_time',
        },
        {
            header: 'Size',
            renderer: Proxmox.Utils.render_size,
            flex: 1,
            dataIndex: 'message_size',
        },
        {
            header: gettext('Sender'),
            flex: 2,
            dataIndex: 'sender',
            renderer: Ext.htmlEncode,
        },
        {
            header: gettext('Receiver'),
            flex: 2,
            dataIndex: 'receiver',
            renderer: Ext.htmlEncode,
        },
        {
            header: gettext('Reason'),
            flex: 8,
            dataIndex: 'reason',
            renderer: Ext.htmlEncode,
        },
    ],

    pendingLoad: false,

    updateProxy: function () {
        let me = this;

        if (me.pendingLoad) {
            return;
        }

        let proxy = {
            type: 'proxmox',
            startParam: 'start',
            limitParam: 'limit',
            sortParam: 'sortfield',
            directionParam: 'sortdir',
            simpleSortMode: true,
            url: '/api2/json/nodes/' + me.nodename + '/postfix/queue/' + me.queuename,
        };

        let filter = me.filter;
        if (filter) {
            proxy.extraParams = { filter: filter };
        }

        me.store.setProxy(proxy);

        let nodename = me.nodename;
        let queuename = me.queuename;
        me.pendingLoad = true;
        me.store.load(function () {
            me.pendingLoad = false;
            if (me.nodename !== nodename || me.filter !== filter || me.queuename !== queuename) {
                setTimeout(function () {
                    me.updateProxy();
                }, 100);
            }
        });
    },

    setFilter: function (filter) {
        this.lookupReference('filter').setValue(filter);
    },

    setNodename: function (nodename) {
        let me = this;
        if (nodename === me.nodename) {
            return; // nothing to do
        }
        me.nodename = nodename;

        me.updateProxy();
    },

    setQueueName: function (queuename) {
        let me = this;

        me.queuename = queuename;

        me.updateProxy();
    },
});
Ext.define('PMG.QueueAdministration', {
    extend: 'Ext.tab.Panel',
    alias: 'widget.pmgQueueAdministration',

    title: gettext('Queue Administration'),

    border: false,
    defaults: { border: false },

    controller: {
        xclass: 'Ext.app.ViewController',

        onSelect: function (grid, rec) {
            var view = this.getView();

            var domain = rec.data.domain;

            var mailq = this.lookupReference('mailq');

            if (domain === 'TOTAL') {
                mailq.setFilter('');
            } else {
                mailq.setFilter('@' + rec.data.domain);
            }

            view.setActiveTab(mailq);
        },

        control: {
            pmgPostfixQShape: {
                itemdblclick: 'onSelect',
            },
        },
    },

    items: [
        {
            title: gettext('Summary'),
            nodename: Proxmox.NodeName,
            itemId: 'qshape',
            xtype: 'pmgPostfixQShape',
        },
        {
            title: gettext('Deferred Mail'),
            nodename: Proxmox.NodeName,
            reference: 'mailq',
            itemId: 'mailqueue',
            xtype: 'pmgPostfixMailQueue',
        },
    ],
});
Ext.define('pmg-cluster', {
    extend: 'Ext.data.Model',
    fields: [
        'type',
        'name',
        'ip',
        'hostrsapubkey',
        'rootrsapubkey',
        'fingerprint',
        { type: 'integer', name: 'cid' },
        { type: 'boolean', name: 'insync' },
        'memory',
        'loadavg',
        'uptime',
        'rootfs',
        'conn_error',
        'level',
        {
            type: 'number',
            name: 'memory_per',
            calculate: function (data) {
                var mem = data.memory;
                return Ext.isObject(mem) ? mem.used / mem.total : 0;
            },
        },
        {
            type: 'number',
            name: 'rootfs_per',
            calculate: function (data) {
                var du = data.rootfs;
                return Ext.isObject(du) ? du.used / du.total : 0;
            },
        },
    ],
    proxy: {
        type: 'proxmox',
        url: '/api2/json/config/cluster/status',
    },
    idProperty: 'cid',
});

Ext.define('PMG.ClusterJoinNodeWindow', {
    extend: 'Proxmox.window.Edit',
    xtype: 'pmgClusterJoinNodeWindow',
    onlineHelp: 'pmgcm_join',

    title: gettext('Cluster Join'),

    width: 800,

    method: 'POST',

    url: '/config/cluster/join',

    items: [
        {
            xtype: 'textfield',
            fieldLabel: 'IP Address',
            name: 'master_ip',
        },
        {
            xtype: 'textfield',
            inputType: 'password',
            fieldLabel: gettext('Password'),
            name: 'password',
        },
        {
            xtype: 'textfield',
            fieldLabel: gettext('Fingerprint'),
            name: 'fingerprint',
        },
    ],
});

Ext.define('PMG.ClusterAddNodeWindow', {
    extend: 'Ext.window.Window',
    xtype: 'pmgClusterAddNodeWindow',
    mixins: ['Proxmox.Mixin.CBind'],

    width: 800,

    modal: true,

    title: gettext('Cluster Join') + ' : ' + gettext('Information'),

    ipAddress: undefined,

    fingerprint: undefined,

    items: [
        {
            xtype: 'component',
            border: false,
            padding: '10 10 10 10',
            html: gettext(
                "Please use the 'Join' button on the node you want to add, using the following IP address and fingerprint.",
            ),
        },
        {
            xtype: 'container',
            layout: 'form',
            border: false,
            padding: '0 10 10 10',
            items: [
                {
                    xtype: 'textfield',
                    fieldLabel: gettext('IP Address'),
                    cbind: { value: '{ipAddress}' },
                    editable: false,
                },
                {
                    xtype: 'textfield',
                    fieldLabel: gettext('Fingerprint'),
                    cbind: { value: '{fingerprint}' },
                    editable: false,
                },
            ],
        },
    ],
});

Ext.define('PMG.ClusterAdministration', {
    extend: 'Ext.tab.Panel',
    xtype: 'pmgClusterAdministration',

    title: gettext('Cluster Administration'),

    border: false,
    defaults: { border: false },

    viewModel: {
        parent: null,
        data: {
            nodecount: 0,
            master: null,
        },
    },

    items: [
        {
            xtype: 'grid',
            title: gettext('Nodes'),
            controller: {
                xclass: 'Ext.app.ViewController',

                init: function (view) {
                    view.store.on('load', this.onLoad, this);
                    Proxmox.Utils.monStoreErrors(view, view.getStore(), true);
                },

                onLoad: function (store, records, success) {
                    var vm = this.getViewModel();
                    if (!success || !records) {
                        return;
                    }
                    vm.set('nodecount', records.length);

                    var master = null;
                    Ext.Array.each(records, function (ni) {
                        if (ni.data.type === 'master') {
                            master = ni;
                        }
                    });
                    vm.set('master', master);
                },

                onCreate: function () {
                    var view = this.getView();

                    Proxmox.Utils.API2Request({
                        url: '/config/cluster/create',
                        method: 'POST',
                        waitMsgTarget: view,
                        failure: function (response, opts) {
                            Ext.Msg.alert(gettext('Error'), response.htmlStatus);
                        },
                        success: function (response, options) {
                            var upid = response.result.data;
                            var win = Ext.create('Proxmox.window.TaskProgress', { upid: upid });
                            win.show();
                            win.on('destroy', function () {
                                view.store.load();
                            });
                        },
                    });
                },

                onJoin: function () {
                    var win = Ext.create('PMG.ClusterJoinNodeWindow', {});
                    win.show();
                    win.on('destroy', function () {
                        // fixme: logout
                    });
                },

                onAdd: function () {
                    var vm = this.getViewModel();

                    var win = Ext.create('PMG.ClusterAddNodeWindow', {
                        ipAddress: vm.get('master').get('ip'),
                        fingerprint: vm.get('master').get('fingerprint'),
                    });

                    win.show();
                },
            },
            store: {
                autoLoad: true,
                model: 'pmg-cluster',
                sorters: ['cid'],
            },
            tbar: [
                {
                    text: gettext('Create'),
                    reference: 'createButton',
                    handler: 'onCreate',
                    bind: {
                        disabled: '{nodecount}',
                    },
                },
                {
                    text: gettext('Add'),
                    reference: 'addButton',
                    handler: 'onAdd',
                    bind: {
                        disabled: '{!master}',
                    },
                },
                {
                    text: gettext('Join'),
                    reference: 'joinButton',
                    handler: 'onJoin',
                    bind: {
                        disabled: '{nodecount}',
                    },
                },
            ],
            columns: [
                {
                    header: gettext('Node'),
                    width: 150,
                    dataIndex: 'name',
                },
                {
                    header: gettext('Role'),
                    width: 100,
                    dataIndex: 'type',
                },
                {
                    header: gettext('ID'),
                    width: 80,
                    dataIndex: 'cid',
                },
                {
                    header: gettext('IP'),
                    width: 150,
                    dataIndex: 'ip',
                },
                {
                    header: gettext('State'),
                    width: 100,
                    renderer: function (value, metaData, record) {
                        var d = record.data;
                        var state = 'active';
                        if (!d.insync) {
                            state = 'syncing';
                        }
                        if (d.conn_error) {
                            metaData.tdCls = 'x-form-invalid-field';
                            let html = '<p>' + Ext.htmlEncode(d.conn_error) + '</p>';
                            html = html.replace(/\n/g, '<br>');
                            metaData.tdAttr =
                                'data-qwidth=600 data-qtitle="ERROR" data-qtip="' +
                                html.replace(/"/g, '&quot;') +
                                '"';
                            state = 'error';
                        }
                        return state;
                    },
                    dataIndex: 'insync',
                },
                {
                    header: gettext('Subscription'),
                    width: 120,
                    renderer: Proxmox.Utils.format_subscription_level,
                    dataIndex: 'level',
                },
                {
                    header: gettext('Uptime'),
                    width: 150,
                    renderer: Proxmox.Utils.render_uptime,
                    dataIndex: 'uptime',
                },
                {
                    header: gettext('Load average'),
                    renderer: function (value) {
                        if (Ext.isDefined(value)) {
                            if (Ext.isArray(value)) {
                                return value[0];
                            }
                            return value.toString();
                        }
                        return '';
                    },
                    dataIndex: 'loadavg',
                },
                {
                    xtype: 'widgetcolumn',
                    widget: {
                        xtype: 'progressbarwidget',
                        textTpl: '{value:percent}',
                    },
                    header: gettext('RAM usage'),
                    dataIndex: 'memory_per',
                },
                {
                    xtype: 'widgetcolumn',
                    widget: {
                        xtype: 'progressbarwidget',
                        textTpl: '{value:percent}',
                    },
                    header: gettext('HD space'),
                    dataIndex: 'rootfs_per',
                },
            ],
        },
    ],
});
/*
 * Base class for all the multitab config panels.
 * Usage: You'd create a subclass of this, and then define your wanted tabs as items like this:
 *
 * items: [{
 *  title: "myTitle",
 *  xytpe: "somextype",
 *  iconCls: 'fa fa-icon',
 *  groups: ['somegroup'],
 *  expandedOnInit: true,
 *  itemId: 'someId'
 * }]
 *
 * this has to be in the declarative syntax, else we  cannot save them for later (so no Ext.create
 * or Ext.apply of an item in the subclass)
 *
 * the groups array expects the itemids of the items which are the parents, which have to come
 * before they are used.
 *
 * If you want following the tree:
 *   Option1
 *   Option2
 *      -> SubOption1
 *         -> SubSubOption1
 *
 * then the suboption1 group array has to look like this:
 *  groups: ['itemid-of-option2']
 *
 * and the subsuboption1 one:
 *  groups: ['itemid-of-option2', 'itemid-of-suboption1']
 *
 * setting the expandedOnInit determines if the item/group is expanded initially (false by default)
 */
Ext.define('PMG.panel.Config', {
    extend: 'Ext.panel.Panel',
    alias: 'widget.pmgPanelConfig',

    viewFilter: undefined, // a filter to pass to that ressource grid

    dockedItems: [
        {
            // this is needed for the overflow handler
            xtype: 'toolbar',
            overflowHandler: 'scroller',
            dock: 'left',
            style: {
                backgroundColor: '#f5f5f5',
                padding: 0,
                margin: 0,
            },
            items: {
                xtype: 'treelist',
                itemId: 'menu',
                ui: 'pve-nav',
                expanderOnly: true,
                expanderFirst: false,
                animation: false,
                singleExpand: false,
                listeners: {
                    selectionchange: function (treeList, selection) {
                        let view = this.up('panel');
                        view.suspendLayout = true;
                        view.activateCard(selection.data.id);
                        view.suspendLayout = false;
                        view.updateLayout();
                    },
                    itemclick: function (treelist, info) {
                        var olditem = treelist.getSelection();
                        var newitem = info.node;

                        // don't select when clicking on the expand arrow, but still want the original behavior
                        if (info.select === false) {
                            return;
                        }

                        // if a different open item is clicked, leave it open, else toggle the clicked item
                        if (olditem.data.id !== newitem.data.id && newitem.data.expanded === true) {
                            info.toggle = false;
                        } else {
                            info.toggle = true;
                        }
                    },
                },
            },
        },
    ],

    firstItem: '',
    layout: 'card',
    border: 0,

    // used for automated test
    selectById: function (cardid) {
        var me = this;

        var root = me.store.getRoot();
        var selection = root.findChild('id', cardid, true);

        if (selection) {
            selection.expand();
            let menu = me.down('#menu');
            menu.setSelection(selection);
            return cardid;
        }
        return null;
    },

    activateCard: function (cardid) {
        var me = this;
        if (me.savedItems[cardid]) {
            let curcard = me.getLayout().getActiveItem();
            me.add(me.savedItems[cardid]);
            if (curcard) {
                me.setActiveItem(cardid);
                me.remove(curcard, true);

                // trigger state change

                let ncard = cardid;
                // Note: '' is alias for first tab. First tab can be 'search' or something else
                if (cardid === me.firstItem) {
                    ncard = '';
                }
                if (me.hstateid) {
                    me.sp.set(me.hstateid, { value: ncard });
                }
            }
        }
    },

    initComponent: function () {
        var me = this;

        var stateid = me.hstateid;

        me.sp = Ext.state.Manager.getProvider();

        var activeTab; // leaving this undefined means items[0] will be the default tab

        if (stateid) {
            let state = me.sp.get(stateid);
            if (state && state.value) {
                // if this tab does not exists, it chooses the first
                activeTab = state.value;
            }
        }

        // include search tab
        me.items = me.items || [];

        me.savedItems = {};
        if (me.items[0]) {
            me.firstItem = me.items[0].itemId;
        }

        me.store = Ext.create('Ext.data.TreeStore', {
            root: {
                expanded: true,
            },
        });
        var root = me.store.getRoot();
        me.items.forEach(function (item) {
            var treeitem = Ext.create('Ext.data.TreeModel', {
                id: item.itemId,
                text: item.title,
                iconCls: item.iconCls,
                leaf: true,
                expanded: item.expandedOnInit,
            });
            item.header = false;
            if (me.savedItems[item.itemId] !== undefined) {
                throw 'itemId already exists, please use another';
            }
            me.savedItems[item.itemId] = item;

            var group;
            var curnode = root;

            // get/create the group items
            while (Ext.isArray(item.groups) && item.groups.length > 0) {
                group = item.groups.shift();

                let child = curnode.findChild('id', group);
                if (child === null) {
                    // did not find the group item so add it where we are
                    break;
                }
                curnode = child;
            }

            // lets see if it already exists
            var node = curnode.findChild('id', item.itemId);

            if (node === null) {
                curnode.appendChild(treeitem); // insert the item
            } else {
                // should not happen!
                throw 'id already exists';
            }
        });

        delete me.items;
        me.defaults = me.defaults || {};
        Ext.apply(me.defaults, {
            viewFilter: me.viewFilter,
            border: 0,
        });

        me.callParent();

        var menu = me.down('#menu');
        var selection = root.findChild('id', activeTab, true) || root.firstChild;
        var node = selection;
        while (node !== root) {
            node.expand();
            node = node.parentNode;
        }
        menu.setStore(me.store);
        menu.setSelection(selection);

        // on a state change select the new item
        const statechange = function (sp, key, newState) {
            // it the state change is for this panel
            if (stateid && key === stateid && newState) {
                // get active item
                let acard = me.getLayout().getActiveItem().itemId;
                // get the itemid of the new value
                let ncard = newState.value || me.firstItem;
                if (ncard && acard !== ncard) {
                    // select the chosen item
                    menu.setSelection(root.findChild('id', ncard, true) || root.firstChild);
                }
            }
        };

        if (stateid) {
            me.mon(me.sp, 'statechange', statechange);
        }
    },
});
Ext.define('PMG.StatTimeSelector', {
    extend: 'Ext.container.Container',
    xtype: 'pmgStatTimeSelector',

    statics: {
        selected_year: undefined,
        selected_month: undefined,
        selected_day: undefined,

        initSelected: function () {
            let today = new Date();
            this.selected_year = today.getFullYear();
            this.selected_month = today.getMonth() + 1;
            this.selected_day = today.getDate();
        },

        getTimeSpan: function () {
            if (this.selected_year === undefined) {
                this.initSelected();
            }
            const year = this.selected_year;
            const month = this.selected_month;
            const day = this.selected_day;

            let starttime, endtime;
            if (!month) {
                starttime = new Date(year, 0);
                endtime = new Date(year + 1, 0);
            } else if (!day) {
                starttime = new Date(year, month - 1);
                endtime = new Date(year, month);
            } else {
                starttime = new Date(year, month - 1, day);
                endtime = new Date(year, month - 1, day + 1);
            }

            return {
                starttime: (starttime.getTime() / 1000).toFixed(0),
                endtime: (endtime.getTime() / 1000).toFixed(0),
            };
        },
    },

    layout: {
        type: 'hbox',
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        updateVisibility: function () {
            let view = this.getView();

            let yearsel = this.lookupReference('yearsel');
            let monthsel = this.lookupReference('monthsel');
            let daysel = this.lookupReference('daysel');

            let year = yearsel.getValue();
            let month = monthsel.getValue();
            daysel.setVisible(month !== 0);
            if (!month) {
                daysel.setValue(0);
            }
            let day = daysel.getValue();

            let statics = Ext.getClass(view);

            statics.selected_year = year;
            statics.selected_month = month;
            statics.selected_day = day;

            let data = statics.getTimeSpan();
            Ext.GlobalEvents.fireEvent('pmgStatTimeSelectorUpdate', data);
        },

        updateMaxDays: function () {
            let year = this.lookup('yearsel').getValue();
            let month = this.lookup('monthsel').getValue();
            // get last day of current month by wrapping back day 0 from next (zero indexed) month
            let maxDays = new Date(year, month, 0).getDate();
            this.lookup('daysel')
                .getStore()
                .setFilters([
                    {
                        property: 'day',
                        operator: '<=',
                        value: maxDays,
                    },
                ]);
        },

        onSelect: function () {
            this.updateMaxDays();
            this.updateVisibility();
        },

        init: function (view) {
            let statics = Ext.getClass(view);

            let yearsel = this.lookupReference('yearsel');
            let monthsel = this.lookupReference('monthsel');
            let daysel = this.lookupReference('daysel');

            yearsel.setValue(statics.selected_year);
            monthsel.setValue(statics.selected_month);
            daysel.setValue(statics.selected_month ? statics.selected_day : 0);

            this.updateVisibility();
        },
    },

    items: [
        {
            xtype: 'combobox',
            reference: 'yearsel',
            store: {
                fields: ['year'],
                data: (function () {
                    let today = new Date();
                    let year = today.getFullYear();
                    return [{ year: year }, { year: year - 1 }, { year: year - 2 }];
                })(),
            },
            listeners: { select: 'onSelect' },
            value: new Date().getFullYear(),
            queryMode: 'local',
            displayField: 'year',
            editable: false,
            valueField: 'year',
        },
        {
            xtype: 'combobox',
            reference: 'monthsel',
            store: {
                fields: ['month', 'name'],
                data: (function () {
                    let i;
                    let data = [{ month: 0, name: gettext('Whole year') }];
                    for (i = 1; i <= 12; i++) {
                        data.push({ month: i, name: Ext.Date.monthNames[i - 1] });
                    }
                    return data;
                })(),
            },
            listeners: { select: 'onSelect' },
            queryMode: 'local',
            displayField: 'name',
            editable: false,
            valueField: 'month',
        },
        {
            xtype: 'combobox',
            reference: 'daysel',
            store: {
                fields: ['day', 'name'],
                data: (function () {
                    let i;
                    let data = [{ day: 0, name: gettext('Whole month') }];
                    for (i = 1; i <= 31; i++) {
                        data.push({ day: i, name: i });
                    }
                    return data;
                })(),
            },
            listeners: { select: 'onSelect' },
            queryMode: 'local',
            displayField: 'name',
            editable: false,
            valueField: 'day',
        },
    ],
});
Ext.define('PMG.data.StatStore', {
    extend: 'Ext.data.Store',
    alias: 'store.pmgStatStore',

    autoDestroy: true,

    staturl: undefined,

    includeTimeSpan: false,

    setUrl: function (url, extraparam) {
        var me = this;

        me.proxy.abort(); // abort pending requests

        me.staturl = url;
        me.proxy.extraParams = {};
        if (extraparam !== undefined) {
            me.proxy.extraParams = extraparam;
        }

        me.setData([]);
    },

    reload: function () {
        var me = this;

        me.proxy.abort(); // abort pending requests

        if (me.staturl === undefined) {
            me.proxy.extraParams = {};
            me.setData([]);
            return;
        }

        var ts = PMG.StatTimeSelector.getTimeSpan();

        var last = me.proxy.extraParams;

        if (last.starttime === ts.starttime && last.endtime === ts.endtime) {
            return; // avoid repeated loads
        }

        me.proxy.url = me.staturl;
        Ext.apply(me.proxy.extraParams, {
            starttime: ts.starttime,
            endtime: ts.endtime,
        });

        var timespan = 3600;
        if (me.includeTimeSpan) {
            let period = ts.endtime - ts.starttime;
            if (period <= 86400 * 7) {
                timespan = 3600;
            } else {
                timespan = 3600 * 24;
            }
            me.proxy.extraParams.timespan = timespan;
        }

        me.load();
    },

    proxy: {
        type: 'proxmox',
    },

    autoReload: true,

    constructor: function (config) {
        var me = this;

        config = config || {};

        me.mon(
            Ext.GlobalEvents,
            'pmgStatTimeSelectorUpdate',
            function () {
                if (me.autoReload) {
                    me.reload();
                }
            },
            me,
        );

        me.callParent([config]);

        me.reload();
    },
});
Ext.define('PMG.MailStatGrid', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgMailStatGrid',

    disableSelection: true,
    hideHeaders: true,

    store: {
        fields: ['name', 'value', 'percentage'],
    },

    columns: [
        {
            flex: 1,
            dataIndex: 'name',
        },
        {
            width: 150,
            dataIndex: 'value',
        },
        {
            width: 300,

            xtype: 'widgetcolumn',
            dataIndex: 'percentage',
            widget: {
                xtype: 'progressbarwidget',
                textTpl: ['{percent:number("0")}%'],
            },

            onWidgetAttach: function (column, widget, rec) {
                if (rec.data.percentage === undefined) {
                    widget.setStyle('visibility: hidden');
                } else {
                    widget.setStyle('visibility: visible');
                }
            },
        },
    ],
});
Ext.define('PMG.VirusCharts', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgVirusCharts',

    title: gettext('Statistics') + ': ' + gettext('Virus Charts'),

    border: false,
    disableSelection: true,

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    emptyText: gettext('No data in database'),
    viewConfig: {
        deferEmptyText: false,
    },

    store: {
        xclass: 'PMG.data.StatStore',
        fields: ['name', 'count'],
        staturl: '/api2/json/statistics/virus',
    },

    columns: [
        {
            header: gettext('Name'),
            flex: 1,
            dataIndex: 'name',
        },
        {
            header: gettext('Count'),
            width: 150,
            dataIndex: 'count',
        },
    ],

    initComponent: function () {
        var me = this;

        me.callParent();

        Proxmox.Utils.monStoreErrors(me, me.store);
    },
});
Ext.define('PMG.SpamScoreDistribution', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgSpamScoreDistribution',

    disableSelection: true,
    border: false,

    title: gettext('Statistics') + ': ' + gettext('Spam Scores'),

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    store: {
        xclass: 'PMG.data.StatStore',
        staturl: '/api2/json/statistics/spamscores',
        fields: [
            'count',
            'ratio',
            {
                type: 'string',
                name: 'label',
                convert: function (v, rec) {
                    if (rec.data.level >= 10) {
                        return PMG.Utils.scoreText + ' >= 10';
                    } else {
                        return PMG.Utils.scoreText + ' ' + rec.data.level.toString();
                    }
                },
            },
        ],
    },

    columns: [
        {
            header: PMG.Utils.scoreText,
            flex: 1,
            dataIndex: 'label',
        },
        {
            header: gettext('Count'),
            width: 150,
            dataIndex: 'count',
        },
        {
            header: gettext('Percentage'),
            width: 300,

            xtype: 'widgetcolumn',
            dataIndex: 'ratio',
            widget: {
                xtype: 'progressbarwidget',
                textTpl: ['{percent:number("0")}%'],
            },
        },
    ],
});
Ext.define('PMG.GeneralMailStatistics', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgGeneralMailStatistics',

    scrollable: true,

    bodyPadding: '10 0 0 0',
    border: false,
    defaults: {
        width: 700,
        padding: '0 0 10 10',
    },

    layout: 'column',

    title: gettext('Statistics') + ': ' + gettext('Mail'),

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    getInData: function (data) {
        var res = [];

        res.push({
            name: gettext('Incoming Mails'),
            value: data.count_in,
            percentage: 1,
        });

        res.push({
            name: gettext('Junk Mails'),
            value: data.junk_in,
            percentage: data.junk_in / data.count_in,
        });

        res.push({
            name: gettext('Greylisted Mails'),
            value: data.glcount,
            percentage: data.glcount / data.count_in,
        });

        res.push({
            name: gettext('Spam Mails'),
            value: data.spamcount_in,
            percentage: data.spamcount_in / data.count_in,
        });

        res.push({
            name: gettext('SPF rejects'),
            value: data.spfcount,
            percentage: data.spfcount / data.count_in,
        });

        res.push({
            name: gettext('Bounces'),
            value: data.bounces_in,
            percentage: data.bounces_in / data.count_in,
        });

        res.push({
            name: gettext('Virus Mails'),
            value: data.viruscount_in,
            percentage: data.viruscount_in / data.count_in,
        });

        return res;
    },

    getOutData: function (data) {
        var res = [];

        res.push({
            name: gettext('Outgoing Mails'),
            value: data.count_out,
            percentage: 1,
        });

        res.push({
            name: gettext('Bounces'),
            value: data.bounces_out,
            percentage: data.bounces_out / data.count_out,
        });

        res.push({
            name: gettext('Virus Mails'),
            value: data.viruscount_out,
            percentage: data.viruscount_out / data.count_out,
        });

        return res;
    },

    getGeneralData: function (data) {
        var res = [];

        res.push({
            name: gettext('Total Mails'),
            value: data.count,
            percentage: 1,
        });

        res.push({
            name: gettext('Incoming Mails'),
            value: data.count_in,
            percentage: data.count_in / data.count,
        });

        res.push({
            name: gettext('Outgoing Mails'),
            value: data.count_out,
            percentage: data.count_out / data.count,
        });

        res.push({
            name: gettext('Virus Outbreaks'),
            value: data.viruscount_out,
        });

        res.push({
            name: gettext('Avg. Mail Processing Time'),
            value: Ext.String.format(gettext('{0} seconds'), Ext.Number.toFixed(data.avptime, 2)),
        });

        res.push({
            name: gettext('Incoming Mail Traffic'),
            value: Ext.Number.toFixed(data.bytes_in / (1024 * 1024), 2) + ' MiB',
        });

        res.push({
            name: gettext('Outgoing Mail Traffic'),
            value: Ext.Number.toFixed(data.bytes_out / (1024 * 1024), 2) + ' MiB',
        });
        return res;
    },

    initComponent: function () {
        var me = this;

        var countstore = Ext.create('PMG.data.StatStore', {
            includeTimeSpan: true,
            staturl: '/api2/json/statistics/mailcount',
            fields: [
                { type: 'integer', name: 'count' },
                { type: 'integer', name: 'count_in' },
                { type: 'integer', name: 'count_out' },
                { type: 'integer', name: 'spamcount_in' },
                { type: 'integer', name: 'spamcount_out' },
                { type: 'integer', name: 'viruscount_in' },
                { type: 'integer', name: 'viruscount_out' },
                { type: 'integer', name: 'bounces_in' },
                { type: 'integer', name: 'bounces_out' },
                { type: 'date', dateFormat: 'timestamp', name: 'time' },
            ],
        });

        var totalgrid = Ext.createWidget('pmgMailStatGrid', {
            dockedItems: [
                {
                    dock: 'top',
                    title: gettext('Total Mail Count'),
                    xtype: 'proxmoxRRDChart',
                    fields: ['count', 'count_in', 'count_out'],
                    fieldTitles: [
                        gettext('Total Mail Count'),
                        gettext('Incoming Mails'),
                        gettext('Outgoing Mails'),
                    ],
                    store: countstore,
                },
            ],
        });

        var ingrid = Ext.createWidget('pmgMailStatGrid', {
            dockedItems: [
                {
                    dock: 'top',
                    title: gettext('Incoming Mails'),
                    xtype: 'proxmoxRRDChart',
                    fields: ['count_in', 'spamcount_in', 'viruscount_in', 'bounces_in'],
                    fieldTitles: [
                        gettext('Incoming Mails'),
                        gettext('Junk Mails'),
                        gettext('Virus Mails'),
                        gettext('Bounces'),
                    ],
                    store: countstore,
                },
            ],
        });

        var outgrid = Ext.createWidget('pmgMailStatGrid', {
            dockedItems: [
                {
                    dock: 'top',
                    title: gettext('Outgoing Mails'),
                    xtype: 'proxmoxRRDChart',
                    fields: ['count_out', 'viruscount_out', 'bounces_out'],
                    fieldTitles: [
                        gettext('Outgoing Mails'),
                        gettext('Virus Mails'),
                        gettext('Bounces'),
                    ],
                    store: countstore,
                },
            ],
        });

        var infostore = Ext.create('PMG.data.StatStore', {
            staturl: '/api2/json/statistics/mail',
            fields: ['name', 'value', 'percentage'],
            listeners: {
                load: function (store, records, success) {
                    if (!success || records.length <= 0) {
                        return;
                    }
                    var data = me.getGeneralData(records[0].data);
                    totalgrid.store.setData(data);
                    data = me.getInData(records[0].data);
                    ingrid.store.setData(data);
                    data = me.getOutData(records[0].data);
                    outgrid.store.setData(data);
                },
            },
        });

        me.items = [totalgrid, ingrid, outgrid];

        me.callParent();

        me.on('destroy', infostore.destroy, infostore);
        me.on('destroy', countstore.destroy, countstore);
    },
});
Ext.define('PMG.RBLStatistics', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgRBLStatistics',

    scrollable: true,
    border: false,

    bodyPadding: '10 0 10 10',

    title: gettext('Statistics') + ': ' + gettext('Postscreen'),

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    items: [
        {
            title: gettext('Rejects'),
            xtype: 'proxmoxRRDChart',
            fields: ['rbl_rejects', 'pregreet_rejects'],
            fieldTitles: ['RBL', 'PREGREET'],
            store: {
                type: 'pmgStatStore',
                includeTimeSpan: true,
                staturl: '/api2/json/statistics/rejectcount',
                fields: [
                    { type: 'integer', name: 'rbl_rejects' },
                    { type: 'integer', name: 'pregreet_rejects' },
                    { type: 'date', dateFormat: 'timestamp', name: 'time' },
                ],
            },
        },
    ],
});
Ext.define('PMG.DomainStatistics', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgDomainStatistics',

    title: gettext('Statistics') + ': ' + gettext('Domain'),

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    layout: 'fit',
    border: false,

    initComponent: function () {
        var me = this;

        var fields = [
            'domain',
            { type: 'integer', name: 'count_in' },
            { type: 'integer', name: 'count_out' },
            { type: 'integer', name: 'spamcount_in' },
            { type: 'integer', name: 'spamcount_out' },
            { type: 'integer', name: 'viruscount_in' },
            { type: 'integer', name: 'viruscount_out' },
            { type: 'number', name: 'bytes_in' },
            { type: 'number', name: 'bytes_out' },
        ];

        var store = Ext.create('PMG.data.StatStore', {
            staturl: '/api2/json/statistics/domains',
            fields: fields,
        });

        var store_in = Ext.create('Ext.data.ArrayStore', {
            fields: fields,
            filters: [
                function (item) {
                    return item.data.count_in > 0;
                },
            ],
        });

        var store_out = Ext.create('Ext.data.ArrayStore', {
            fields: fields,
            filters: [
                function (item) {
                    return item.data.count_out > 0;
                },
            ],
        });

        store.on('load', function (s, records, successful) {
            if (!successful) {
                store_in.setData([]);
                store_out.setData([]);
            } else {
                store_in.setData(records);
                store_out.setData(records);
            }
        });

        var render_domain = function (v) {
            return v === '' ? '--- EMPTY ADDRESS ---' : Ext.htmlEncode(v);
        };

        me.items = [
            {
                xtype: 'tabpanel',
                border: false,
                items: [
                    {
                        xtype: 'grid',
                        title: gettext('Incoming'),
                        border: false,
                        disableSelection: true,
                        store: store_in,
                        emptyText: gettext('No data in database'),
                        viewConfig: {
                            deferEmptyText: false,
                        },
                        columns: [
                            {
                                text: gettext('Domain') + ' (' + gettext('Receiver') + ')',
                                flex: 1,
                                renderer: render_domain,
                                dataIndex: 'domain',
                            },
                            {
                                text: gettext('Traffic') + ' (MB)',
                                dataIndex: 'bytes_in',
                                renderer: function (v) {
                                    return Ext.Number.toFixed(v / (1024 * 1024), 2);
                                },
                            },
                            {
                                text: gettext('Count'),
                                columns: [
                                    {
                                        text: gettext('Mail'),
                                        dataIndex: 'count_in',
                                    },
                                    {
                                        header: gettext('Virus'),
                                        dataIndex: 'viruscount_in',
                                    },
                                    {
                                        header: gettext('Spam'),
                                        dataIndex: 'spamcount_in',
                                    },
                                ],
                            },
                        ],
                    },
                    {
                        xtype: 'grid',
                        title: gettext('Outgoing'),
                        border: false,
                        disableSelection: true,
                        store: store_out,
                        emptyText: gettext('No data in database'),
                        viewConfig: {
                            deferEmptyText: false,
                        },
                        columns: [
                            {
                                text: gettext('Domain') + ' (' + gettext('Receiver') + ')',
                                flex: 1,
                                renderer: render_domain,
                                dataIndex: 'domain',
                            },
                            {
                                text: gettext('Traffic') + ' (MB)',
                                dataIndex: 'bytes_out',
                                renderer: function (v) {
                                    return Ext.Number.toFixed(v / (1024 * 1024), 2);
                                },
                            },
                            {
                                text: gettext('Count'),
                                columns: [
                                    {
                                        text: gettext('Mail'),
                                        dataIndex: 'count_out',
                                    },
                                    {
                                        header: gettext('Virus'),
                                        dataIndex: 'viruscount_out',
                                    },
                                ],
                            },
                        ],
                    },
                ],
            },
        ];

        me.callParent();

        Proxmox.Utils.monStoreErrors(me, store);

        me.on('destroy', store.destroy, store);
    },
});
Ext.define('PMG.SenderDetails', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgSenderDetails',

    dockedItems: [
        {
            dock: 'top',
            xtype: 'panel',
            itemId: 'info',
            bodyPadding: 10,
            html: gettext('Please select a sender.'),
        },
    ],

    disableSelection: true,

    plugins: 'gridfilters',

    setUrl: function (url, extraparam, title) {
        var me = this;

        me.store.setUrl(url, extraparam);
        me.store.setRemoteFilter(url !== undefined);
        Proxmox.Utils.setErrorMask(me, false);
        me.store.reload();

        var infopanel = me.getComponent('info');
        if (title) {
            infopanel.update(title);
        } else {
            infopanel.update(gettext('Please select a sender.'));
        }
    },

    store: {
        type: 'pmgStatStore',
        autoReload: false,
        remoteSort: true,
        remoteFilter: false, // enabled dynamically
        fields: [
            'receiver',
            'virusinfo',
            { type: 'integer', name: 'bytes' },
            { type: 'boolean', name: 'blocked' },
            { type: 'integer', name: 'spamlevel' },
            { type: 'date', dateFormat: 'timestamp', name: 'time' },
        ],
        proxy: {
            type: 'pmgfilterproxy',
            filterId: 'x-gridfilter-receiver',
            sortParam: 'orderby',
        },
        sorters: [
            {
                property: 'time',
                direction: 'ASC',
            },
        ],
    },

    columns: [
        {
            text: gettext('Receiver'),
            renderer: Ext.htmlEncode,
            flex: 1,
            filter: { type: 'string' },
            dataIndex: 'receiver',
        },
        {
            header: gettext('Size') + ' (KB)',
            renderer: function (v) {
                return Ext.Number.toFixed(v / 1024, 0);
            },
            dataIndex: 'bytes',
        },
        {
            xtype: 'datecolumn',
            header: gettext('Date'),
            format: 'Y-m-d',
            dataIndex: 'time',
        },
        {
            xtype: 'datecolumn',
            header: gettext('Time'),
            format: 'H:i:s',
            dataIndex: 'time',
        },
    ],

    initComponent: function () {
        var me = this;
        me.callParent();

        Proxmox.Utils.monStoreErrors(me, me.store, true);
    },
});

Ext.define('PMG.SenderList', {
    extend: 'Ext.grid.GridPanel',
    alias: 'widget.pmgSenderList',

    title: gettext('Statistics') + ': ' + gettext('Sender') + ' (' + gettext('Outgoing') + ')',

    multiColumnSort: true,
    plugins: 'gridfilters',

    emptyText: gettext('No data in database'),
    viewConfig: {
        deferEmptyText: false,
    },

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    store: {
        type: 'pmgStatStore',
        staturl: '/api2/json/statistics/sender',
        remoteSort: true,
        remoteFilter: true,
        fields: [
            'sender',
            { type: 'integer', name: 'count' },
            { type: 'integer', name: 'bytes' },
            { type: 'integer', name: 'viruscount' },
        ],
        proxy: {
            type: 'pmgfilterproxy',
            sortParam: 'orderby',
            filterId: 'x-gridfilter-sender',
        },
        sorters: [
            {
                property: 'count',
                direction: 'DESC',
            },
            {
                property: 'bytes',
                direction: 'DESC',
            },
            {
                property: 'sender',
                direction: 'ASC',
            },
        ],
    },

    columns: [
        {
            text: gettext('Sender'),
            flex: 1,
            renderer: Ext.htmlEncode,
            dataIndex: 'sender',
            filter: {
                type: 'string',
                itemDefaults: {
                    // any Ext.form.field.Text configs accepted
                },
            },
        },
        {
            text: gettext('Count'),
            columns: [
                {
                    text: gettext('Mail'),
                    dataIndex: 'count',
                },
                {
                    header: gettext('Virus'),
                    dataIndex: 'viruscount',
                },
            ],
        },
        {
            text: gettext('Size') + ' (KB)',
            dataIndex: 'bytes',
            renderer: function (v) {
                return Ext.Number.toFixed(v / 1024, 0);
            },
        },
    ],

    initComponent: function () {
        var me = this;
        me.callParent();

        Proxmox.Utils.monStoreErrors(me, me.store, true);
    },
});

Ext.define('PMG.SenderStatistics', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgSenderStatistics',

    layout: 'border',
    border: false,
    defaults: {
        border: false,
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        selectionChange: function (grid, selected, eOpts) {
            var details = this.lookupReference('details');
            if (selected.length > 0) {
                let sender = selected[0].data.sender;
                let extraparam = { address: sender, type: 'sender' };
                let url = '/api2/json/statistics/detail';
                details.setUrl(
                    url,
                    extraparam,
                    '<b>' + gettext('Sender') + ':</b> ' + Ext.htmlEncode(sender),
                );
            } else {
                details.setUrl();
            }
        },
    },

    items: [
        {
            xtype: 'pmgSenderList',
            multiColumnSort: true,
            region: 'center',
            layout: 'fit',
            flex: 1,

            listeners: { selectionchange: 'selectionChange' },
        },
        {
            xtype: 'pmgSenderDetails',
            region: 'east',
            reference: 'details',
            split: true,
            flex: 1,
        },
    ],
});
Ext.define('PMG.ReceiverDetails', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgReceiverDetails',

    dockedItems: [
        {
            dock: 'top',
            xtype: 'panel',
            itemId: 'info',
            bodyPadding: 10,
            html: gettext('Please select a receiver.'),
        },
    ],

    disableSelection: true,

    plugins: 'gridfilters',

    setUrl: function (url, extraparam, title) {
        var me = this;

        me.store.setUrl(url, extraparam);
        me.store.setRemoteFilter(url !== undefined);
        Proxmox.Utils.setErrorMask(me, false);
        me.store.reload();

        var infopanel = me.getComponent('info');
        if (title) {
            infopanel.update(title);
        } else {
            infopanel.update(gettext('Please select a sender.'));
        }
    },

    store: {
        type: 'pmgStatStore',
        autoReload: false,
        remoteSort: true,
        remoteFilter: false, // enabled dynamically
        fields: [
            'sender',
            'virusinfo',
            { type: 'integer', name: 'bytes' },
            { type: 'boolean', name: 'blocked' },
            { type: 'integer', name: 'spamlevel' },
            { type: 'date', dateFormat: 'timestamp', name: 'time' },
        ],
        proxy: {
            type: 'pmgfilterproxy',
            filterId: 'x-gridfilter-sender',
            sortParam: 'orderby',
        },
        sorters: [
            {
                property: 'time',
                direction: 'ASC',
            },
        ],
    },

    columns: [
        {
            text: gettext('Sender'),
            renderer: Ext.htmlEncode,
            flex: 1,
            filter: { type: 'string' },
            dataIndex: 'sender',
        },
        {
            header: gettext('Size') + ' (KB)',
            renderer: function (v) {
                return Ext.Number.toFixed(v / 1024, 0);
            },
            dataIndex: 'bytes',
        },
        {
            xtype: 'datecolumn',
            header: gettext('Date'),
            format: 'Y-m-d',
            dataIndex: 'time',
        },
        {
            xtype: 'datecolumn',
            header: gettext('Time'),
            format: 'H:i:s',
            dataIndex: 'time',
        },
        {
            header: gettext('Virus info'),
            dataIndex: 'virusinfo',
        },
        {
            header: gettext('Score'),
            dataIndex: 'spamlevel',
        },
    ],

    initComponent: function () {
        var me = this;
        me.callParent();

        Proxmox.Utils.monStoreErrors(me, me.store, true);
    },
});

Ext.define('PMG.ReceiverList', {
    extend: 'Ext.grid.GridPanel',
    alias: 'widget.pmgReceiverList',

    title: gettext('Statistics') + ': ' + gettext('Receiver') + ' (' + gettext('Incoming') + ')',

    multiColumnSort: true,
    plugins: 'gridfilters',

    emptyText: gettext('No data in database'),
    viewConfig: {
        deferEmptyText: false,
    },

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    store: {
        type: 'pmgStatStore',
        staturl: '/api2/json/statistics/receiver',
        remoteSort: true,
        remoteFilter: true,
        fields: [
            'receiver',
            { type: 'integer', name: 'count' },
            { type: 'integer', name: 'bytes' },
            { type: 'integer', name: 'viruscount' },
        ],
        proxy: {
            type: 'pmgfilterproxy',
            sortParam: 'orderby',
            filterId: 'x-gridfilter-receiver',
        },
        sorters: [
            {
                property: 'count',
                direction: 'DESC',
            },
            {
                property: 'bytes',
                direction: 'DESC',
            },
            {
                property: 'receiver',
                direction: 'ASC',
            },
        ],
    },

    columns: [
        {
            text: gettext('Receiver'),
            flex: 1,
            renderer: Ext.htmlEncode,
            dataIndex: 'receiver',
            filter: {
                type: 'string',
                itemDefaults: {
                    // any Ext.form.field.Text configs accepted
                },
            },
        },
        {
            text: gettext('Count'),
            columns: [
                {
                    text: gettext('Mail'),
                    dataIndex: 'count',
                },
                {
                    header: gettext('Virus'),
                    dataIndex: 'viruscount',
                },
                {
                    header: gettext('Spam'),
                    dataIndex: 'spamcount',
                },
            ],
        },
        {
            text: gettext('Size') + ' (KB)',
            dataIndex: 'bytes',
            renderer: function (v) {
                return Ext.Number.toFixed(v / 1024, 0);
            },
        },
    ],

    initComponent: function () {
        var me = this;
        me.callParent();

        Proxmox.Utils.monStoreErrors(me, me.store, true);
    },
});

Ext.define('PMG.ReceiverStatistics', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgReceiverStatistics',

    layout: 'border',
    border: false,
    defaults: {
        border: false,
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        selectionChange: function (grid, selected, eOpts) {
            var details = this.lookupReference('details');
            if (selected.length > 0) {
                let url = '/api2/json/statistics/detail';
                let receiver = selected[0].data.receiver;
                let extraparam = { address: receiver, type: 'receiver' };
                details.setUrl(
                    url,
                    extraparam,
                    '<b>' + gettext('Receiver') + ':</b> ' + Ext.htmlEncode(receiver),
                );
            } else {
                details.setUrl();
            }
        },
    },

    items: [
        {
            xtype: 'pmgReceiverList',
            multiColumnSort: true,
            region: 'center',
            layout: 'fit',
            flex: 1,

            listeners: { selectionchange: 'selectionChange' },
        },
        {
            xtype: 'pmgReceiverDetails',
            region: 'east',
            reference: 'details',
            split: true,
            flex: 1,
        },
    ],
});
Ext.define('PMG.ContactDetails', {
    extend: 'Ext.grid.GridPanel',
    xtype: 'pmgContactDetails',

    dockedItems: [
        {
            dock: 'top',
            xtype: 'panel',
            itemId: 'info',
            bodyPadding: 10,
            html: gettext('Please select a contact'),
        },
    ],

    disableSelection: true,

    plugins: 'gridfilters',

    setUrl: function (url, extraparam, title) {
        var me = this;

        me.store.setUrl(url, extraparam);
        me.store.setRemoteFilter(url !== undefined);
        Proxmox.Utils.setErrorMask(me, false);
        me.store.reload();

        var infopanel = me.getComponent('info');
        if (title) {
            infopanel.update(title);
        } else {
            infopanel.update(gettext('Please select a contact'));
        }
    },

    store: {
        type: 'pmgStatStore',
        autoReload: false,
        remoteSort: true,
        remoteFilter: false, // enabled dynamically
        fields: [
            'sender',
            'virusinfo',
            { type: 'integer', name: 'bytes' },
            { type: 'boolean', name: 'blocked' },
            { type: 'integer', name: 'spamlevel' },
            { type: 'date', dateFormat: 'timestamp', name: 'time' },
        ],
        proxy: {
            type: 'pmgfilterproxy',
            filterId: 'x-gridfilter-sender',
            sortParam: 'orderby',
        },
        sorters: [
            {
                property: 'time',
                direction: 'ASC',
            },
        ],
    },

    columns: [
        {
            text: gettext('Sender'),
            renderer: Ext.htmlEncode,
            flex: 1,
            filter: { type: 'string' },
            dataIndex: 'sender',
        },
        {
            header: gettext('Size') + ' (KB)',
            renderer: function (v) {
                return Ext.Number.toFixed(v / 1024, 0);
            },
            dataIndex: 'bytes',
        },
        {
            xtype: 'datecolumn',
            header: gettext('Date'),
            format: 'Y-m-d',
            dataIndex: 'time',
        },
        {
            xtype: 'datecolumn',
            header: gettext('Time'),
            format: 'H:i:s',
            dataIndex: 'time',
        },
    ],

    initComponent: function () {
        var me = this;
        me.callParent();

        Proxmox.Utils.monStoreErrors(me, me.store, true);
    },
});

Ext.define('PMG.ContactList', {
    extend: 'Ext.grid.GridPanel',
    alias: 'widget.pmgContactList',

    title:
        gettext('Statistics') +
        ': ' +
        gettext('Contact') +
        ' (' +
        gettext('Receiver') +
        ', ' +
        gettext('Outgoing') +
        ')',

    multiColumnSort: true,
    plugins: 'gridfilters',

    emptyText: gettext('No data in database'),
    viewConfig: {
        deferEmptyText: false,
    },

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    store: {
        type: 'pmgStatStore',
        staturl: '/api2/json/statistics/contact',
        remoteSort: true,
        remoteFilter: true,
        fields: [
            'contact',
            { type: 'integer', name: 'count' },
            { type: 'integer', name: 'viruscount' },
            { type: 'integer', name: 'bytes' },
        ],
        proxy: {
            type: 'pmgfilterproxy',
            sortParam: 'orderby',
            filterId: 'x-gridfilter-contact',
        },
        sorters: [
            {
                property: 'count',
                direction: 'DESC',
            },
            {
                property: 'bytes',
                direction: 'DESC',
            },
            {
                property: 'contact',
                direction: 'ASC',
            },
        ],
    },

    columns: [
        {
            text: gettext('Contact'),
            flex: 1,
            renderer: Ext.htmlEncode,
            dataIndex: 'contact',
            filter: {
                type: 'string',
                itemDefaults: {
                    // any Ext.form.field.Text configs accepted
                },
            },
        },
        {
            text: gettext('Count'),
            columns: [
                {
                    text: gettext('Mail'),
                    dataIndex: 'count',
                },
                {
                    header: gettext('Virus'),
                    dataIndex: 'viruscount',
                },
            ],
        },
        {
            text: gettext('Size') + ' (KB)',
            dataIndex: 'bytes',
            renderer: function (v) {
                return Ext.Number.toFixed(v / 1024, 0);
            },
        },
    ],

    initComponent: function () {
        var me = this;
        me.callParent();

        Proxmox.Utils.monStoreErrors(me, me.store, true);
    },
});

Ext.define('PMG.ContactStatistics', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgContactStatistics',

    layout: 'border',
    border: false,
    defaults: {
        border: false,
    },

    controller: {
        xclass: 'Ext.app.ViewController',

        selectionChange: function (grid, selected, eOpts) {
            var details = this.lookupReference('details');
            if (selected.length > 0) {
                let contact = selected[0].data.contact;
                let extraparam = { address: contact, type: 'contact' };
                let url = '/api2/json/statistics/detail';
                details.setUrl(
                    url,
                    extraparam,
                    '<b>' + gettext('Contact') + ':</b> ' + Ext.htmlEncode(contact),
                );
            } else {
                details.setUrl();
            }
        },
    },

    items: [
        {
            xtype: 'pmgContactList',
            multiColumnSort: true,
            region: 'center',
            layout: 'fit',
            flex: 1,

            listeners: { selectionchange: 'selectionChange' },
        },
        {
            xtype: 'pmgContactDetails',
            region: 'east',
            reference: 'details',
            split: true,
            flex: 1,
        },
    ],
});
Ext.define('PMG.MailDistChart', {
    extend: 'Ext.chart.CartesianChart',
    xtype: 'pmgMailDistChart',

    width: 770,
    height: 300,
    animation: false,

    axes: [
        {
            type: 'numeric',
            title: gettext('Count'),
            position: 'left',
            grid: true,
            minimum: 0,
        },
        {
            type: 'category',
            position: 'bottom',
            fields: ['index'],
        },
    ],

    checkThemeColors: function () {
        let me = this;
        let rootStyle = getComputedStyle(document.documentElement);

        // get colors
        let background = rootStyle.getPropertyValue('--pwt-panel-background').trim() || '#ffffff';
        let text = rootStyle.getPropertyValue('--pwt-text-color').trim() || '#000000';
        let primary = rootStyle.getPropertyValue('--pwt-chart-primary').trim() || '#000000';
        let gridStroke = rootStyle.getPropertyValue('--pwt-chart-grid-stroke').trim() || '#dddddd';

        // set the colors
        me.setBackground(background);
        me.axes.forEach((axis) => {
            axis.setLabel({ color: text });
            axis.setTitle({ color: text });
            axis.setStyle({ strokeStyle: primary });
            axis.setGrid({ stroke: gridStroke });
        });
        me.redraw();
    },

    initComponent: function () {
        var me = this;

        if (!me.store) {
            throw 'cannot work without store';
        }

        if (!me.field) {
            throw 'cannot work without a field';
        }

        me.callParent();

        me.addSeries({
            type: 'bar',
            xField: 'index',
            yField: me.field,
            tooltip: {
                trackMouse: true,
                renderer: function (tooltip, record, item) {
                    var start = record.get('index');
                    var end = start + 1;
                    tooltip.setHtml(
                        'Time: ' +
                            start.toString() +
                            ' - ' +
                            end.toString() +
                            '<br>' +
                            'Count: ' +
                            record.get(item.field),
                    );
                },
            },
        });

        me.checkThemeColors();

        // switch colors on media query changes
        me.mediaQueryList = window.matchMedia('(prefers-color-scheme: dark)');
        me.themeListener = (e) => {
            me.checkThemeColors();
        };
        me.mediaQueryList.addEventListener('change', me.themeListener);
    },

    doDestroy: function () {
        let me = this;

        me.mediaQueryList.removeEventListener('change', me.themeListener);

        me.callParent();
    },
});

Ext.define('PMG.HourlyMailDistribution', {
    extend: 'Ext.panel.Panel',
    xtype: 'pmgHourlyMailDistribution',

    scrollable: true,
    border: false,

    bodyPadding: '10 0 0 0',
    defaults: {
        width: 700,
        padding: '0 0 10 10',
    },

    layout: 'column',

    title: gettext('Statistics') + ': ' + gettext('Hourly Distribution'),

    tbar: [{ xtype: 'pmgStatTimeSelector' }],

    initComponent: function () {
        var me = this;

        var store = Ext.create('PMG.data.StatStore', {
            staturl: '/api2/json/statistics/maildistribution',
            fields: [
                { type: 'integer', name: 'index' },
                { type: 'integer', name: 'count' },
                { type: 'integer', name: 'count_in' },
                { type: 'integer', name: 'count_out' },
                { type: 'integer', name: 'spamcount_in' },
                { type: 'integer', name: 'spamcount_out' },
                { type: 'integer', name: 'viruscount_in' },
                { type: 'integer', name: 'viruscount_ou' },
                { type: 'integer', name: 'bounces_in' },
                { type: 'integer', name: 'bounces_out' },
            ],
        });

        me.items = [
            {
                xtype: 'pmgMailDistChart',
                title: gettext('Incoming Mails'),
                field: 'count_in',
                store: store,
            },
            {
                xtype: 'pmgMailDistChart',
                title: gettext('Outgoing Mails'),
                field: 'count_out',
                store: store,
            },
        ];

        me.callParent();

        me.on('destroy', store.destroy, store);
    },
});
Ext.define('PMG.menu.SpamContextMenu', {
    extend: 'PMG.menu.QuarantineContextMenu',

    items: [
        {
            text: gettext('Deliver'),
            iconCls: 'fa fa-fw fa-paper-plane-o info-blue',
            action: 'deliver',
            handler: 'callCallback',
        },
        {
            text: gettext('Delete'),
            iconCls: 'fa fa-fw fa-trash-o critical',
            action: 'delete',
            handler: 'callCallback',
        },
        { xtype: 'menuseparator' },
        {
            text: gettext('Welcomelist'),
            iconCls: 'fa fa-fw fa-check',
            action: 'welcomelist',
            handler: 'callCallback',
        },
        {
            text: gettext('Blocklist'),
            iconCls: 'fa fa-fw fa-times',
            action: 'blocklist',
            handler: 'callCallback',
        },
    ],
});
Ext.define('PMG.CertificateConfiguration', {
    extend: 'Ext.tab.Panel',
    alias: 'widget.pmgCertificateConfiguration',

    title: gettext('Certificates'),

    ...PMG.Utils.onlineHelpTool('sysadmin_certificate_management'),

    border: false,
    defaults: { border: false },

    items: [
        {
            xtype: 'pmgCertificatesView',
            itemId: 'certificates',
            iconCls: 'fa fa-certificate',
        },
        {
            xtype: 'pmgACMEConfigView',
            itemId: 'acme',
            iconCls: 'fa fa-file-text',
        },
    ],
});

Ext.define('PMG.CertificateView', {
    extend: 'Ext.panel.Panel',
    alias: 'widget.pmgCertificatesView',

    title: gettext('Certificates'),
    scrollable: 'y',

    items: [
        {
            xtype: 'pmxCertificates',
            border: 0,
            infoUrl: '/nodes/' + Proxmox.NodeName + '/certificates/info',
            uploadButtons: [
                {
                    name: 'API',
                    id: 'pmg-api.pem',
                    url: `/nodes/${Proxmox.NodeName}/certificates/custom/api`,
                    deletable: false,
                    reloadUi: true,
                },
                {
                    name: 'SMTP',
                    id: 'pmg-tls.pem',
                    url: `/nodes/${Proxmox.NodeName}/certificates/custom/smtp`,
                    deletable: true,
                },
            ],
        },
        {
            xtype: 'pmxACMEDomains',
            border: 0,
            url: `/nodes/${Proxmox.NodeName}/config`,
            nodename: Proxmox.NodeName,
            acmeUrl: '/config/acme',
            domainUsages: [
                {
                    usage: 'api',
                    name: 'API',
                    url: `/nodes/${Proxmox.NodeName}/certificates/acme/api`,
                    reloadUi: true,
                },
                {
                    usage: 'smtp',
                    name: 'SMTP',
                    url: `/nodes/${Proxmox.NodeName}/certificates/acme/smtp`,
                },
            ],
        },
    ],
});

Ext.define('PMG.ACMEConfigView', {
    extend: 'Ext.panel.Panel',
    alias: 'widget.pmgACMEConfigView',

    title: gettext('ACME Accounts/Challenges'),

    //onlineHelp: 'sysadmin_certificate_management',

    items: [
        {
            xtype: 'pmxACMEAccounts',
            region: 'north',
            border: false,
            acmeUrl: '/config/acme',
        },
        {
            xtype: 'pmxACMEPluginView',
            region: 'center',
            border: false,
            acmeUrl: '/config/acme',
        },
    ],
});
Ext.define('PMG.window.Settings', {
    extend: 'Ext.window.Window',

    width: '800px',
    title: gettext('My Settings'),
    iconCls: 'fa fa-gear',
    modal: true,
    bodyPadding: 10,
    resizable: false,

    buttons: [
        '->',
        {
            text: gettext('Close'),
            handler: function () {
                this.up('window').close();
            },
        },
    ],

    layout: 'hbox',

    controller: {
        xclass: 'Ext.app.ViewController',

        init: function (view) {
            let me = this;
            let sp = Ext.state.Manager.getProvider();

            let username = sp.get('login-username') || Proxmox.Utils.noneText;
            me.lookupReference('savedUserName').setValue(Ext.String.htmlEncode(username));

            let summarycolumns = sp.get('summarycolumns', 'auto');
            me.lookup('summarycolumns').setValue(summarycolumns);

            let settings = ['fontSize', 'fontFamily', 'letterSpacing', 'lineHeight'];
            settings.forEach(function (setting) {
                let val = localStorage.getItem('pve-xterm-' + setting);
                if (val !== undefined && val !== null) {
                    let field = me.lookup(setting);
                    field.setValue(val);
                    field.resetOriginalValue();
                }
            });
        },

        set_button_status: function () {
            let me = this;

            let form = me.lookup('xtermform');
            let valid = form.isValid();
            let dirty = form.isDirty();

            let hasvalues = false;
            let values = form.getValues();
            Ext.Object.eachValue(values, function (value) {
                if (value) {
                    hasvalues = true;
                    return false;
                }
                return true;
            });

            me.lookup('xtermsave').setDisabled(!dirty || !valid);
            me.lookup('xtermreset').setDisabled(!hasvalues);
        },

        control: {
            '#xtermjs form': {
                dirtychange: 'set_button_status',
                validitychange: 'set_button_status',
            },
            '#xtermjs button': {
                click: function (button) {
                    let me = this;
                    let settings = ['fontSize', 'fontFamily', 'letterSpacing', 'lineHeight'];
                    settings.forEach(function (setting) {
                        let field = me.lookup(setting);
                        if (button.reference === 'xtermsave') {
                            let value = field.getValue();
                            if (value) {
                                localStorage.setItem('pve-xterm-' + setting, value);
                            } else {
                                localStorage.removeItem('pve-xterm-' + setting);
                            }
                        } else if (button.reference === 'xtermreset') {
                            field.setValue(undefined);
                            localStorage.removeItem('pve-xterm-' + setting);
                        }
                        field.resetOriginalValue();
                    });
                    me.set_button_status();
                },
            },
            'button[name=reset]': {
                click: function () {
                    let blocklist = ['login-username'];
                    let sp = Ext.state.Manager.getProvider();
                    for (const state of Object.values(sp.state)) {
                        if (blocklist.indexOf(state) !== -1) {
                            continue;
                        }

                        sp.clear(state);
                    }

                    window.location.reload();
                },
            },
            'button[name=clear-username]': {
                click: function () {
                    let me = this;
                    let usernamefield = me.lookupReference('savedUserName');
                    let sp = Ext.state.Manager.getProvider();

                    usernamefield.setValue(Proxmox.Utils.noneText);
                    sp.clear('login-username');
                },
            },
            'field[reference=summarycolumns]': {
                change: function (el, newValue) {
                    var sp = Ext.state.Manager.getProvider();
                    sp.set('summarycolumns', newValue);
                },
            },
        },
    },

    items: [
        {
            xtype: 'fieldset',
            flex: 1,
            title: gettext('Webinterface Settings'),
            margin: '5',
            layout: {
                type: 'vbox',
                align: 'left',
            },
            defaults: {
                width: '100%',
                margin: '0 0 10 0',
            },
            items: [
                {
                    xtype: 'container',
                    layout: 'hbox',
                    items: [
                        {
                            xtype: 'displayfield',
                            fieldLabel: gettext('Saved User Name') + ':',
                            labelWidth: 150,
                            stateId: 'login-username',
                            reference: 'savedUserName',
                            flex: 1,
                            value: '',
                        },
                        {
                            xtype: 'button',
                            cls: 'x-btn-default-toolbar-small proxmox-inline-button',
                            text: gettext('Reset'),
                            name: 'clear-username',
                        },
                    ],
                },
                {
                    xtype: 'box',
                    autoEl: { tag: 'hr' },
                },
                {
                    xtype: 'container',
                    layout: 'hbox',
                    items: [
                        {
                            xtype: 'displayfield',
                            fieldLabel: gettext('Layout') + ':',
                            flex: 1,
                        },
                        {
                            xtype: 'button',
                            cls: 'x-btn-default-toolbar-small proxmox-inline-button',
                            text: gettext('Reset'),
                            tooltip: gettext(
                                'Reset all layout changes (for example, column widths)',
                            ),
                            name: 'reset',
                        },
                    ],
                },
                {
                    xtype: 'box',
                    autoEl: { tag: 'hr' },
                },
                {
                    xtype: 'proxmoxKVComboBox',
                    fieldLabel: gettext('Summary/Dashboard columns') + ':',
                    labelWidth: 150,
                    stateId: 'summarycolumns',
                    reference: 'summarycolumns',
                    comboItems: [
                        ['auto', 'auto'],
                        ['1', '1'],
                        ['2', '2'],
                        ['3', '3'],
                    ],
                },
            ],
        },
        {
            xtype: 'container',
            layout: 'vbox',
            flex: 1,
            margin: '5',
            defaults: {
                width: '100%',
                // right margin ensures that the right border of the fieldsets
                // is shown
                margin: '0 2 10 0',
            },
            items: [
                {
                    xtype: 'fieldset',
                    itemId: 'xtermjs',
                    title: gettext('xterm.js Settings'),
                    items: [
                        {
                            xtype: 'form',
                            reference: 'xtermform',
                            border: false,
                            layout: {
                                type: 'vbox',
                                algin: 'left',
                            },
                            defaults: {
                                width: '100%',
                                margin: '0 0 10 0',
                            },
                            items: [
                                {
                                    xtype: 'textfield',
                                    name: 'fontFamily',
                                    reference: 'fontFamily',
                                    emptyText: Proxmox.Utils.defaultText,
                                    fieldLabel: gettext('Font-Family'),
                                },
                                {
                                    xtype: 'proxmoxintegerfield',
                                    emptyText: Proxmox.Utils.defaultText,
                                    name: 'fontSize',
                                    reference: 'fontSize',
                                    minValue: 1,
                                    fieldLabel: gettext('Font-Size'),
                                },
                                {
                                    xtype: 'numberfield',
                                    name: 'letterSpacing',
                                    reference: 'letterSpacing',
                                    emptyText: Proxmox.Utils.defaultText,
                                    fieldLabel: gettext('Letter Spacing'),
                                },
                                {
                                    xtype: 'numberfield',
                                    name: 'lineHeight',
                                    minValue: 0.1,
                                    reference: 'lineHeight',
                                    emptyText: Proxmox.Utils.defaultText,
                                    fieldLabel: gettext('Line Height'),
                                },
                                {
                                    xtype: 'container',
                                    layout: {
                                        type: 'hbox',
                                        pack: 'end',
                                    },
                                    defaults: {
                                        margin: '0 0 0 5',
                                    },
                                    items: [
                                        {
                                            xtype: 'button',
                                            reference: 'xtermreset',
                                            disabled: true,
                                            text: gettext('Reset'),
                                        },
                                        {
                                            xtype: 'button',
                                            reference: 'xtermsave',
                                            disabled: true,
                                            text: gettext('Save'),
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                },
            ],
        },
    ],
});
Ext.define('PMG.Application', {
    extend: 'Ext.app.Application',

    name: 'PMG',
    appProperty: 'app',

    stores: ['NavigationStore'],

    layout: 'fit',

    realignWindows: function () {
        var modalwindows = Ext.ComponentQuery.query('window[modal]');
        Ext.Array.forEach(modalwindows, function (item) {
            item.center();
        });
    },

    logout: function () {
        var me = this;
        Proxmox.Utils.authClear();
        me.changeView('loginview', true);
    },

    changeView: function (view, skipCheck) {
        var me = this;
        PMG.view = view;
        me.view = view;
        me.currentView.destroy();
        me.currentView = Ext.create({
            xtype: view,
            targetview: me.targetview,
        });
        if (skipCheck !== true) {
            Proxmox.Utils.checked_command(Ext.emptyFn); // display subscription status
        }
    },

    view: 'loginview',
    targetview: 'mainview',

    launch: function () {
        var me = this;
        Ext.on('resize', me.realignWindows);

        var provider = new Ext.state.LocalStorageProvider({ prefix: 'ext-pmg-' });
        Ext.state.Manager.setProvider(provider);

        // show login window if not loggedin
        var loggedin = Proxmox.Utils.authOK();
        var cookie = Ext.util.Cookies.get(Proxmox.Setup.auth_cookie_name);
        var qs = Ext.Object.fromQueryString(location.search);

        var pathname = location.pathname.replace(/\/+$/, '');

        if (pathname === '/quarantine') {
            me.targetview = 'quarantineview';

            if (qs.ticket === undefined && loggedin) {
                me.view = 'quarantineview';
            }
        } else if (loggedin && cookie.substr(0, 7) !== 'PMGQUAR') {
            me.view = 'mainview';
        }

        PMG.view = me.view;
        me.currentView = Ext.create({
            xtype: me.view,
            targetview: me.targetview,
        });
    },
});

Ext.application('PMG.Application');
